"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const fs = require("fs");
const path = require("path");
const lib_1 = require("../lib");
const bundling_1 = require("../lib/bundling");
jest.mock('../lib/bundling', () => {
    return {
        Bundling: {
            parcel: jest.fn().mockReturnValue({
                bind: () => {
                    return { inlineCode: 'code' };
                },
                bindToResource: () => { return; },
            }),
        },
    };
});
let stack;
beforeEach(() => {
    stack = new core_1.Stack();
    jest.clearAllMocks();
});
test('NodejsFunction with .ts handler', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1');
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringContaining('function.test.handler1.ts'),
    }));
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
    });
});
test('NodejsFunction with .js handler', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler2');
    // THEN
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringContaining('function.test.handler2.js'),
    }));
});
test('NodejsFunction with container env vars', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'handler1', {
        parcelEnvironment: {
            KEY: 'VALUE',
        },
    });
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        parcelEnvironment: {
            KEY: 'VALUE',
        },
    }));
});
test('throws when entry is not js/ts', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'handler.py',
    })).toThrow(/Only JavaScript or TypeScript entry files are supported/);
});
test('accepts tsx', () => {
    const entry = path.join(__dirname, 'handler.tsx');
    fs.symlinkSync(path.join(__dirname, 'function.test.handler1.ts'), entry);
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry,
    })).not.toThrow();
    fs.unlinkSync(entry);
});
test('throws when entry does not exist', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn', {
        entry: 'notfound.ts',
    })).toThrow(/Cannot find entry file at/);
});
test('throws when entry cannot be automatically found', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'Fn')).toThrow(/Cannot find entry file./);
});
test('throws with the wrong runtime family', () => {
    expect(() => new lib_1.NodejsFunction(stack, 'handler1', {
        runtime: aws_lambda_1.Runtime.PYTHON_3_8,
    })).toThrow(/Only `NODEJS` runtimes are supported/);
});
test('resolves entry to an absolute path', () => {
    // WHEN
    new lib_1.NodejsFunction(stack, 'fn', {
        entry: 'lib/index.ts',
    });
    expect(bundling_1.Bundling.parcel).toHaveBeenCalledWith(expect.objectContaining({
        entry: expect.stringMatching(/@aws-cdk\/aws-lambda-nodejs\/lib\/index.ts$/),
    }));
});
//# sourceMappingURL=data:application/json;base64,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