import difflib
import re

from litellm import AllMessageValues, ChatCompletionAssistantMessage
from litellm.utils import token_counter
from opensearchpy import AsyncOpenSearch

from sifts.analysis.criteria import generate_vuln_context_markdown, get_vuln_criteria
from sifts.common_types.snippets import SnippetHit
from sifts.llm.constants import MAX_TOKEN_COUNT_FUNCTION
from sifts.llm.fix_code import fix_the_code
from sifts.llm.router import RouterStrict


def generate_patch(string1: str, string2: str) -> str:
    # Split the strings into lines for comparison
    lines1 = string1.splitlines(keepends=True)
    lines2 = string2.splitlines(keepends=True)

    # Generate the diff
    diff = list(difflib.unified_diff(lines1, lines2))
    return "".join(diff)


async def get_vulnerability_cause(
    *,
    router: RouterStrict,
    code_function_snippet: str,
    code_function_snippet_fixed: str,
    finding_code: str,
) -> str | None:
    try:
        vuln_criteria = get_vuln_criteria(finding_code)
    except KeyError:
        return None
    vulnerability_context = generate_vuln_context_markdown(vuln_criteria)
    patch_diff = generate_patch(code_function_snippet, code_function_snippet_fixed)
    messages: list[AllMessageValues] = [
        {
            "content": (
                "This is a code snippet with "
                f"a vulnerability {vuln_criteria['en']['title']}: \n{code_function_snippet}\n"
                f"The vulnerability is described as follows: {vulnerability_context}\n"
                f"The correct way to fix it is by\n{patch_diff}\nThe code after modification"
                f" is as follows:\n{code_function_snippet_fixed}\n"
                "Why is the above modification necessary?"
            ),
            "role": "user",
        },
    ]
    if token_counter(messages=messages) > MAX_TOKEN_COUNT_FUNCTION:
        return None
    response = await router.acompletion(
        messages=messages,
        model="nova-pro",
        caching=True,
    )
    if not response.choices or not response.choices[0].message.content:
        return None

    messages.append(
        ChatCompletionAssistantMessage(
            {
                "content": str(response.choices[0].message.content),
                "role": "assistant",
            },
        ),
    )
    messages.append(
        {
            "content": (
                "I want you to act as a vulnerability detection expert and organize "
                "vulnerability knowledge based on the above vulnerability repair "
                "information. Please summarize the generalizable specific behavior "
                "of the code that leads to the vulnerability and the specific solution "
                "to fix it.\n"
                "Here are some examples to guide you on the level of detail expected in your"
                " extraction:\n"
                "# Vulnerability Causes\n"
                "Abstract Vulnerability Description: Use of RCU read lock without proper"
                " synchronization mechanism to protect shared data structures.\n"
                "Trigger Action: Concurrent access to the shared data structure while under RCU"
                " read lock context.\n"
                "Detailed Vulnerability Description: The code accesses a shared data structure"
                " within an RCU read lock context without ensuring exclusive access, allowing a"
                " race condition where the object can be freed while still being accessed.\n"
                "#  Fixing Solution\n"
                "Solution Description: To mitigate the vulnerability, it is necessary to use a"
                " synchronization mechanism, such as a spin lock, to protect the shared data"
                " structure from concurrent modifications. By replacing the RCU read lock with a"
                " spin lock, exclusive access to the data structure is ensured, preventing the"
                " race condition and use-after-free vulnerability...."
            ),
            "role": "user",
        },
    )
    response = await router.acompletion(
        model="nova-pro",
        messages=messages,
        caching=True,
    )
    if not response.choices:
        return None
    messages.append(
        ChatCompletionAssistantMessage(
            {
                "content": response.choices[0].message.content,
                "role": "assistant",
            },
        ),
    )
    messages.append(
        {
            "content": (
                "With the detailed vulnerability knowledge extracted from the previous stage, your"
                " task is to "
                "abstract and generalize this knowledge to enhance its applicability "
                "across different scenarios. Please adhere to the following guidelines "
                "and examples provided:\n"
                "Abstracting Method Invocations. The extracted knowledge might "
                "contain concrete method invocations with detailed function "
                "identifiers (e.g., io_worker_handle_work function) and parameters"
                " (e.g., mutex_lock(&dmxdev->mutex)), which can be "
                "abstracted into the generalized description (e.g., “during handling "
                "of IO work processes” and “employing a locking mechanism akin "
                "to mutex_lock()”).\n"
                "Abstracting Variable Names and Types. The extracted knowledge "
                "might contain concrete variable names or types (e.g., “without "
                "&dev->ref initialization”), which can be abstracted into the more "
                "general description (e.g., “without proper reference counter initialization”)."
            ),
            "role": "user",
        },
    )
    final_response = await router.acompletion(
        messages=messages,
        model="nova-pro",
        caching=True,
    )
    return (
        str(final_response.choices[0].message.content)
        if final_response and final_response.choices
        else None
    )


async def get_vulnerability_vulnerability_knowledge(
    router: RouterStrict,
    open_client: AsyncOpenSearch,
    candidate: SnippetHit,
) -> str | None:
    vulnerability_knowledge: str | None = None
    if "fixed_function_code" in candidate["_source"]:
        code_function_snippet_fixed = candidate["_source"]["fixed_function_code"]
    else:
        code_function_snippet_fixed = await fix_the_code(
            router=router,
            finding_code=candidate["_source"]["metadata"]["criteria_code"],
            vulnerable_function=candidate["_source"]["vulnerable_function_code"],
            vulnerable_line_content=candidate["_source"]["vulnerable_line_content"],
        )
        if code_function_snippet_fixed:
            await open_client.update(
                "vulnerabilities_candidates",
                id=candidate["_id"],
                body={"doc": {"fixed_function_code": code_function_snippet_fixed}},
            )
    if not code_function_snippet_fixed:
        return None
    vulnerability_knowledge_source = candidate["_source"].get("vulnerability_knowledge")
    if (
        not vulnerability_knowledge_source or vulnerability_knowledge_source.count("{") > 2  # noqa: PLR2004
    ):
        vulnerability_knowledge = await get_vulnerability_cause(
            code_function_snippet=candidate["_source"]["vulnerable_function_code"],
            code_function_snippet_fixed=code_function_snippet_fixed,
            finding_code=candidate["_source"]["metadata"]["criteria_code"],
            router=router,
        )
        if not vulnerability_knowledge:
            return None

        await open_client.update(
            index="vulnerabilities_candidates",
            id=candidate["_id"],
            body={"doc": {"vulnerability_knowledge": vulnerability_knowledge}},
        )
    else:
        vulnerability_knowledge = candidate["_source"]["vulnerability_knowledge"]

    return vulnerability_knowledge


async def get_vulnerability_specific(
    *,
    router: RouterStrict,
    reason: str,
    snippet_content: str,
) -> int:
    lines = snippet_content.split("\n")
    snippet_content = "\n".join(
        [f"{index} |  {x}" for index, x in enumerate(lines, 1)],
    )
    messages: list[AllMessageValues] = [
        {
            "content": ("Is this code vulnerable?\n" + f"# Code\n{snippet_content}"),
            "role": "user",
        },
        {
            "content": (reason),
            "role": "assistant",
        },
        {
            "content": (
                "STRICT FORMATTING REQUIRED: Using your previous analysis, return ONLY "
                "the numeric line number where the vulnerability occurs (1-based index).\n\n"
                "If vulnerable, respond with exactly: 'LINE_NUMBER: <INT>'\n"
                "If not vulnerable, respond with exactly: 'LINE_NUMBER: 0'"
            ),
            "role": "user",
        },
    ]

    response = await router.acompletion(
        messages=messages,
        model="nova-pro",
        caching=True,
    )

    if not response.choices:
        return 0
    number: int = 0
    if match := re.search(r"LINE_NUMBER:\s*(\d+)", response.choices[0].message.content):
        number = int(match.group(1))

    return number
