"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Annotations = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const constructs_1 = require("constructs");
const DEPRECATIONS_SYMBOL = Symbol.for('@aws-cdk/core.deprecations');
/**
 * Includes API for attaching annotations such as warning messages to constructs.
 */
class Annotations {
    constructor(scope) {
        this.scope = scope;
    }
    /**
     * Returns the annotations API for a construct scope.
     * @param scope The scope
     */
    static of(scope) {
        return new Annotations(scope);
    }
    /**
     * Adds a warning metadata entry to this construct.
     *
     * The CLI will display the warning when an app is synthesized, or fail if run
     * in --strict mode.
     *
     * @param message The warning message.
     */
    addWarning(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an info metadata entry to this construct.
     *
     * The CLI will display the info message when apps are synthesized.
     *
     * @param message The info message.
     */
    addInfo(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message) {
        this.addMessage(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Adds a deprecation warning for a specific API.
     *
     * Deprecations will be added only once per construct as a warning and will be
     * deduplicated based on the `api`.
     *
     * If the environment variable `CDK_BLOCK_DEPRECATIONS` is set, this method
     * will throw an error instead with the deprecation message.
     *
     * @param api The API being deprecated in the format `module.Class.property`
     * (e.g. `@aws-cdk/core.Construct.node`).
     * @param message The deprecation message to display, with information about
     * alternatives.
     */
    addDeprecation(api, message) {
        const text = `The API ${api} is deprecated: ${message}. This API will be removed in the next major release`;
        // throw if CDK_BLOCK_DEPRECATIONS is set
        if (process.env.CDK_BLOCK_DEPRECATIONS) {
            throw new Error(`${constructs_1.Node.of(this.scope).path}: ${text}`);
        }
        // de-dup based on api key
        const set = this.deprecationsReported;
        if (set.has(api)) {
            return;
        }
        this.addWarning(text);
        set.add(api);
    }
    /**
     * Adds a message metadata entry to the construct node, to be displayed by the CDK CLI.
     * @param level The message level
     * @param message The message itself
     */
    addMessage(level, message) {
        constructs_1.Node.of(this.scope).addMetadata(level, message);
    }
    /**
     * Returns the set of deprecations reported on this construct.
     */
    get deprecationsReported() {
        let set = this.scope[DEPRECATIONS_SYMBOL];
        if (!set) {
            set = new Set();
            Object.defineProperty(this.scope, DEPRECATIONS_SYMBOL, { value: set });
        }
        return set;
    }
}
exports.Annotations = Annotations;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYW5ub3RhdGlvbnMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbm5vdGF0aW9ucy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSwyREFBMkQ7QUFDM0QsMkNBQThDO0FBRTlDLE1BQU0sbUJBQW1CLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0FBRXJFOztHQUVHO0FBQ0gsTUFBYSxXQUFXO0lBU3RCLFlBQXFDLEtBQWlCO1FBQWpCLFVBQUssR0FBTCxLQUFLLENBQVk7SUFFdEQsQ0FBQztJQVZEOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxFQUFFLENBQUMsS0FBaUI7UUFDaEMsT0FBTyxJQUFJLFdBQVcsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUNoQyxDQUFDO0lBTUQ7Ozs7Ozs7T0FPRztJQUNJLFVBQVUsQ0FBQyxPQUFlO1FBQy9CLElBQUksQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLHlCQUF5QixDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNwRSxDQUFDO0lBRUQ7Ozs7OztPQU1HO0lBQ0ksT0FBTyxDQUFDLE9BQWU7UUFDNUIsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMseUJBQXlCLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3BFLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksUUFBUSxDQUFDLE9BQWU7UUFDN0IsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMseUJBQXlCLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBQ3JFLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0ksY0FBYyxDQUFDLEdBQVcsRUFBRSxPQUFlO1FBQ2hELE1BQU0sSUFBSSxHQUFHLFdBQVcsR0FBRyxtQkFBbUIsT0FBTyxzREFBc0QsQ0FBQztRQUU1Ryx5Q0FBeUM7UUFDekMsSUFBSSxPQUFPLENBQUMsR0FBRyxDQUFDLHNCQUFzQixFQUFFO1lBQ3RDLE1BQU0sSUFBSSxLQUFLLENBQUMsR0FBRyxpQkFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsSUFBSSxLQUFLLElBQUksRUFBRSxDQUFDLENBQUM7U0FDekQ7UUFFRCwwQkFBMEI7UUFDMUIsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDO1FBQ3RDLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUNoQixPQUFPO1NBQ1I7UUFFRCxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ3RCLEdBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDZixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNLLFVBQVUsQ0FBQyxLQUFhLEVBQUUsT0FBZTtRQUMvQyxpQkFBSSxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQ7O09BRUc7SUFDSCxJQUFZLG9CQUFvQjtRQUM5QixJQUFJLEdBQUcsR0FBSSxJQUFJLENBQUMsS0FBYSxDQUFDLG1CQUFtQixDQUFDLENBQUM7UUFDbkQsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNSLEdBQUcsR0FBRyxJQUFJLEdBQUcsRUFBRSxDQUFDO1lBQ2hCLE1BQU0sQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRSxFQUFFLEtBQUssRUFBRSxHQUFHLEVBQUUsQ0FBQyxDQUFDO1NBQ3hFO1FBRUQsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0NBQ0Y7QUFsR0Qsa0NBa0dDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY3hzY2hlbWEgZnJvbSAnQGF3cy1jZGsvY2xvdWQtYXNzZW1ibHktc2NoZW1hJztcbmltcG9ydCB7IElDb25zdHJ1Y3QsIE5vZGUgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuY29uc3QgREVQUkVDQVRJT05TX1NZTUJPTCA9IFN5bWJvbC5mb3IoJ0Bhd3MtY2RrL2NvcmUuZGVwcmVjYXRpb25zJyk7XG5cbi8qKlxuICogSW5jbHVkZXMgQVBJIGZvciBhdHRhY2hpbmcgYW5ub3RhdGlvbnMgc3VjaCBhcyB3YXJuaW5nIG1lc3NhZ2VzIHRvIGNvbnN0cnVjdHMuXG4gKi9cbmV4cG9ydCBjbGFzcyBBbm5vdGF0aW9ucyB7XG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBhbm5vdGF0aW9ucyBBUEkgZm9yIGEgY29uc3RydWN0IHNjb3BlLlxuICAgKiBAcGFyYW0gc2NvcGUgVGhlIHNjb3BlXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIG9mKHNjb3BlOiBJQ29uc3RydWN0KSB7XG4gICAgcmV0dXJuIG5ldyBBbm5vdGF0aW9ucyhzY29wZSk7XG4gIH1cblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKHByaXZhdGUgcmVhZG9ubHkgc2NvcGU6IElDb25zdHJ1Y3QpIHtcblxuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSB3YXJuaW5nIG1ldGFkYXRhIGVudHJ5IHRvIHRoaXMgY29uc3RydWN0LlxuICAgKlxuICAgKiBUaGUgQ0xJIHdpbGwgZGlzcGxheSB0aGUgd2FybmluZyB3aGVuIGFuIGFwcCBpcyBzeW50aGVzaXplZCwgb3IgZmFpbCBpZiBydW5cbiAgICogaW4gLS1zdHJpY3QgbW9kZS5cbiAgICpcbiAgICogQHBhcmFtIG1lc3NhZ2UgVGhlIHdhcm5pbmcgbWVzc2FnZS5cbiAgICovXG4gIHB1YmxpYyBhZGRXYXJuaW5nKG1lc3NhZ2U6IHN0cmluZykge1xuICAgIHRoaXMuYWRkTWVzc2FnZShjeHNjaGVtYS5BcnRpZmFjdE1ldGFkYXRhRW50cnlUeXBlLldBUk4sIG1lc3NhZ2UpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gaW5mbyBtZXRhZGF0YSBlbnRyeSB0byB0aGlzIGNvbnN0cnVjdC5cbiAgICpcbiAgICogVGhlIENMSSB3aWxsIGRpc3BsYXkgdGhlIGluZm8gbWVzc2FnZSB3aGVuIGFwcHMgYXJlIHN5bnRoZXNpemVkLlxuICAgKlxuICAgKiBAcGFyYW0gbWVzc2FnZSBUaGUgaW5mbyBtZXNzYWdlLlxuICAgKi9cbiAgcHVibGljIGFkZEluZm8obWVzc2FnZTogc3RyaW5nKTogdm9pZCB7XG4gICAgdGhpcy5hZGRNZXNzYWdlKGN4c2NoZW1hLkFydGlmYWN0TWV0YWRhdGFFbnRyeVR5cGUuSU5GTywgbWVzc2FnZSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhbiB7IFwiZXJyb3JcIjogPG1lc3NhZ2U+IH0gbWV0YWRhdGEgZW50cnkgdG8gdGhpcyBjb25zdHJ1Y3QuXG4gICAqIFRoZSB0b29sa2l0IHdpbGwgZmFpbCBzeW50aGVzaXMgd2hlbiBlcnJvcnMgYXJlIHJlcG9ydGVkLlxuICAgKiBAcGFyYW0gbWVzc2FnZSBUaGUgZXJyb3IgbWVzc2FnZS5cbiAgICovXG4gIHB1YmxpYyBhZGRFcnJvcihtZXNzYWdlOiBzdHJpbmcpIHtcbiAgICB0aGlzLmFkZE1lc3NhZ2UoY3hzY2hlbWEuQXJ0aWZhY3RNZXRhZGF0YUVudHJ5VHlwZS5FUlJPUiwgbWVzc2FnZSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkcyBhIGRlcHJlY2F0aW9uIHdhcm5pbmcgZm9yIGEgc3BlY2lmaWMgQVBJLlxuICAgKlxuICAgKiBEZXByZWNhdGlvbnMgd2lsbCBiZSBhZGRlZCBvbmx5IG9uY2UgcGVyIGNvbnN0cnVjdCBhcyBhIHdhcm5pbmcgYW5kIHdpbGwgYmVcbiAgICogZGVkdXBsaWNhdGVkIGJhc2VkIG9uIHRoZSBgYXBpYC5cbiAgICpcbiAgICogSWYgdGhlIGVudmlyb25tZW50IHZhcmlhYmxlIGBDREtfQkxPQ0tfREVQUkVDQVRJT05TYCBpcyBzZXQsIHRoaXMgbWV0aG9kXG4gICAqIHdpbGwgdGhyb3cgYW4gZXJyb3IgaW5zdGVhZCB3aXRoIHRoZSBkZXByZWNhdGlvbiBtZXNzYWdlLlxuICAgKlxuICAgKiBAcGFyYW0gYXBpIFRoZSBBUEkgYmVpbmcgZGVwcmVjYXRlZCBpbiB0aGUgZm9ybWF0IGBtb2R1bGUuQ2xhc3MucHJvcGVydHlgXG4gICAqIChlLmcuIGBAYXdzLWNkay9jb3JlLkNvbnN0cnVjdC5ub2RlYCkuXG4gICAqIEBwYXJhbSBtZXNzYWdlIFRoZSBkZXByZWNhdGlvbiBtZXNzYWdlIHRvIGRpc3BsYXksIHdpdGggaW5mb3JtYXRpb24gYWJvdXRcbiAgICogYWx0ZXJuYXRpdmVzLlxuICAgKi9cbiAgcHVibGljIGFkZERlcHJlY2F0aW9uKGFwaTogc3RyaW5nLCBtZXNzYWdlOiBzdHJpbmcpIHtcbiAgICBjb25zdCB0ZXh0ID0gYFRoZSBBUEkgJHthcGl9IGlzIGRlcHJlY2F0ZWQ6ICR7bWVzc2FnZX0uIFRoaXMgQVBJIHdpbGwgYmUgcmVtb3ZlZCBpbiB0aGUgbmV4dCBtYWpvciByZWxlYXNlYDtcblxuICAgIC8vIHRocm93IGlmIENES19CTE9DS19ERVBSRUNBVElPTlMgaXMgc2V0XG4gICAgaWYgKHByb2Nlc3MuZW52LkNES19CTE9DS19ERVBSRUNBVElPTlMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgJHtOb2RlLm9mKHRoaXMuc2NvcGUpLnBhdGh9OiAke3RleHR9YCk7XG4gICAgfVxuXG4gICAgLy8gZGUtZHVwIGJhc2VkIG9uIGFwaSBrZXlcbiAgICBjb25zdCBzZXQgPSB0aGlzLmRlcHJlY2F0aW9uc1JlcG9ydGVkO1xuICAgIGlmIChzZXQuaGFzKGFwaSkpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICB0aGlzLmFkZFdhcm5pbmcodGV4dCk7XG4gICAgc2V0LmFkZChhcGkpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBtZXNzYWdlIG1ldGFkYXRhIGVudHJ5IHRvIHRoZSBjb25zdHJ1Y3Qgbm9kZSwgdG8gYmUgZGlzcGxheWVkIGJ5IHRoZSBDREsgQ0xJLlxuICAgKiBAcGFyYW0gbGV2ZWwgVGhlIG1lc3NhZ2UgbGV2ZWxcbiAgICogQHBhcmFtIG1lc3NhZ2UgVGhlIG1lc3NhZ2UgaXRzZWxmXG4gICAqL1xuICBwcml2YXRlIGFkZE1lc3NhZ2UobGV2ZWw6IHN0cmluZywgbWVzc2FnZTogc3RyaW5nKSB7XG4gICAgTm9kZS5vZih0aGlzLnNjb3BlKS5hZGRNZXRhZGF0YShsZXZlbCwgbWVzc2FnZSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgc2V0IG9mIGRlcHJlY2F0aW9ucyByZXBvcnRlZCBvbiB0aGlzIGNvbnN0cnVjdC5cbiAgICovXG4gIHByaXZhdGUgZ2V0IGRlcHJlY2F0aW9uc1JlcG9ydGVkKCkge1xuICAgIGxldCBzZXQgPSAodGhpcy5zY29wZSBhcyBhbnkpW0RFUFJFQ0FUSU9OU19TWU1CT0xdO1xuICAgIGlmICghc2V0KSB7XG4gICAgICBzZXQgPSBuZXcgU2V0KCk7XG4gICAgICBPYmplY3QuZGVmaW5lUHJvcGVydHkodGhpcy5zY29wZSwgREVQUkVDQVRJT05TX1NZTUJPTCwgeyB2YWx1ZTogc2V0IH0pO1xuICAgIH1cblxuICAgIHJldHVybiBzZXQ7XG4gIH1cbn1cbiJdfQ==