"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AssetStaging = void 0;
const crypto = require("crypto");
const os = require("os");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs-extra");
const assets_1 = require("./assets");
const construct_compat_1 = require("./construct-compat");
const fs_1 = require("./fs");
const stack_1 = require("./stack");
const stage_1 = require("./stage");
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        const outdir = (_a = stage_1.Stage.of(this)) === null || _a === void 0 ? void 0 : _a.outdir;
        if (!outdir) {
            throw new Error('unable to determine cloud assembly output directory. Assets must be defined indirectly within a "Stage" or an "App" scope');
        }
        // Determine the hash type based on the props as props.assetHashType is
        // optional from a caller perspective.
        const hashType = determineHashType(props.assetHashType, props.assetHash);
        if (props.bundling) {
            // Check if we actually have to bundle for this stack
            const bundlingStacks = (_b = this.node.tryGetContext(cxapi.BUNDLING_STACKS)) !== null && _b !== void 0 ? _b : ['*'];
            const runBundling = bundlingStacks.includes(stack_1.Stack.of(this).stackName) || bundlingStacks.includes('*');
            if (runBundling) {
                // Determine the source hash in advance of bundling if the asset hash type
                // is SOURCE so that the bundler can opt to re-use its previous output.
                const sourceHash = hashType === assets_1.AssetHashType.SOURCE
                    ? this.calculateHash(hashType, props.assetHash, props.bundling)
                    : undefined;
                this.bundleDir = this.bundle(props.bundling, outdir, sourceHash);
                this.assetHash = sourceHash !== null && sourceHash !== void 0 ? sourceHash : this.calculateHash(hashType, props.assetHash, props.bundling);
                this.relativePath = renderAssetFilename(this.assetHash);
                this.stagedPath = this.relativePath;
            }
            else { // Bundling is skipped
                this.assetHash = props.assetHashType === assets_1.AssetHashType.BUNDLE || props.assetHashType === assets_1.AssetHashType.OUTPUT
                    ? this.calculateHash(assets_1.AssetHashType.CUSTOM, this.node.path) // Use node path as dummy hash because we're not bundling
                    : this.calculateHash(hashType, props.assetHash);
                this.stagedPath = this.sourcePath;
            }
        }
        else {
            this.assetHash = this.calculateHash(hashType, props.assetHash);
            this.relativePath = renderAssetFilename(this.assetHash, path.extname(this.sourcePath));
            this.stagedPath = this.relativePath;
        }
        const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
        if (stagingDisabled) {
            this.relativePath = undefined;
            this.stagedPath = (_c = this.bundleDir) !== null && _c !== void 0 ? _c : this.sourcePath;
        }
        this.sourceHash = this.assetHash;
        this.stageAsset(outdir);
    }
    stageAsset(outdir) {
        // Staging is disabled
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(outdir, this.relativePath);
        // Staging the bundling asset.
        if (this.bundleDir) {
            const isAlreadyStaged = fs.existsSync(targetPath);
            if (isAlreadyStaged && path.resolve(this.bundleDir) !== path.resolve(targetPath)) {
                // When an identical asset is already staged and the bundler used an
                // intermediate bundling directory, we remove the extra directory.
                fs.removeSync(this.bundleDir);
            }
            else if (!isAlreadyStaged) {
                fs.renameSync(this.bundleDir, targetPath);
            }
            return;
        }
        // Already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
    /**
     * Bundles an asset and provides the emitted asset's directory in return.
     *
     * @param options Bundling options
     * @param outdir Parent directory to create the bundle output directory in
     * @param sourceHash The asset source hash if known in advance. If this field
     * is provided, the bundler may opt to skip bundling, providing any already-
     * emitted bundle. If this field is not provided, the bundler uses an
     * intermediate directory in outdir.
     * @returns The fully resolved bundle output directory.
     */
    bundle(options, outdir, sourceHash) {
        var _a, _b, _c;
        let bundleDir;
        if (sourceHash) {
            // When an asset hash is known in advance of bundling, the bundler outputs
            // directly to the assembly output directory.
            bundleDir = path.resolve(path.join(outdir, renderAssetFilename(sourceHash)));
            if (fs.existsSync(bundleDir)) {
                // Pre-existing bundle directory. The bundle has already been generated
                // once before, so we'll give the caller nothing.
                return bundleDir;
            }
            fs.ensureDirSync(bundleDir);
        }
        else {
            // When the asset hash isn't known in advance, bundler outputs to an
            // intermediate directory.
            // Create temp directory for bundling inside the temp staging directory
            bundleDir = path.resolve(fs.mkdtempSync(path.join(outdir, 'bundling-temp-')));
            // Chmod the bundleDir to full access.
            fs.chmodSync(bundleDir, 0o777);
        }
        let user;
        if (options.user) {
            user = options.user;
        }
        else { // Default to current user
            const userInfo = os.userInfo();
            user = userInfo.uid !== -1 // uid is -1 on Windows
                ? `${userInfo.uid}:${userInfo.gid}`
                : '1000:1000';
        }
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: AssetStaging.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: AssetStaging.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        let localBundling;
        try {
            process.stderr.write(`Bundling asset ${this.node.path}...\n`);
            localBundling = (_b = options.local) === null || _b === void 0 ? void 0 : _b.tryBundle(bundleDir, options);
            if (!localBundling) {
                options.image._run({
                    command: options.command,
                    user,
                    volumes,
                    environment: options.environment,
                    workingDirectory: (_c = options.workingDirectory) !== null && _c !== void 0 ? _c : AssetStaging.BUNDLING_INPUT_DIR,
                });
            }
        }
        catch (err) {
            // When bundling fails, keep the bundle output for diagnosability, but
            // rename it out of the way so that the next run doesn't assume it has a
            // valid bundleDir.
            const bundleErrorDir = bundleDir + '-error';
            if (fs.existsSync(bundleErrorDir)) {
                // Remove the last bundleErrorDir.
                fs.removeSync(bundleErrorDir);
            }
            fs.renameSync(bundleDir, bundleErrorDir);
            throw new Error(`Failed to bundle asset ${this.node.path}, bundle output is located at ${bundleErrorDir}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            const outputDir = localBundling ? bundleDir : AssetStaging.BUNDLING_OUTPUT_DIR;
            throw new Error(`Bundling did not produce any output. Check that content is written to ${outputDir}.`);
        }
        return bundleDir;
    }
    calculateHash(hashType, assetHash, bundling) {
        if (hashType === assets_1.AssetHashType.CUSTOM && !assetHash) {
            throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
        }
        // When bundling a CUSTOM or SOURCE asset hash type, we want the hash to include
        // the bundling configuration. We handle CUSTOM and bundled SOURCE hash types
        // as a special case to preserve existing user asset hashes in all other cases.
        if (hashType == assets_1.AssetHashType.CUSTOM || (hashType == assets_1.AssetHashType.SOURCE && bundling)) {
            const hash = crypto.createHash('sha256');
            // if asset hash is provided by user, use it, otherwise fingerprint the source.
            hash.update(assetHash !== null && assetHash !== void 0 ? assetHash : fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions));
            // If we're bundling an asset, include the bundling configuration in the hash
            if (bundling) {
                hash.update(JSON.stringify(bundling));
            }
            return hash.digest('hex');
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
            case assets_1.AssetHashType.OUTPUT:
                if (!this.bundleDir) {
                    throw new Error(`Cannot use \`${hashType}\` hash type when \`bundling\` is not specified.`);
                }
                return fs_1.FileSystem.fingerprint(this.bundleDir, this.fingerprintOptions);
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
/**
 * The directory inside the bundling container into which the asset sources will be mounted.
 * @experimental
 */
AssetStaging.BUNDLING_INPUT_DIR = '/asset-input';
/**
 * The directory inside the bundling container into which the bundled output should be written.
 * @experimental
 */
AssetStaging.BUNDLING_OUTPUT_DIR = '/asset-output';
function renderAssetFilename(assetHash, extension = '') {
    return `asset.${assetHash}${extension}`;
}
/**
 * Determines the hash type from user-given prop values.
 *
 * @param assetHashType Asset hash type construct prop
 * @param assetHash Asset hash given in the construct props
 */
function determineHashType(assetHashType, assetHash) {
    if (assetHash) {
        if (assetHashType && assetHashType !== assets_1.AssetHashType.CUSTOM) {
            throw new Error(`Cannot specify \`${assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
        }
        return assets_1.AssetHashType.CUSTOM;
    }
    else if (assetHashType) {
        return assetHashType;
    }
    else {
        return assets_1.AssetHashType.SOURCE;
    }
}
//# sourceMappingURL=data:application/json;base64,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