"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resource = void 0;
// v2 - leave this as a separate section so it reduces merge conflicts when compat is removed
// eslint-disable-next-line import/order
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * A construct which represents an AWS resource.
 */
class Resource extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.stack = stack_1.Stack.of(this);
        this.env = {
            account: (_a = props.account) !== null && _a !== void 0 ? _a : this.stack.account,
            region: (_b = props.region) !== null && _b !== void 0 ? _b : this.stack.region,
        };
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.stringValue({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * Commonly this is the resource's `ref`.
     * @experimental
     */
    getResourceNameAttribute(nameAttr) {
        return lazy_1.Lazy.stringValue({
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            },
        });
    }
    /**
     * Returns an environment-sensitive token that should be used for the
     * resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * Commonly it will be called "Arn" (e.g. `resource.attrArn`), but sometimes
     * it's the CFN resource's `ref`.
     * @param arnComponents The format of the ARN of this resource. You must
     * reference `this.physicalName` somewhere within the ARN in order for
     * cross-environment references to work.
     *
     * @experimental
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return token_1.Token.asString({
            resolve: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            },
        });
    }
}
exports.Resource = Resource;
//# sourceMappingURL=data:application/json;base64,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