"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const construct_compat_1 = require("./construct-compat");
const synthesis_1 = require("./private/synthesis");
/**
 * An abstract application modeling unit consisting of Stacks that should be
 * deployed together.
 *
 * Derive a subclass of `Stage` and use it to model a single instance of your
 * application.
 *
 * You can then instantiate your subclass multiple times to model multiple
 * copies of your application which should be be deployed to different
 * environments.
 */
class Stage extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        if (id !== '' && !/^[a-z][a-z0-9\-\_\.]+$/i.test(id)) {
            throw new Error(`invalid stage name "${id}". Stage name must start with a letter and contain only alphanumeric characters, hypens ('-'), underscores ('_') and periods ('.')`);
        }
        this.parentStage = Stage.of(this);
        this.region = (_b = (_a = props.env) === null || _a === void 0 ? void 0 : _a.region) !== null && _b !== void 0 ? _b : (_c = this.parentStage) === null || _c === void 0 ? void 0 : _c.region;
        this.account = (_e = (_d = props.env) === null || _d === void 0 ? void 0 : _d.account) !== null && _e !== void 0 ? _e : (_f = this.parentStage) === null || _f === void 0 ? void 0 : _f.account;
        this._assemblyBuilder = this.createBuilder(props.outdir);
        this.stageName = [(_g = this.parentStage) === null || _g === void 0 ? void 0 : _g.stageName, id].filter(x => x).join('-');
    }
    /**
     * Return the stage this construct is contained with, if available. If called
     * on a nested stage, returns its parent.
     *
     * @experimental
     */
    static of(construct) {
        return constructs_1.Node.of(construct).scopes.reverse().slice(1).find(Stage.isStage);
    }
    /**
     * Test whether the given construct is a stage.
     *
     * @experimental
     */
    static isStage(x) {
        return x !== null && x instanceof Stage;
    }
    /**
     * The cloud assembly output directory.
     */
    get outdir() {
        return this._assemblyBuilder.outdir;
    }
    /**
     * Artifact ID of the assembly if it is a nested stage. The root stage (app)
     * will return an empty string.
     *
     * Derived from the construct path.
     *
     * @experimental
     */
    get artifactId() {
        if (!this.node.path) {
            return '';
        }
        return `assembly-${this.node.path.replace(/\//g, '-').replace(/^-+|-+$/g, '')}`;
    }
    /**
     * Synthesize this stage into a cloud assembly.
     *
     * Once an assembly has been synthesized, it cannot be modified. Subsequent
     * calls will return the same assembly.
     */
    synth(options = {}) {
        if (!this.assembly || options.force) {
            this.assembly = synthesis_1.synthesize(this, {
                skipValidation: options.skipValidation,
            });
        }
        return this.assembly;
    }
    createBuilder(outdir) {
        // cannot specify "outdir" if we are a nested stage
        if (this.parentStage && outdir) {
            throw new Error('"outdir" cannot be specified for nested stages');
        }
        // Need to determine fixed output directory already, because we must know where
        // to write sub-assemblies (which must happen before we actually get to this app's
        // synthesize() phase).
        return this.parentStage
            ? this.parentStage._assemblyBuilder.createNestedAssembly(this.artifactId, this.node.path)
            : new cxapi.CloudAssemblyBuilder(outdir);
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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