"use strict";
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
const annotations_1 = require("../lib/annotations");
const app_1 = require("../lib/app");
const metadata_resource_1 = require("../lib/private/metadata-resource");
function withApp(props, block) {
    const app = new app_1.App({
        stackTraces: false,
        ...props,
    });
    block(app);
    return app.synth();
}
function synth(context) {
    return withApp({ context }, app => {
        const stack1 = new lib_1.Stack(app, 'stack1', { env: { account: '12345', region: 'us-east-1' } });
        new lib_1.CfnResource(stack1, 's1c1', { type: 'DummyResource', properties: { Prop1: 'Prop1' } });
        const r2 = new lib_1.CfnResource(stack1, 's1c2', { type: 'DummyResource', properties: { Foo: 123 } });
        const stack2 = new lib_1.Stack(app, 'stack2');
        new lib_1.CfnResource(stack2, 's2c1', { type: 'DummyResource', properties: { Prog2: 'Prog2' } });
        const c1 = new MyConstruct(stack2, 's1c2');
        // add some metadata
        stack1.node.addMetadata('meta', 111);
        annotations_1.Annotations.of(r2).addWarning('warning1');
        annotations_1.Annotations.of(r2).addWarning('warning2');
        c1.node.addMetadata('meta', { key: 'value' });
        app.node.addMetadata('applevel', 123); // apps can also have metadata
    });
}
function synthStack(name, includeMetadata = false, context) {
    const response = synth(context);
    const stack = response.getStackByName(name);
    if (!includeMetadata) {
        delete stack.metadata;
    }
    return stack;
}
class MyConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
        new lib_1.CfnResource(this, 'r1', { type: 'ResourceType1' });
        new lib_1.CfnResource(this, 'r2', { type: 'ResourceType2', properties: { FromContext: this.node.tryGetContext('ctx1') } });
    }
}
function parseModules(x) {
    if (x === undefined) {
        return {};
    }
    const ret = {};
    for (const clause of x.split(',')) {
        const [key, value] = clause.split('=');
        if (key !== undefined && value !== undefined) {
            ret[key] = value;
        }
    }
    return ret;
}
/**
 * Set the CLI_VERSION_ENV environment variable
 *
 * This is necessary to get the Stack to emit the metadata resource
 */
function withCliVersion(block) {
    process.env[cxapi.CLI_VERSION_ENV] = '1.2.3';
    try {
        return block();
    }
    finally {
        delete process.env[cxapi.CLI_VERSION_ENV];
    }
}
module.exports = {
    'synthesizes all stacks and returns synthesis result'(test) {
        const response = synth();
        delete response.dir;
        test.deepEqual(response.stacks.length, 2);
        const stack1 = response.stacks[0];
        test.deepEqual(stack1.stackName, 'stack1');
        test.deepEqual(stack1.id, 'stack1');
        test.deepEqual(stack1.environment.account, 12345);
        test.deepEqual(stack1.environment.region, 'us-east-1');
        test.deepEqual(stack1.environment.name, 'aws://12345/us-east-1');
        test.deepEqual(stack1.template, {
            Resources: {
                s1c1: { Type: 'DummyResource', Properties: { Prop1: 'Prop1' } },
                s1c2: { Type: 'DummyResource', Properties: { Foo: 123 } },
            },
        });
        test.deepEqual(stack1.manifest.metadata, {
            '/stack1': [{ type: 'meta', data: 111 }],
            '/stack1/s1c1': [{ type: 'aws:cdk:logicalId', data: 's1c1' }],
            '/stack1/s1c2': [{ type: 'aws:cdk:logicalId', data: 's1c2' },
                { type: 'aws:cdk:warning', data: 'warning1' },
                { type: 'aws:cdk:warning', data: 'warning2' }],
        });
        const stack2 = response.stacks[1];
        test.deepEqual(stack2.stackName, 'stack2');
        test.deepEqual(stack2.id, 'stack2');
        test.deepEqual(stack2.environment.name, 'aws://unknown-account/unknown-region');
        test.deepEqual(stack2.template, {
            Resources: {
                s2c1: { Type: 'DummyResource', Properties: { Prog2: 'Prog2' } },
                s1c2r1D1791C01: { Type: 'ResourceType1' },
                s1c2r25F685FFF: { Type: 'ResourceType2' },
            },
        });
        test.deepEqual(stack2.manifest.metadata, {
            '/stack2/s2c1': [{ type: 'aws:cdk:logicalId', data: 's2c1' }],
            '/stack2/s1c2': [{ type: 'meta', data: { key: 'value' } }],
            '/stack2/s1c2/r1': [{ type: 'aws:cdk:logicalId', data: 's1c2r1D1791C01' }],
            '/stack2/s1c2/r2': [{ type: 'aws:cdk:logicalId', data: 's1c2r25F685FFF' }],
        });
        test.done();
    },
    'context can be passed through CDK_CONTEXT'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2',
        });
        const prog = new app_1.App();
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context passed through CDK_CONTEXT has precedence'(test) {
        process.env[cxapi.CONTEXT_ENV] = JSON.stringify({
            key1: 'val1',
            key2: 'val2',
        });
        const prog = new app_1.App({
            context: {
                key1: 'val3',
                key2: 'val4',
            },
        });
        test.deepEqual(prog.node.tryGetContext('key1'), 'val1');
        test.deepEqual(prog.node.tryGetContext('key2'), 'val2');
        test.done();
    },
    'context from the command line can be used when creating the stack'(test) {
        const output = synthStack('stack2', false, { ctx1: 'HELLO' });
        test.deepEqual(output.template, {
            Resources: {
                s2c1: {
                    Type: 'DummyResource',
                    Properties: {
                        Prog2: 'Prog2',
                    },
                },
                s1c2r1D1791C01: {
                    Type: 'ResourceType1',
                },
                s1c2r25F685FFF: {
                    Type: 'ResourceType2',
                    Properties: {
                        FromContext: 'HELLO',
                    },
                },
            },
        });
        test.done();
    },
    'setContext(k,v) can be used to set context programmatically'(test) {
        const prog = new app_1.App({
            context: {
                foo: 'bar',
            },
        });
        test.deepEqual(prog.node.tryGetContext('foo'), 'bar');
        test.done();
    },
    'setContext(k,v) cannot be called after stacks have been added because stacks may use the context'(test) {
        const prog = new app_1.App();
        new lib_1.Stack(prog, 's1');
        test.throws(() => prog.node.setContext('foo', 'bar'));
        test.done();
    },
    'app.synth() performs validation first and if there are errors, it returns the errors'(test) {
        class Child extends lib_1.Construct {
            validate() {
                return [`Error from ${this.node.id}`];
            }
        }
        class Parent extends lib_1.Stack {
        }
        const app = new app_1.App();
        const parent = new Parent(app, 'Parent');
        new Child(parent, 'C1');
        new Child(parent, 'C2');
        test.throws(() => app.synth(), /Validation failed with the following errors/);
        test.done();
    },
    'app.synthesizeStack(stack) will return a list of missing contextual information'(test) {
        class MyStack extends lib_1.Stack {
            constructor(scope, id, props) {
                super(scope, id, props);
                this.reportMissingContext({
                    key: 'missing-context-key',
                    provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                    props: {
                        account: '12345689012',
                        region: 'ab-north-1',
                    },
                });
                this.reportMissingContext({
                    key: 'missing-context-key-2',
                    provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                    props: {
                        account: '12345689012',
                        region: 'ab-south-1',
                    },
                });
            }
        }
        const assembly = withApp({}, app => {
            new MyStack(app, 'MyStack');
        });
        test.deepEqual(assembly.manifest.missing, [
            {
                key: 'missing-context-key',
                provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                props: {
                    account: '12345689012',
                    region: 'ab-north-1',
                },
            },
            {
                key: 'missing-context-key-2',
                provider: cloud_assembly_schema_1.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
                props: {
                    account: '12345689012',
                    region: 'ab-south-1',
                },
            },
        ]);
        test.done();
    },
    /**
     * Runtime library versions are now synthesized into the Stack templates directly
     *
     * The are not emitted into Cloud Assembly metadata anymore
     */
    'runtime library versions are not emitted in asm anymore'(test) {
        const assembly = withApp({ analyticsReporting: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        test.deepEqual(assembly.runtime, { libraries: {} });
        test.done();
    },
    'runtime library versions'(test) {
        var _a, _b, _c;
        metadata_resource_1.MetadataResource.clearModulesCache();
        const response = withApp({ analyticsReporting: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const stackTemplate = response.getStackByName('stack1').template;
        const libs = parseModules((_c = (_b = (_a = stackTemplate.Resources) === null || _a === void 0 ? void 0 : _a.CDKMetadata) === null || _b === void 0 ? void 0 : _b.Properties) === null || _c === void 0 ? void 0 : _c.Modules);
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const version = require('../package.json').version;
        test.deepEqual(libs['@aws-cdk/core'], version);
        test.deepEqual(libs['@aws-cdk/cx-api'], version);
        test.deepEqual(libs['jsii-runtime'], `node.js/${process.version}`);
        test.done();
    },
    'CDK version'(test) {
        metadata_resource_1.MetadataResource.clearModulesCache();
        withCliVersion(() => {
            var _a, _b, _c;
            const response = withApp({ analyticsReporting: true }, app => {
                const stack = new lib_1.Stack(app, 'stack1');
                new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
            });
            const stackTemplate = response.getStackByName('stack1').template;
            const libs = parseModules((_c = (_b = (_a = stackTemplate.Resources) === null || _a === void 0 ? void 0 : _a.CDKMetadata) === null || _b === void 0 ? void 0 : _b.Properties) === null || _c === void 0 ? void 0 : _c.Modules);
            // eslint-disable-next-line @typescript-eslint/no-require-imports
            test.deepEqual(libs['aws-cdk'], '1.2.3');
        });
        test.done();
    },
    'jsii-runtime version loaded from JSII_AGENT'(test) {
        process.env.JSII_AGENT = 'Java/1.2.3.4';
        metadata_resource_1.MetadataResource.clearModulesCache();
        withCliVersion(() => {
            var _a, _b, _c;
            const response = withApp({ analyticsReporting: true }, app => {
                const stack = new lib_1.Stack(app, 'stack1');
                new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
            });
            const stackTemplate = response.getStackByName('stack1').template;
            const libs = parseModules((_c = (_b = (_a = stackTemplate.Resources) === null || _a === void 0 ? void 0 : _a.CDKMetadata) === null || _b === void 0 ? void 0 : _b.Properties) === null || _c === void 0 ? void 0 : _c.Modules);
            test.deepEqual(libs['jsii-runtime'], 'Java/1.2.3.4');
        });
        delete process.env.JSII_AGENT;
        test.done();
    },
    'version reporting includes only @aws-cdk, aws-cdk and jsii libraries'(test) {
        var _a, _b, _c;
        metadata_resource_1.MetadataResource.clearModulesCache();
        const response = withApp({ analyticsReporting: true }, app => {
            const stack = new lib_1.Stack(app, 'stack1');
            new lib_1.CfnResource(stack, 'MyResource', { type: 'Resource::Type' });
        });
        const stackTemplate = response.getStackByName('stack1').template;
        const libs = parseModules((_c = (_b = (_a = stackTemplate.Resources) === null || _a === void 0 ? void 0 : _a.CDKMetadata) === null || _b === void 0 ? void 0 : _b.Properties) === null || _c === void 0 ? void 0 : _c.Modules);
        const libNames = Object.keys(libs).sort();
        test.deepEqual(libNames, [
            '@aws-cdk/cloud-assembly-schema',
            '@aws-cdk/core',
            '@aws-cdk/cx-api',
            '@aws-cdk/region-info',
            'jsii-runtime',
        ]);
        test.done();
    },
    'deep stack is shown and synthesized properly'(test) {
        // WHEN
        const response = withApp({}, (app) => {
            const topStack = new lib_1.Stack(app, 'Stack');
            const topResource = new lib_1.CfnResource(topStack, 'Res', { type: 'CDK::TopStack::Resource' });
            const bottomStack = new lib_1.Stack(topResource, 'Stack');
            new lib_1.CfnResource(bottomStack, 'Res', { type: 'CDK::BottomStack::Resource' });
        });
        // THEN
        test.deepEqual(response.stacks.map(s => ({ name: s.stackName, template: s.template })), [
            {
                name: 'Stack',
                template: { Resources: { Res: { Type: 'CDK::TopStack::Resource' } } },
            },
            {
                name: 'StackResStack7E4AFA86',
                template: { Resources: { Res: { Type: 'CDK::BottomStack::Resource' } } },
            },
        ]);
        test.done();
    },
    'stacks are written to the assembly file in a topological order'(test) {
        // WHEN
        const assembly = withApp({}, (app) => {
            const stackC = new lib_1.Stack(app, 'StackC');
            const stackD = new lib_1.Stack(app, 'StackD');
            const stackA = new lib_1.Stack(app, 'StackA');
            const stackB = new lib_1.Stack(app, 'StackB');
            // Create the following dependency order:
            // A ->
            //      C -> D
            // B ->
            stackC.addDependency(stackA);
            stackC.addDependency(stackB);
            stackD.addDependency(stackC);
        });
        // THEN
        const artifactsIds = assembly.artifacts.map(a => a.id);
        test.ok(artifactsIds.indexOf('StackA') < artifactsIds.indexOf('StackC'));
        test.ok(artifactsIds.indexOf('StackB') < artifactsIds.indexOf('StackC'));
        test.ok(artifactsIds.indexOf('StackC') < artifactsIds.indexOf('StackD'));
        test.done();
    },
    'application support any type in context'(test) {
        const app = new app_1.App({
            context: {
                isString: 'string',
                isNumber: 10,
                isObject: { isString: 'string', isNumber: 10 },
            },
        });
        test.ok(app.node.tryGetContext('isString') === 'string');
        test.ok(app.node.tryGetContext('isNumber') === 10);
        test.deepEqual(app.node.tryGetContext('isObject'), { isString: 'string', isNumber: 10 });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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