"use strict";
const fs = require("fs");
const os = require("os");
const path = require("path");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cdk = require("../lib");
function createModernApp() {
    return new cdk.App();
}
function list(outdir) {
    return fs.readdirSync(outdir).sort();
}
function readJson(outdir, file) {
    return JSON.parse(fs.readFileSync(path.join(outdir, file), 'utf-8'));
}
function writeJson(outdir, file, data) {
    fs.writeFileSync(path.join(outdir, file), JSON.stringify(data, undefined, 2));
}
module.exports = {
    'synthesis with an empty app'(test) {
        // GIVEN
        const app = createModernApp();
        // WHEN
        const session = app.synth();
        // THEN
        test.same(app.synth(), session); // same session if we synth() again
        test.deepEqual(list(session.directory), ['cdk.out', 'manifest.json', 'tree.json']);
        test.deepEqual(readJson(session.directory, 'manifest.json').artifacts, {
            Tree: {
                type: 'cdk:tree',
                properties: { file: 'tree.json' },
            },
        });
        test.deepEqual(readJson(session.directory, 'tree.json'), {
            version: 'tree-0.1',
            tree: {
                id: 'App',
                path: '',
                children: {
                    Tree: { id: 'Tree', path: 'Tree' },
                },
            },
        });
        test.done();
    },
    'synthesis respects disabling tree metadata'(test) {
        const app = new cdk.App({
            treeMetadata: false,
        });
        const assembly = app.synth();
        test.deepEqual(list(assembly.directory), ['cdk.out', 'manifest.json']);
        test.done();
    },
    'single empty stack'(test) {
        // GIVEN
        const app = createModernApp();
        new cdk.Stack(app, 'one-stack');
        // WHEN
        const session = app.synth();
        // THEN
        test.ok(list(session.directory).includes('one-stack.template.json'));
        test.done();
    },
    'some random construct implements "synthesize"'(test) {
        // GIVEN
        const app = createModernApp();
        const stack = new cdk.Stack(app, 'one-stack');
        class MyConstruct extends cdk.Construct {
            synthesize(s) {
                writeJson(s.assembly.outdir, 'foo.json', { bar: 123 });
                s.assembly.addArtifact('my-random-construct', {
                    type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
                    environment: 'aws://12345/bar',
                    properties: {
                        templateFile: 'foo.json',
                    },
                });
            }
        }
        new MyConstruct(stack, 'MyConstruct');
        // WHEN
        const session = app.synth();
        // THEN
        test.ok(list(session.directory).includes('one-stack.template.json'));
        test.ok(list(session.directory).includes('foo.json'));
        test.deepEqual(readJson(session.directory, 'foo.json'), { bar: 123 });
        test.deepEqual(session.manifest, {
            version: cxschema.Manifest.version(),
            artifacts: {
                'Tree': {
                    type: 'cdk:tree',
                    properties: { file: 'tree.json' },
                },
                'my-random-construct': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://12345/bar',
                    properties: { templateFile: 'foo.json' },
                },
                'one-stack': {
                    type: 'aws:cloudformation:stack',
                    environment: 'aws://unknown-account/unknown-region',
                    properties: { templateFile: 'one-stack.template.json' },
                },
            },
        });
        test.done();
    },
    'it should be possible to synthesize without an app'(test) {
        const calls = new Array();
        class SynthesizeMe extends cdk.Construct {
            constructor() {
                super(undefined, 'id');
            }
            synthesize(session) {
                calls.push('synthesize');
                session.assembly.addArtifact('art', {
                    type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
                    properties: {
                        templateFile: 'hey.json',
                        parameters: {
                            paramId: 'paramValue',
                            paramId2: 'paramValue2',
                        },
                    },
                    environment: 'aws://unknown-account/us-east-1',
                });
                writeJson(session.assembly.outdir, 'hey.json', { hello: 123 });
            }
            validate() {
                calls.push('validate');
                return [];
            }
            prepare() {
                calls.push('prepare');
            }
        }
        const root = new SynthesizeMe();
        const assembly = cdk.ConstructNode.synth(root.node, { outdir: fs.mkdtempSync(path.join(os.tmpdir(), 'outdir')) });
        test.deepEqual(calls, ['prepare', 'validate', 'synthesize']);
        const stack = assembly.getStackByName('art');
        test.deepEqual(stack.template, { hello: 123 });
        test.deepEqual(stack.templateFile, 'hey.json');
        test.deepEqual(stack.parameters, { paramId: 'paramValue', paramId2: 'paramValue2' });
        test.deepEqual(stack.environment, { region: 'us-east-1', account: 'unknown-account', name: 'aws://unknown-account/us-east-1' });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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