# Copyright (c) 2024 Graham R King
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice (including the
# next paragraph) shall be included in all copies or substantial
# portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
# BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
# ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
from __future__ import annotations

from copy import deepcopy


class TypeHinter:
    HEADERS = (
        "# DO NOT EDIT this file, it is automatically generated.",
        "# ",
        "# This file is only used as a code completion helper",
        "# for editors, it is not used at runtime.",
        "",
        "from typing import TypeAlias, Annotated",
        "from enum import Enum, IntFlag",
        "from wayland.proxy import Proxy",
        "new_id: TypeAlias = int",
        "object: TypeAlias = int",
        "uint: TypeAlias = int",
        "string: TypeAlias = str",
        "fd: TypeAlias = int",
        "array: TypeAlias = list",
        "fixed: TypeAlias = float",
        "",
        "def get_package_root() -> str:",
        "    # Returns the directory that this project is sitting in",
        "",
        "    ...",
        "",
        "def initialise(auto: bool = None) -> None or Proxy:",
        "    # Initialise the wayland connection",
        "",
        "    ...",
        "",
        "def process_messages() -> None:",
        "    # Process incoming messages",
        "",
        "    ...",
        "",
    )

    def create_type_hinting(self, structure: dict, path: str) -> None:
        file_name = f"{path}/__init__.pyi"
        headers = [line + "\n" for line in self.HEADERS]
        class_definitions = []

        for class_name, details in structure.items():
            class_declaration = self._create_class_declaration(class_name, details)
            class_body = self._create_class_body(class_name, details)
            class_definitions.append(class_declaration + class_body)

        with open(file_name, "w", encoding="utf-8") as outfile:
            outfile.writelines(headers + class_definitions)

    def _create_class_declaration(self, class_name: str, details: dict) -> str:
        declaration = f"class {class_name}:\n"
        declaration += self.indent(details["description"], 4, comment=True)
        declaration += f"    object_id = 0\n    version = {details['version']}\n\n"
        return declaration

    def _create_class_body(self, class_name: str, details: dict) -> str:
        body = ""
        body += self.process_enums(details.get("enums", []))

        body += self.process_members(class_name, details.get("requests", []))

        events = self.process_members(
            class_name, details.get("events", []), events=True
        )
        if events:
            body += "    class events:\n" + events

        return body

    def process_members(
        self, class_name: str, members: list[dict], *, events: bool = False
    ) -> str:
        indent_declaration = 8 if events else 4
        indent_body = 12 if events else 8
        pad = " " * indent_declaration
        pad_body = " " * indent_body

        definitions = []

        for member in members:
            new_args, return_type = self._process_args(
                class_name, member["args"], events
            )

            signature = [
                f"{pad}@staticmethod",
                f"{pad}def {member['name']}({', '.join(f'{arg['name']}: {arg['type']}' for arg in new_args)})"
                f" -> {return_type or 'None'}:",
                self.indent(member["description"], indent_body),
                f"{pad_body}...\n",
            ]

            definitions.append("\n".join(signature))

        return "\n".join(definitions)

    def _process_args(
        self, class_name: str, args: list[dict], events: bool
    ) -> tuple[list[dict], str | None]:
        new_args = []
        return_type = None

        for arg in deepcopy(args):
            if arg["type"] == "new_id":
                interface = arg.get("interface")
                if interface and not events:
                    return_type = interface
                    continue
                if interface and events:
                    arg["type"] = interface
            elif arg.get("enum"):
                arg["type"] = f"{class_name}.{arg['enum']}"
            new_args.append(arg)

        return new_args, return_type

    def process_enums(self, members: list[dict]) -> str:
        indent_declaration = 4
        indent_body = 8
        pad = " " * indent_declaration
        pad_body = " " * indent_body

        definitions = []

        for member in members:
            enum_name = member["name"]
            enum_type = "IntFlag" if member.get("bitfield") else "Enum"

            signature = [f"{pad}class {enum_name}({enum_type}):"]
            for arg in member["args"]:
                value_name = arg["name"]
                try:
                    int(value_name)
                    value_name = f"{enum_name}_{value_name}"
                except ValueError:
                    pass
                signature.append(f"{pad_body}{value_name}: int")

            definitions.append("\n".join(signature) + "\n\n")

        return "\n".join(definitions)

    @staticmethod
    def indent(input_string: str, indent_columns: int, *, comment: bool = True) -> str:
        indent = " " * indent_columns
        indented_string = "\n".join(indent + line for line in input_string.splitlines())
        if comment:
            indented_string = f'{indent}"""\n{indented_string}\n{indent}"""\n'
        return indented_string
