"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('IAM role', () => {
    test('default role', () => {
        const stack = new core_1.Stack();
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com')
        });
        expect(stack).toMatchTemplate({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'sns.amazonaws.com' } }],
                            Version: '2012-10-17' } } } } });
    });
    test('a role can grant PassRole permissions', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const role = new lib_1.Role(stack, 'Role', { assumedBy: new lib_1.ServicePrincipal('henk.amazonaws.com') });
        const user = new lib_1.User(stack, 'User');
        // WHEN
        role.grantPassRole(user);
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "iam:PassRole",
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["Role1ABCC5F0", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            },
        });
    });
    test('can supply externalId', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            externalId: 'SomeSecret',
        });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Condition: {
                            StringEquals: { "sts:ExternalId": "SomeSecret" }
                        },
                        Effect: "Allow",
                        Principal: { Service: "sns.amazonaws.com" }
                    }
                ],
                Version: "2012-10-17"
            }
        });
    });
    test('can supply single externalIds', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            externalIds: ['SomeSecret'],
        });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Condition: {
                            StringEquals: { "sts:ExternalId": "SomeSecret" }
                        },
                        Effect: "Allow",
                        Principal: { Service: "sns.amazonaws.com" }
                    }
                ],
                Version: "2012-10-17"
            }
        });
    });
    test('can supply multiple externalIds', () => {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            externalIds: ['SomeSecret', 'AnotherSecret'],
        });
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Condition: {
                            StringEquals: { "sts:ExternalId": ["SomeSecret", "AnotherSecret"] }
                        },
                        Effect: "Allow",
                        Principal: { Service: "sns.amazonaws.com" }
                    }
                ],
                Version: "2012-10-17"
            }
        });
    });
    test('policy is created automatically when permissions are added', () => {
        // by default we don't expect a role policy
        const before = new core_1.Stack();
        new lib_1.Role(before, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        expect(before).not.toHaveResource('AWS::IAM::Policy');
        // add a policy to the role
        const after = new core_1.Stack();
        const afterRole = new lib_1.Role(after, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
        afterRole.addToPolicy(new lib_1.PolicyStatement({ resources: ['myresource'], actions: ['service:myaction'] }));
        expect(after).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "service:myaction",
                        Effect: "Allow",
                        Resource: "myresource"
                    }
                ],
                Version: "2012-10-17"
            },
            PolicyName: "MyRoleDefaultPolicyA36BE1DD",
            Roles: [
                {
                    Ref: "MyRoleF48FFE04"
                }
            ]
        });
    });
    test('managed policy arns can be supplied upon initialization and also added later', () => {
        const stack = new core_1.Stack();
        const role = new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('test.service'),
            managedPolicies: [{ managedPolicyArn: 'managed1' }, { managedPolicyArn: 'managed2' }]
        });
        role.addManagedPolicy({ managedPolicyArn: 'managed3' });
        expect(stack).toMatchTemplate({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'test.service' } }],
                            Version: '2012-10-17' },
                        ManagedPolicyArns: ['managed1', 'managed2', 'managed3'] } } } });
    });
    test('federated principal can change AssumeRoleAction', () => {
        const stack = new core_1.Stack();
        const cognitoPrincipal = new lib_1.FederatedPrincipal('foo', { StringEquals: { key: 'value' } }, 'sts:AssumeSomething');
        new lib_1.Role(stack, 'MyRole', { assumedBy: cognitoPrincipal });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Version: "2012-10-17",
                Statement: [
                    {
                        Principal: { Federated: "foo" },
                        Condition: {
                            StringEquals: { key: "value" }
                        },
                        Action: "sts:AssumeSomething",
                        Effect: "Allow",
                    }
                ],
            }
        });
    });
    describe('maxSessionDuration', () => {
        test('is not specified by default', () => {
            const stack = new core_1.Stack();
            new lib_1.Role(stack, 'MyRole', { assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
            expect(stack).toMatchTemplate({
                Resources: {
                    MyRoleF48FFE04: {
                        Type: "AWS::IAM::Role",
                        Properties: {
                            AssumeRolePolicyDocument: {
                                Statement: [
                                    {
                                        Action: "sts:AssumeRole",
                                        Effect: "Allow",
                                        Principal: {
                                            Service: "sns.amazonaws.com"
                                        }
                                    }
                                ],
                                Version: "2012-10-17"
                            }
                        }
                    }
                }
            });
        });
        test('can be used to specify the maximum session duration for assuming the role', () => {
            const stack = new core_1.Stack();
            new lib_1.Role(stack, 'MyRole', { maxSessionDuration: core_1.Duration.seconds(3700), assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com') });
            expect(stack).toHaveResource('AWS::IAM::Role', {
                MaxSessionDuration: 3700
            });
        });
        test('must be between 3600 and 43200', () => {
            const stack = new core_1.Stack();
            const assumedBy = new lib_1.ServicePrincipal('bla');
            new lib_1.Role(stack, 'MyRole1', { assumedBy, maxSessionDuration: core_1.Duration.hours(1) });
            new lib_1.Role(stack, 'MyRole2', { assumedBy, maxSessionDuration: core_1.Duration.hours(12) });
            const expected = (val) => `maxSessionDuration is set to ${val}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`;
            expect(() => new lib_1.Role(stack, 'MyRole3', { assumedBy, maxSessionDuration: core_1.Duration.minutes(1) }))
                .toThrow(expected(60));
            expect(() => new lib_1.Role(stack, 'MyRole4', { assumedBy, maxSessionDuration: core_1.Duration.seconds(3599) }))
                .toThrow(expected(3599));
            expect(() => new lib_1.Role(stack, 'MyRole5', { assumedBy, maxSessionDuration: core_1.Duration.seconds(43201) }))
                .toThrow(expected(43201));
        });
    });
    test('allow role with multiple principals', () => {
        const stack = new core_1.Stack();
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.CompositePrincipal(new lib_1.ServicePrincipal('boom.amazonaws.test'), new lib_1.ArnPrincipal('1111111'))
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "boom.amazonaws.test",
                            AWS: "1111111"
                        }
                    }
                ],
                Version: "2012-10-17"
            }
        });
    });
    test('can supply permissions boundary managed policy', () => {
        // GIVEN
        const stack = new core_1.Stack();
        const permissionsBoundary = lib_1.ManagedPolicy.fromAwsManagedPolicyName('managed-policy');
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            permissionsBoundary,
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            PermissionsBoundary: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":iam::aws:policy/managed-policy"
                    ]
                ]
            }
        });
    });
    test('Principal-* in an AssumeRolePolicyDocument gets translated to { "AWS": "*" }', () => {
        // The docs say that "Principal: *" and "Principal: { AWS: * }" are equivalent
        // (https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_principal.html)
        // but in practice CreateRole errors out if you use "Principal: *" in an AssumeRolePolicyDocument:
        // An error occurred (MalformedPolicyDocument) when calling the CreateRole operation: AssumeRolepolicy contained an invalid principal: "STAR":"*".
        // Make sure that we handle this case specially.
        const stack = new core_1.Stack();
        new lib_1.Role(stack, 'Role', {
            assumedBy: new lib_1.AnyPrincipal(),
        });
        expect(stack).toHaveResource('AWS::IAM::Role', {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: { AWS: "*" },
                    }
                ],
                Version: "2012-10-17"
            }
        });
    });
    test('can have a description', () => {
        const stack = new core_1.Stack();
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            description: "This is a role description."
        });
        expect(stack).toMatchTemplate({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'sns.amazonaws.com' } }],
                            Version: '2012-10-17' },
                        Description: 'This is a role description.' } } } });
    });
    test('should not have an empty description', () => {
        const stack = new core_1.Stack();
        new lib_1.Role(stack, 'MyRole', {
            assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
            description: ""
        });
        expect(stack).toMatchTemplate({ Resources: { MyRoleF48FFE04: { Type: 'AWS::IAM::Role',
                    Properties: { AssumeRolePolicyDocument: { Statement: [{ Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'sns.amazonaws.com' } }],
                            Version: '2012-10-17' } } } } });
    });
    test('description can only be 1000 characters long', () => {
        const stack = new core_1.Stack();
        expect(() => {
            new lib_1.Role(stack, 'MyRole', {
                assumedBy: new lib_1.ServicePrincipal('sns.amazonaws.com'),
                description: "1000+ character long description: Lorem ipsum dolor sit amet, consectetuer adipiscing elit. \
        Aenean commodo ligula eget dolor. Aenean massa. Cum sociis natoque penatibus et magnis dis parturient montes, \
        nascetur ridiculus mus. Donec quam felis, ultricies nec, pellentesque eu, pretium quis, sem. Nulla consequat \
        massa quis enim. Donec pede justo, fringilla vel, aliquet nec, vulputate eget, arcu. In enim justo, rhoncus ut, \
        imperdiet a, venenatis vitae, justo. Nullam dictum felis eu pede mollis pretium. Integer tincidunt. Cras dapibus. \
        Vivamus elementum semper nisi. Aenean vulputate eleifend tellus. Aenean leo ligula, porttitor eu, consequat vitae, \
        eleifend ac, enim. Aliquam lorem ante, dapibus in, viverra quis, feugiat a, tellus. Phasellus viverra nulla ut metus \
        varius laoreet. Quisque rutrum. Aenean imperdiet. Etiam ultricies nisi vel augue. Curabitur ullamcorper ultricies nisi. \
        Nam eget dui. Etiam rhoncus. Maecenas tempus, tellus eget condimentum rhoncus, sem quam semper libero, sit amet adipiscing \
        sem neque sed ipsum."
            });
        }).toThrow(/Role description must be no longer than 1000 characters./);
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicm9sZS50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsicm9sZS50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsZ0NBQThCO0FBQzlCLHdDQUFnRDtBQUNoRCxnQ0FBMEo7QUFFMUosUUFBUSxDQUFDLFVBQVUsRUFBRSxHQUFHLEVBQUU7SUFDeEIsSUFBSSxDQUFDLGNBQWMsRUFBRSxHQUFHLEVBQUU7UUFDeEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixJQUFJLFVBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ3hCLFNBQVMsRUFBRSxJQUFJLHNCQUFnQixDQUFDLG1CQUFtQixDQUFDO1NBQ3JELENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxlQUFlLENBQUMsRUFBRSxTQUFTLEVBQ3ZDLEVBQUUsY0FBYyxFQUNiLEVBQUUsSUFBSSxFQUFFLGdCQUFnQjtvQkFDeEIsVUFBVSxFQUNULEVBQUUsd0JBQXdCLEVBQ3pCLEVBQUUsU0FBUyxFQUNWLENBQUUsRUFBRSxNQUFNLEVBQUUsZ0JBQWdCO29DQUMxQixNQUFNLEVBQUUsT0FBTztvQ0FDZixTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsbUJBQW1CLEVBQUUsRUFBRSxDQUFFOzRCQUNoRCxPQUFPLEVBQUUsWUFBWSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsQ0FBQyxDQUFDO0lBQzVDLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHVDQUF1QyxFQUFFLEdBQUcsRUFBRTtRQUNqRCxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUMxQixNQUFNLElBQUksR0FBRyxJQUFJLFVBQUksQ0FBQyxLQUFLLEVBQUUsTUFBTSxFQUFFLEVBQUUsU0FBUyxFQUFFLElBQUksc0JBQWdCLENBQUMsb0JBQW9CLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDaEcsTUFBTSxJQUFJLEdBQUcsSUFBSSxVQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBRXJDLE9BQU87UUFDUCxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRXpCLE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsa0JBQWtCLENBQUMsa0JBQWtCLEVBQUU7WUFDbkQsY0FBYyxFQUFFO2dCQUNkLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxNQUFNLEVBQUUsY0FBYzt3QkFDdEIsTUFBTSxFQUFFLE9BQU87d0JBQ2YsUUFBUSxFQUFFLEVBQUUsWUFBWSxFQUFFLENBQUUsY0FBYyxFQUFFLEtBQUssQ0FBRSxFQUFFO3FCQUN0RDtpQkFDRjtnQkFDRCxPQUFPLEVBQUUsWUFBWTthQUN0QjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHVCQUF1QixFQUFFLEdBQUcsRUFBRTtRQUNqQyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixPQUFPO1FBQ1AsSUFBSSxVQUFJLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUN4QixTQUFTLEVBQUUsSUFBSSxzQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztZQUNwRCxVQUFVLEVBQUUsWUFBWTtTQUN6QixDQUFDLENBQUM7UUFFSCxPQUFPO1FBQ1AsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRTtZQUM3Qyx3QkFBd0IsRUFBRTtnQkFDeEIsU0FBUyxFQUFFO29CQUNUO3dCQUNFLE1BQU0sRUFBRSxnQkFBZ0I7d0JBQ3hCLFNBQVMsRUFBRTs0QkFDVCxZQUFZLEVBQUUsRUFBRSxnQkFBZ0IsRUFBRSxZQUFZLEVBQUU7eUJBQ2pEO3dCQUNELE1BQU0sRUFBRSxPQUFPO3dCQUNmLFNBQVMsRUFBRSxFQUFFLE9BQU8sRUFBRSxtQkFBbUIsRUFBRTtxQkFDNUM7aUJBQ0Y7Z0JBQ0QsT0FBTyxFQUFFLFlBQVk7YUFDdEI7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQywrQkFBK0IsRUFBRSxHQUFHLEVBQUU7UUFDekMsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFFMUIsT0FBTztRQUNQLElBQUksVUFBSSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDeEIsU0FBUyxFQUFFLElBQUksc0JBQWdCLENBQUMsbUJBQW1CLENBQUM7WUFDcEQsV0FBVyxFQUFFLENBQUMsWUFBWSxDQUFDO1NBQzVCLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO1lBQzdDLHdCQUF3QixFQUFFO2dCQUN4QixTQUFTLEVBQUU7b0JBQ1Q7d0JBQ0UsTUFBTSxFQUFFLGdCQUFnQjt3QkFDeEIsU0FBUyxFQUFFOzRCQUNULFlBQVksRUFBRSxFQUFFLGdCQUFnQixFQUFFLFlBQVksRUFBRTt5QkFDakQ7d0JBQ0QsTUFBTSxFQUFFLE9BQU87d0JBQ2YsU0FBUyxFQUFFLEVBQUUsT0FBTyxFQUFFLG1CQUFtQixFQUFFO3FCQUM1QztpQkFDRjtnQkFDRCxPQUFPLEVBQUUsWUFBWTthQUN0QjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLGlDQUFpQyxFQUFFLEdBQUcsRUFBRTtRQUMzQyxRQUFRO1FBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixPQUFPO1FBQ1AsSUFBSSxVQUFJLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUN4QixTQUFTLEVBQUUsSUFBSSxzQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztZQUNwRCxXQUFXLEVBQUUsQ0FBQyxZQUFZLEVBQUUsZUFBZSxDQUFDO1NBQzdDLENBQUMsQ0FBQztRQUVILE9BQU87UUFDUCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO1lBQzdDLHdCQUF3QixFQUFFO2dCQUN4QixTQUFTLEVBQUU7b0JBQ1Q7d0JBQ0UsTUFBTSxFQUFFLGdCQUFnQjt3QkFDeEIsU0FBUyxFQUFFOzRCQUNULFlBQVksRUFBRSxFQUFFLGdCQUFnQixFQUFFLENBQUMsWUFBWSxFQUFFLGVBQWUsQ0FBQyxFQUFFO3lCQUNwRTt3QkFDRCxNQUFNLEVBQUUsT0FBTzt3QkFDZixTQUFTLEVBQUUsRUFBRSxPQUFPLEVBQUUsbUJBQW1CLEVBQUU7cUJBQzVDO2lCQUNGO2dCQUNELE9BQU8sRUFBRSxZQUFZO2FBQ3RCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsNERBQTRELEVBQUUsR0FBRyxFQUFFO1FBQ3RFLDJDQUEyQztRQUMzQyxNQUFNLE1BQU0sR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzNCLElBQUksVUFBSSxDQUFDLE1BQU0sRUFBRSxRQUFRLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxzQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUNyRixNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBRXRELDJCQUEyQjtRQUMzQixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLE1BQU0sU0FBUyxHQUFHLElBQUksVUFBSSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxzQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUN0RyxTQUFTLENBQUMsV0FBVyxDQUFDLElBQUkscUJBQWUsQ0FBQyxFQUFFLFNBQVMsRUFBRSxDQUFDLFlBQVksQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLGtCQUFrQixDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDekcsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRTtZQUMvQyxjQUFjLEVBQUU7Z0JBQ2QsU0FBUyxFQUFFO29CQUNUO3dCQUNFLE1BQU0sRUFBRSxrQkFBa0I7d0JBQzFCLE1BQU0sRUFBRSxPQUFPO3dCQUNmLFFBQVEsRUFBRSxZQUFZO3FCQUN2QjtpQkFDRjtnQkFDRCxPQUFPLEVBQUUsWUFBWTthQUN0QjtZQUNELFVBQVUsRUFBRSw2QkFBNkI7WUFDekMsS0FBSyxFQUFFO2dCQUNMO29CQUNFLEdBQUcsRUFBRSxnQkFBZ0I7aUJBQ3RCO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFFTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw4RUFBOEUsRUFBRSxHQUFHLEVBQUU7UUFDeEYsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixNQUFNLElBQUksR0FBRyxJQUFJLFVBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ3JDLFNBQVMsRUFBRSxJQUFJLHNCQUFnQixDQUFDLGNBQWMsQ0FBQztZQUMvQyxlQUFlLEVBQUUsQ0FBRSxFQUFFLGdCQUFnQixFQUFFLFVBQVUsRUFBRSxFQUFFLEVBQUUsZ0JBQWdCLEVBQUUsVUFBVSxFQUFFLENBQUU7U0FDeEYsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEVBQUUsZ0JBQWdCLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQztRQUN4RCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsZUFBZSxDQUFDLEVBQUUsU0FBUyxFQUN2QyxFQUFFLGNBQWMsRUFDYixFQUFFLElBQUksRUFBRSxnQkFBZ0I7b0JBQ3hCLFVBQVUsRUFDVCxFQUFFLHdCQUF3QixFQUN6QixFQUFFLFNBQVMsRUFDVixDQUFFLEVBQUUsTUFBTSxFQUFFLGdCQUFnQjtvQ0FDMUIsTUFBTSxFQUFFLE9BQU87b0NBQ2YsU0FBUyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxFQUFFLENBQUU7NEJBQzNDLE9BQU8sRUFBRSxZQUFZLEVBQUU7d0JBQzFCLGlCQUFpQixFQUFFLENBQUUsVUFBVSxFQUFFLFVBQVUsRUFBRSxVQUFVLENBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFFM0UsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsaURBQWlELEVBQUUsR0FBRyxFQUFFO1FBQzNELE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFDMUIsTUFBTSxnQkFBZ0IsR0FBRyxJQUFJLHdCQUFrQixDQUM3QyxLQUFLLEVBQ0wsRUFBRSxZQUFZLEVBQUUsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFLEVBQUUsRUFDbEMscUJBQXFCLENBQUMsQ0FBQztRQUV6QixJQUFJLFVBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLEVBQUUsU0FBUyxFQUFFLGdCQUFnQixFQUFFLENBQUMsQ0FBQztRQUUzRCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO1lBQzdDLHdCQUF3QixFQUFFO2dCQUN4QixPQUFPLEVBQUUsWUFBWTtnQkFDckIsU0FBUyxFQUFFO29CQUNUO3dCQUNFLFNBQVMsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUU7d0JBQy9CLFNBQVMsRUFBRTs0QkFDVCxZQUFZLEVBQUUsRUFBRSxHQUFHLEVBQUUsT0FBTyxFQUFFO3lCQUMvQjt3QkFDRCxNQUFNLEVBQUUscUJBQXFCO3dCQUM3QixNQUFNLEVBQUUsT0FBTztxQkFDaEI7aUJBQ0Y7YUFDRjtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLG9CQUFvQixFQUFFLEdBQUcsRUFBRTtRQUVsQyxJQUFJLENBQUMsNkJBQTZCLEVBQUUsR0FBRyxFQUFFO1lBQ3ZDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7WUFDMUIsSUFBSSxVQUFJLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRSxFQUFFLFNBQVMsRUFBRSxJQUFJLHNCQUFnQixDQUFDLG1CQUFtQixDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBQ3BGLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxlQUFlLENBQUM7Z0JBQzVCLFNBQVMsRUFBRTtvQkFDVCxjQUFjLEVBQUU7d0JBQ2hCLElBQUksRUFBRSxnQkFBZ0I7d0JBQ3RCLFVBQVUsRUFBRTs0QkFDVix3QkFBd0IsRUFBRTtnQ0FDMUIsU0FBUyxFQUFFO29DQUNUO3dDQUNBLE1BQU0sRUFBRSxnQkFBZ0I7d0NBQ3hCLE1BQU0sRUFBRSxPQUFPO3dDQUNmLFNBQVMsRUFBRTs0Q0FDVCxPQUFPLEVBQUUsbUJBQW1CO3lDQUM3QjtxQ0FDQTtpQ0FDRjtnQ0FDRCxPQUFPLEVBQUUsWUFBWTs2QkFDcEI7eUJBQ0Y7cUJBQ0E7aUJBQ0Y7YUFDRixDQUFDLENBQUM7UUFDTCxDQUFDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQywyRUFBMkUsRUFBRSxHQUFHLEVBQUU7WUFDckYsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztZQUUxQixJQUFJLFVBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFLEVBQUUsa0JBQWtCLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRSxTQUFTLEVBQUUsSUFBSSxzQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUVoSSxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO2dCQUM3QyxrQkFBa0IsRUFBRSxJQUFJO2FBQ3pCLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGdDQUFnQyxFQUFFLEdBQUcsRUFBRTtZQUMxQyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1lBRTFCLE1BQU0sU0FBUyxHQUFHLElBQUksc0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUM7WUFFOUMsSUFBSSxVQUFJLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxFQUFFLFNBQVMsRUFBRSxrQkFBa0IsRUFBRSxlQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUNqRixJQUFJLFVBQUksQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLEVBQUUsU0FBUyxFQUFFLGtCQUFrQixFQUFFLGVBQVEsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1lBRWxGLE1BQU0sUUFBUSxHQUFHLENBQUMsR0FBUSxFQUFFLEVBQUUsQ0FBQyxnQ0FBZ0MsR0FBRyx3REFBd0QsQ0FBQztZQUMzSCxNQUFNLENBQUMsR0FBRyxFQUFFLENBQUMsSUFBSSxVQUFJLENBQUMsS0FBSyxFQUFFLFNBQVMsRUFBRSxFQUFFLFNBQVMsRUFBRSxrQkFBa0IsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztpQkFDN0YsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBQ3pCLE1BQU0sQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLFVBQUksQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFLEVBQUUsU0FBUyxFQUFFLGtCQUFrQixFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO2lCQUNoRyxPQUFPLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7WUFDM0IsTUFBTSxDQUFDLEdBQUcsRUFBRSxDQUFDLElBQUksVUFBSSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUUsRUFBRSxTQUFTLEVBQUUsa0JBQWtCLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLENBQUM7aUJBQ2pHLE9BQU8sQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUM5QixDQUFDLENBQUMsQ0FBQztJQUNMLENBQUMsQ0FBQyxDQUFDO0lBRUgsSUFBSSxDQUFDLHFDQUFxQyxFQUFFLEdBQUcsRUFBRTtRQUMvQyxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBRTFCLElBQUksVUFBSSxDQUFDLEtBQUssRUFBRSxRQUFRLEVBQUU7WUFDeEIsU0FBUyxFQUFFLElBQUksd0JBQWtCLENBQy9CLElBQUksc0JBQWdCLENBQUMscUJBQXFCLENBQUMsRUFDM0MsSUFBSSxrQkFBWSxDQUFDLFNBQVMsQ0FBQyxDQUM1QjtTQUNGLENBQUMsQ0FBQztRQUVILE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLEVBQUU7WUFDN0Msd0JBQXdCLEVBQUU7Z0JBQ3hCLFNBQVMsRUFBRTtvQkFDVDt3QkFDRSxNQUFNLEVBQUUsZ0JBQWdCO3dCQUN4QixNQUFNLEVBQUUsT0FBTzt3QkFDZixTQUFTLEVBQUU7NEJBQ1QsT0FBTyxFQUFFLHFCQUFxQjs0QkFDOUIsR0FBRyxFQUFFLFNBQVM7eUJBQ2Y7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsT0FBTyxFQUFFLFlBQVk7YUFDdEI7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyxnREFBZ0QsRUFBRSxHQUFHLEVBQUU7UUFDMUQsUUFBUTtRQUNSLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFFMUIsTUFBTSxtQkFBbUIsR0FBRyxtQkFBYSxDQUFDLHdCQUF3QixDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFFckYsSUFBSSxVQUFJLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUN4QixTQUFTLEVBQUUsSUFBSSxzQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztZQUNwRCxtQkFBbUI7U0FDcEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRTtZQUM3QyxtQkFBbUIsRUFBRTtnQkFDbkIsVUFBVSxFQUFFO29CQUNWLEVBQUU7b0JBQ0Y7d0JBQ0UsTUFBTTt3QkFDTjs0QkFDRSxHQUFHLEVBQUUsZ0JBQWdCO3lCQUN0Qjt3QkFDRCxpQ0FBaUM7cUJBQ2xDO2lCQUNGO2FBQ0Y7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw4RUFBOEUsRUFBRSxHQUFHLEVBQUU7UUFDeEYsOEVBQThFO1FBQzlFLGdHQUFnRztRQUNoRyxrR0FBa0c7UUFDbEcsa0pBQWtKO1FBRWxKLGdEQUFnRDtRQUNoRCxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO1FBQzFCLElBQUksVUFBSSxDQUFDLEtBQUssRUFBRSxNQUFNLEVBQUU7WUFDdEIsU0FBUyxFQUFFLElBQUksa0JBQVksRUFBRTtTQUM5QixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsY0FBYyxDQUFDLGdCQUFnQixFQUFFO1lBQzdDLHdCQUF3QixFQUFFO2dCQUN4QixTQUFTLEVBQUU7b0JBQ1Q7d0JBQ0UsTUFBTSxFQUFFLGdCQUFnQjt3QkFDeEIsTUFBTSxFQUFFLE9BQU87d0JBQ2YsU0FBUyxFQUFFLEVBQUUsR0FBRyxFQUFFLEdBQUcsRUFBRTtxQkFDeEI7aUJBQ0Y7Z0JBQ0QsT0FBTyxFQUFFLFlBQVk7YUFDdEI7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyx3QkFBd0IsRUFBRSxHQUFHLEVBQUU7UUFDbEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixJQUFJLFVBQUksQ0FBQyxLQUFLLEVBQUUsUUFBUSxFQUFFO1lBQ3hCLFNBQVMsRUFBRSxJQUFJLHNCQUFnQixDQUFDLG1CQUFtQixDQUFDO1lBQ3BELFdBQVcsRUFBRSw2QkFBNkI7U0FDM0MsQ0FBQyxDQUFDO1FBRUgsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGVBQWUsQ0FBQyxFQUFFLFNBQVMsRUFDdkMsRUFBRSxjQUFjLEVBQ2IsRUFBRSxJQUFJLEVBQUUsZ0JBQWdCO29CQUN4QixVQUFVLEVBQ1QsRUFBRSx3QkFBd0IsRUFDekIsRUFBRSxTQUFTLEVBQ1YsQ0FBRSxFQUFFLE1BQU0sRUFBRSxnQkFBZ0I7b0NBQzFCLE1BQU0sRUFBRSxPQUFPO29DQUNmLFNBQVMsRUFBRSxFQUFFLE9BQU8sRUFBRSxtQkFBbUIsRUFBRSxFQUFFLENBQUU7NEJBQy9DLE9BQU8sRUFBRSxZQUFZLEVBQUU7d0JBQ3pCLFdBQVcsRUFBRSw2QkFBNkIsRUFBRSxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDOUQsQ0FBQyxDQUFDLENBQUM7SUFFSCxJQUFJLENBQUMsc0NBQXNDLEVBQUUsR0FBRyxFQUFFO1FBQ2hELE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7UUFFMUIsSUFBSSxVQUFJLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtZQUN4QixTQUFTLEVBQUUsSUFBSSxzQkFBZ0IsQ0FBQyxtQkFBbUIsQ0FBQztZQUNwRCxXQUFXLEVBQUUsRUFBRTtTQUNoQixDQUFDLENBQUM7UUFFSCxNQUFNLENBQUMsS0FBSyxDQUFDLENBQUMsZUFBZSxDQUFDLEVBQUUsU0FBUyxFQUN2QyxFQUFFLGNBQWMsRUFDYixFQUFFLElBQUksRUFBRSxnQkFBZ0I7b0JBQ3hCLFVBQVUsRUFDVCxFQUFFLHdCQUF3QixFQUN6QixFQUFFLFNBQVMsRUFDVixDQUFFLEVBQUUsTUFBTSxFQUFFLGdCQUFnQjtvQ0FDMUIsTUFBTSxFQUFFLE9BQU87b0NBQ2YsU0FBUyxFQUFFLEVBQUUsT0FBTyxFQUFFLG1CQUFtQixFQUFFLEVBQUUsQ0FBRTs0QkFDL0MsT0FBTyxFQUFFLFlBQVksRUFBRSxFQUFDLEVBQUUsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUM1QyxDQUFDLENBQUMsQ0FBQztJQUVILElBQUksQ0FBQyw4Q0FBOEMsRUFBRSxHQUFHLEVBQUU7UUFDeEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztRQUUxQixNQUFNLENBQUMsR0FBRyxFQUFFO1lBQ1YsSUFBSSxVQUFJLENBQUMsS0FBSyxFQUFFLFFBQVEsRUFBRTtnQkFDeEIsU0FBUyxFQUFFLElBQUksc0JBQWdCLENBQUMsbUJBQW1CLENBQUM7Z0JBQ3BELFdBQVcsRUFBRTs7Ozs7Ozs7OzZCQVNRO2FBQ3RCLENBQUMsQ0FBQztRQUNMLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQywwREFBMEQsQ0FBQyxDQUFDO0lBQ3pFLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgJ0Bhd3MtY2RrL2Fzc2VydC9qZXN0JztcbmltcG9ydCB7IER1cmF0aW9uLCBTdGFjayB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQW55UHJpbmNpcGFsLCBBcm5QcmluY2lwYWwsIENvbXBvc2l0ZVByaW5jaXBhbCwgRmVkZXJhdGVkUHJpbmNpcGFsLCBNYW5hZ2VkUG9saWN5LCBQb2xpY3lTdGF0ZW1lbnQsIFJvbGUsIFNlcnZpY2VQcmluY2lwYWwsIFVzZXIgfSBmcm9tICcuLi9saWInO1xuXG5kZXNjcmliZSgnSUFNIHJvbGUnLCAoKSA9PiB7XG4gIHRlc3QoJ2RlZmF1bHQgcm9sZScsICgpID0+IHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgbmV3IFJvbGUoc3RhY2ssICdNeVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdzbnMuYW1hem9uYXdzLmNvbScpXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2hUZW1wbGF0ZSh7IFJlc291cmNlczpcbiAgICAgIHsgTXlSb2xlRjQ4RkZFMDQ6XG4gICAgICAgICB7IFR5cGU6ICdBV1M6OklBTTo6Um9sZScsXG4gICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgIHsgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OlxuICAgICAgICAgICB7IFN0YXRlbWVudDpcbiAgICAgICAgICAgIFsgeyBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgICAgUHJpbmNpcGFsOiB7IFNlcnZpY2U6ICdzbnMuYW1hem9uYXdzLmNvbScgfSB9IF0sXG4gICAgICAgICAgICAgVmVyc2lvbjogJzIwMTItMTAtMTcnIH0gfSB9IH0gfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2Egcm9sZSBjYW4gZ3JhbnQgUGFzc1JvbGUgcGVybWlzc2lvbnMnLCAoKSA9PiB7XG4gICAgLy8gR0lWRU5cbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHJvbGUgPSBuZXcgUm9sZShzdGFjaywgJ1JvbGUnLCB7IGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ2hlbmsuYW1hem9uYXdzLmNvbScpIH0pO1xuICAgIGNvbnN0IHVzZXIgPSBuZXcgVXNlcihzdGFjaywgJ1VzZXInKTtcblxuICAgIC8vIFdIRU5cbiAgICByb2xlLmdyYW50UGFzc1JvbGUodXNlcik7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246IFwiaWFtOlBhc3NSb2xlXCIsXG4gICAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICAgIFJlc291cmNlOiB7IFwiRm46OkdldEF0dFwiOiBbIFwiUm9sZTFBQkNDNUYwXCIsIFwiQXJuXCIgXSB9XG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgICAgfSxcbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnY2FuIHN1cHBseSBleHRlcm5hbElkJywgKCkgPT4ge1xuICAgIC8vIEdJVkVOXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIC8vIFdIRU5cbiAgICBuZXcgUm9sZShzdGFjaywgJ015Um9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ3Nucy5hbWF6b25hd3MuY29tJyksXG4gICAgICBleHRlcm5hbElkOiAnU29tZVNlY3JldCcsXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBcInN0czpBc3N1bWVSb2xlXCIsXG4gICAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7IFwic3RzOkV4dGVybmFsSWRcIjogXCJTb21lU2VjcmV0XCIgfVxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgICAgUHJpbmNpcGFsOiB7IFNlcnZpY2U6IFwic25zLmFtYXpvbmF3cy5jb21cIiB9XG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgICAgfVxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdjYW4gc3VwcGx5IHNpbmdsZSBleHRlcm5hbElkcycsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IFJvbGUoc3RhY2ssICdNeVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdzbnMuYW1hem9uYXdzLmNvbScpLFxuICAgICAgZXh0ZXJuYWxJZHM6IFsnU29tZVNlY3JldCddLFxuICAgIH0pO1xuXG4gICAgLy8gVEhFTlxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpSb2xlJywge1xuICAgICAgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogXCJzdHM6QXNzdW1lUm9sZVwiLFxuICAgICAgICAgICAgQ29uZGl0aW9uOiB7XG4gICAgICAgICAgICAgIFN0cmluZ0VxdWFsczogeyBcInN0czpFeHRlcm5hbElkXCI6IFwiU29tZVNlY3JldFwiIH1cbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICAgIFByaW5jaXBhbDogeyBTZXJ2aWNlOiBcInNucy5hbWF6b25hd3MuY29tXCIgfVxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICAgIH1cbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnY2FuIHN1cHBseSBtdWx0aXBsZSBleHRlcm5hbElkcycsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAvLyBXSEVOXG4gICAgbmV3IFJvbGUoc3RhY2ssICdNeVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdzbnMuYW1hem9uYXdzLmNvbScpLFxuICAgICAgZXh0ZXJuYWxJZHM6IFsnU29tZVNlY3JldCcsICdBbm90aGVyU2VjcmV0J10sXG4gICAgfSk7XG5cbiAgICAvLyBUSEVOXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBcInN0czpBc3N1bWVSb2xlXCIsXG4gICAgICAgICAgICBDb25kaXRpb246IHtcbiAgICAgICAgICAgICAgU3RyaW5nRXF1YWxzOiB7IFwic3RzOkV4dGVybmFsSWRcIjogW1wiU29tZVNlY3JldFwiLCBcIkFub3RoZXJTZWNyZXRcIl0gfVxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgICAgUHJpbmNpcGFsOiB7IFNlcnZpY2U6IFwic25zLmFtYXpvbmF3cy5jb21cIiB9XG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgICAgfVxuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdwb2xpY3kgaXMgY3JlYXRlZCBhdXRvbWF0aWNhbGx5IHdoZW4gcGVybWlzc2lvbnMgYXJlIGFkZGVkJywgKCkgPT4ge1xuICAgIC8vIGJ5IGRlZmF1bHQgd2UgZG9uJ3QgZXhwZWN0IGEgcm9sZSBwb2xpY3lcbiAgICBjb25zdCBiZWZvcmUgPSBuZXcgU3RhY2soKTtcbiAgICBuZXcgUm9sZShiZWZvcmUsICdNeVJvbGUnLCB7IGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ3Nucy5hbWF6b25hd3MuY29tJykgfSk7XG4gICAgZXhwZWN0KGJlZm9yZSkubm90LnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6UG9saWN5Jyk7XG5cbiAgICAvLyBhZGQgYSBwb2xpY3kgdG8gdGhlIHJvbGVcbiAgICBjb25zdCBhZnRlciA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IGFmdGVyUm9sZSA9IG5ldyBSb2xlKGFmdGVyLCAnTXlSb2xlJywgeyBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdzbnMuYW1hem9uYXdzLmNvbScpIH0pO1xuICAgIGFmdGVyUm9sZS5hZGRUb1BvbGljeShuZXcgUG9saWN5U3RhdGVtZW50KHsgcmVzb3VyY2VzOiBbJ215cmVzb3VyY2UnXSwgYWN0aW9uczogWydzZXJ2aWNlOm15YWN0aW9uJ10gfSkpO1xuICAgIGV4cGVjdChhZnRlcikudG9IYXZlUmVzb3VyY2UoJ0FXUzo6SUFNOjpQb2xpY3knLCB7XG4gICAgICBQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246IFwic2VydmljZTpteWFjdGlvblwiLFxuICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICBSZXNvdXJjZTogXCJteXJlc291cmNlXCJcbiAgICAgICAgICB9XG4gICAgICAgIF0sXG4gICAgICAgIFZlcnNpb246IFwiMjAxMi0xMC0xN1wiXG4gICAgICB9LFxuICAgICAgUG9saWN5TmFtZTogXCJNeVJvbGVEZWZhdWx0UG9saWN5QTM2QkUxRERcIixcbiAgICAgIFJvbGVzOiBbXG4gICAgICAgIHtcbiAgICAgICAgICBSZWY6IFwiTXlSb2xlRjQ4RkZFMDRcIlxuICAgICAgICB9XG4gICAgICBdXG4gICAgfSk7XG5cbiAgfSk7XG5cbiAgdGVzdCgnbWFuYWdlZCBwb2xpY3kgYXJucyBjYW4gYmUgc3VwcGxpZWQgdXBvbiBpbml0aWFsaXphdGlvbiBhbmQgYWxzbyBhZGRlZCBsYXRlcicsICgpID0+IHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgY29uc3Qgcm9sZSA9IG5ldyBSb2xlKHN0YWNrLCAnTXlSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgndGVzdC5zZXJ2aWNlJyksXG4gICAgICBtYW5hZ2VkUG9saWNpZXM6IFsgeyBtYW5hZ2VkUG9saWN5QXJuOiAnbWFuYWdlZDEnIH0sIHsgbWFuYWdlZFBvbGljeUFybjogJ21hbmFnZWQyJyB9IF1cbiAgICB9KTtcblxuICAgIHJvbGUuYWRkTWFuYWdlZFBvbGljeSh7IG1hbmFnZWRQb2xpY3lBcm46ICdtYW5hZ2VkMycgfSk7XG4gICAgZXhwZWN0KHN0YWNrKS50b01hdGNoVGVtcGxhdGUoeyBSZXNvdXJjZXM6XG4gICAgICB7IE15Um9sZUY0OEZGRTA0OlxuICAgICAgICAgeyBUeXBlOiAnQVdTOjpJQU06OlJvbGUnLFxuICAgICAgICAgUHJvcGVydGllczpcbiAgICAgICAgICB7IEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDpcbiAgICAgICAgICAgeyBTdGF0ZW1lbnQ6XG4gICAgICAgICAgICBbIHsgQWN0aW9uOiAnc3RzOkFzc3VtZVJvbGUnLFxuICAgICAgICAgICAgICBFZmZlY3Q6ICdBbGxvdycsXG4gICAgICAgICAgICAgIFByaW5jaXBhbDogeyBTZXJ2aWNlOiAndGVzdC5zZXJ2aWNlJyB9IH0gXSxcbiAgICAgICAgICAgICBWZXJzaW9uOiAnMjAxMi0xMC0xNycgfSxcbiAgICAgICAgICBNYW5hZ2VkUG9saWN5QXJuczogWyAnbWFuYWdlZDEnLCAnbWFuYWdlZDInLCAnbWFuYWdlZDMnIF0gfSB9IH0gfSk7XG5cbiAgfSk7XG5cbiAgdGVzdCgnZmVkZXJhdGVkIHByaW5jaXBhbCBjYW4gY2hhbmdlIEFzc3VtZVJvbGVBY3Rpb24nLCAoKSA9PiB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBjb2duaXRvUHJpbmNpcGFsID0gbmV3IEZlZGVyYXRlZFByaW5jaXBhbChcbiAgICAgICdmb28nLFxuICAgICAgeyBTdHJpbmdFcXVhbHM6IHsga2V5OiAndmFsdWUnIH0gfSxcbiAgICAgICdzdHM6QXNzdW1lU29tZXRoaW5nJyk7XG5cbiAgICBuZXcgUm9sZShzdGFjaywgJ015Um9sZScsIHsgYXNzdW1lZEJ5OiBjb2duaXRvUHJpbmNpcGFsIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCIsXG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIFByaW5jaXBhbDogeyBGZWRlcmF0ZWQ6IFwiZm9vXCIgfSxcbiAgICAgICAgICAgIENvbmRpdGlvbjoge1xuICAgICAgICAgICAgICBTdHJpbmdFcXVhbHM6IHsga2V5OiBcInZhbHVlXCIgfVxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIEFjdGlvbjogXCJzdHM6QXNzdW1lU29tZXRoaW5nXCIsXG4gICAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICB9XG4gICAgICAgIF0sXG4gICAgICB9XG4gICAgfSk7XG4gIH0pO1xuXG4gIGRlc2NyaWJlKCdtYXhTZXNzaW9uRHVyYXRpb24nLCAoKSA9PiB7XG5cbiAgICB0ZXN0KCdpcyBub3Qgc3BlY2lmaWVkIGJ5IGRlZmF1bHQnLCAoKSA9PiB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgICAgbmV3IFJvbGUoc3RhY2ssICdNeVJvbGUnLCB7IGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ3Nucy5hbWF6b25hd3MuY29tJykgfSk7XG4gICAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2hUZW1wbGF0ZSh7XG4gICAgICAgIFJlc291cmNlczoge1xuICAgICAgICAgIE15Um9sZUY0OEZGRTA0OiB7XG4gICAgICAgICAgVHlwZTogXCJBV1M6OklBTTo6Um9sZVwiLFxuICAgICAgICAgIFByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgQWN0aW9uOiBcInN0czpBc3N1bWVSb2xlXCIsXG4gICAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgICAgICBQcmluY2lwYWw6IHtcbiAgICAgICAgICAgICAgICBTZXJ2aWNlOiBcInNucy5hbWF6b25hd3MuY29tXCJcbiAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICB9XG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ2NhbiBiZSB1c2VkIHRvIHNwZWNpZnkgdGhlIG1heGltdW0gc2Vzc2lvbiBkdXJhdGlvbiBmb3IgYXNzdW1pbmcgdGhlIHJvbGUnLCAoKSA9PiB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgICBuZXcgUm9sZShzdGFjaywgJ015Um9sZScsIHsgbWF4U2Vzc2lvbkR1cmF0aW9uOiBEdXJhdGlvbi5zZWNvbmRzKDM3MDApLCBhc3N1bWVkQnk6IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdzbnMuYW1hem9uYXdzLmNvbScpIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6Um9sZScsIHtcbiAgICAgICAgTWF4U2Vzc2lvbkR1cmF0aW9uOiAzNzAwXG4gICAgICB9KTtcbiAgICB9KTtcblxuICAgIHRlc3QoJ211c3QgYmUgYmV0d2VlbiAzNjAwIGFuZCA0MzIwMCcsICgpID0+IHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICAgIGNvbnN0IGFzc3VtZWRCeSA9IG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdibGEnKTtcblxuICAgICAgbmV3IFJvbGUoc3RhY2ssICdNeVJvbGUxJywgeyBhc3N1bWVkQnksIG1heFNlc3Npb25EdXJhdGlvbjogRHVyYXRpb24uaG91cnMoMSkgfSk7XG4gICAgICBuZXcgUm9sZShzdGFjaywgJ015Um9sZTInLCB7IGFzc3VtZWRCeSwgbWF4U2Vzc2lvbkR1cmF0aW9uOiBEdXJhdGlvbi5ob3VycygxMikgfSk7XG5cbiAgICAgIGNvbnN0IGV4cGVjdGVkID0gKHZhbDogYW55KSA9PiBgbWF4U2Vzc2lvbkR1cmF0aW9uIGlzIHNldCB0byAke3ZhbH0sIGJ1dCBtdXN0IGJlID49IDM2MDBzZWMgKDFocikgYW5kIDw9IDQzMjAwc2VjICgxMmhycylgO1xuICAgICAgZXhwZWN0KCgpID0+IG5ldyBSb2xlKHN0YWNrLCAnTXlSb2xlMycsIHsgYXNzdW1lZEJ5LCBtYXhTZXNzaW9uRHVyYXRpb246IER1cmF0aW9uLm1pbnV0ZXMoMSkgfSkpXG4gICAgICAgIC50b1Rocm93KGV4cGVjdGVkKDYwKSk7XG4gICAgICBleHBlY3QoKCkgPT4gbmV3IFJvbGUoc3RhY2ssICdNeVJvbGU0JywgeyBhc3N1bWVkQnksIG1heFNlc3Npb25EdXJhdGlvbjogRHVyYXRpb24uc2Vjb25kcygzNTk5KSB9KSlcbiAgICAgICAgLnRvVGhyb3coZXhwZWN0ZWQoMzU5OSkpO1xuICAgICAgZXhwZWN0KCgpID0+IG5ldyBSb2xlKHN0YWNrLCAnTXlSb2xlNScsIHsgYXNzdW1lZEJ5LCBtYXhTZXNzaW9uRHVyYXRpb246IER1cmF0aW9uLnNlY29uZHMoNDMyMDEpIH0pKVxuICAgICAgICAudG9UaHJvdyhleHBlY3RlZCg0MzIwMSkpO1xuICAgIH0pO1xuICB9KTtcblxuICB0ZXN0KCdhbGxvdyByb2xlIHdpdGggbXVsdGlwbGUgcHJpbmNpcGFscycsICgpID0+IHtcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gICAgbmV3IFJvbGUoc3RhY2ssICdNeVJvbGUnLCB7XG4gICAgICBhc3N1bWVkQnk6IG5ldyBDb21wb3NpdGVQcmluY2lwYWwoXG4gICAgICAgIG5ldyBTZXJ2aWNlUHJpbmNpcGFsKCdib29tLmFtYXpvbmF3cy50ZXN0JyksXG4gICAgICAgIG5ldyBBcm5QcmluY2lwYWwoJzExMTExMTEnKVxuICAgICAgKVxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICBBc3N1bWVSb2xlUG9saWN5RG9jdW1lbnQ6IHtcbiAgICAgICAgU3RhdGVtZW50OiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBcInN0czpBc3N1bWVSb2xlXCIsXG4gICAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICAgIFByaW5jaXBhbDoge1xuICAgICAgICAgICAgICBTZXJ2aWNlOiBcImJvb20uYW1hem9uYXdzLnRlc3RcIixcbiAgICAgICAgICAgICAgQVdTOiBcIjExMTExMTFcIlxuICAgICAgICAgICAgfVxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICAgIH1cbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnY2FuIHN1cHBseSBwZXJtaXNzaW9ucyBib3VuZGFyeSBtYW5hZ2VkIHBvbGljeScsICgpID0+IHtcbiAgICAvLyBHSVZFTlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICBjb25zdCBwZXJtaXNzaW9uc0JvdW5kYXJ5ID0gTWFuYWdlZFBvbGljeS5mcm9tQXdzTWFuYWdlZFBvbGljeU5hbWUoJ21hbmFnZWQtcG9saWN5Jyk7XG5cbiAgICBuZXcgUm9sZShzdGFjaywgJ015Um9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ3Nucy5hbWF6b25hd3MuY29tJyksXG4gICAgICBwZXJtaXNzaW9uc0JvdW5kYXJ5LFxuICAgIH0pO1xuXG4gICAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZSgnQVdTOjpJQU06OlJvbGUnLCB7XG4gICAgICBQZXJtaXNzaW9uc0JvdW5kYXJ5OiB7XG4gICAgICAgIFwiRm46OkpvaW5cIjogW1xuICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgW1xuICAgICAgICAgICAgXCJhcm46XCIsXG4gICAgICAgICAgICB7XG4gICAgICAgICAgICAgIFJlZjogXCJBV1M6OlBhcnRpdGlvblwiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgXCI6aWFtOjphd3M6cG9saWN5L21hbmFnZWQtcG9saWN5XCJcbiAgICAgICAgICBdXG4gICAgICAgIF1cbiAgICAgIH1cbiAgICB9KTtcbiAgfSk7XG5cbiAgdGVzdCgnUHJpbmNpcGFsLSogaW4gYW4gQXNzdW1lUm9sZVBvbGljeURvY3VtZW50IGdldHMgdHJhbnNsYXRlZCB0byB7IFwiQVdTXCI6IFwiKlwiIH0nLCAoKSA9PiB7XG4gICAgLy8gVGhlIGRvY3Mgc2F5IHRoYXQgXCJQcmluY2lwYWw6ICpcIiBhbmQgXCJQcmluY2lwYWw6IHsgQVdTOiAqIH1cIiBhcmUgZXF1aXZhbGVudFxuICAgIC8vIChodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vSUFNL2xhdGVzdC9Vc2VyR3VpZGUvcmVmZXJlbmNlX3BvbGljaWVzX2VsZW1lbnRzX3ByaW5jaXBhbC5odG1sKVxuICAgIC8vIGJ1dCBpbiBwcmFjdGljZSBDcmVhdGVSb2xlIGVycm9ycyBvdXQgaWYgeW91IHVzZSBcIlByaW5jaXBhbDogKlwiIGluIGFuIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDpcbiAgICAvLyBBbiBlcnJvciBvY2N1cnJlZCAoTWFsZm9ybWVkUG9saWN5RG9jdW1lbnQpIHdoZW4gY2FsbGluZyB0aGUgQ3JlYXRlUm9sZSBvcGVyYXRpb246IEFzc3VtZVJvbGVwb2xpY3kgY29udGFpbmVkIGFuIGludmFsaWQgcHJpbmNpcGFsOiBcIlNUQVJcIjpcIipcIi5cblxuICAgIC8vIE1ha2Ugc3VyZSB0aGF0IHdlIGhhbmRsZSB0aGlzIGNhc2Ugc3BlY2lhbGx5LlxuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgbmV3IFJvbGUoc3RhY2ssICdSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgQW55UHJpbmNpcGFsKCksXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKCdBV1M6OklBTTo6Um9sZScsIHtcbiAgICAgIEFzc3VtZVJvbGVQb2xpY3lEb2N1bWVudDoge1xuICAgICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBBY3Rpb246IFwic3RzOkFzc3VtZVJvbGVcIixcbiAgICAgICAgICAgIEVmZmVjdDogXCJBbGxvd1wiLFxuICAgICAgICAgICAgUHJpbmNpcGFsOiB7IEFXUzogXCIqXCIgfSxcbiAgICAgICAgICB9XG4gICAgICAgIF0sXG4gICAgICAgIFZlcnNpb246IFwiMjAxMi0xMC0xN1wiXG4gICAgICB9XG4gICAgfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2NhbiBoYXZlIGEgZGVzY3JpcHRpb24nLCAoKSA9PiB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIG5ldyBSb2xlKHN0YWNrLCAnTXlSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnc25zLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIGRlc2NyaXB0aW9uOiBcIlRoaXMgaXMgYSByb2xlIGRlc2NyaXB0aW9uLlwiXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2hUZW1wbGF0ZSh7IFJlc291cmNlczpcbiAgICAgIHsgTXlSb2xlRjQ4RkZFMDQ6XG4gICAgICAgICB7IFR5cGU6ICdBV1M6OklBTTo6Um9sZScsXG4gICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgIHsgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OlxuICAgICAgICAgICB7IFN0YXRlbWVudDpcbiAgICAgICAgICAgIFsgeyBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgICAgUHJpbmNpcGFsOiB7IFNlcnZpY2U6ICdzbnMuYW1hem9uYXdzLmNvbScgfSB9IF0sXG4gICAgICAgICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyB9LFxuICAgICAgICAgICAgRGVzY3JpcHRpb246ICdUaGlzIGlzIGEgcm9sZSBkZXNjcmlwdGlvbi4nIH0gfSB9IH0pO1xuICB9KTtcblxuICB0ZXN0KCdzaG91bGQgbm90IGhhdmUgYW4gZW1wdHkgZGVzY3JpcHRpb24nLCAoKSA9PiB7XG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICAgIG5ldyBSb2xlKHN0YWNrLCAnTXlSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnc25zLmFtYXpvbmF3cy5jb20nKSxcbiAgICAgIGRlc2NyaXB0aW9uOiBcIlwiXG4gICAgfSk7XG5cbiAgICBleHBlY3Qoc3RhY2spLnRvTWF0Y2hUZW1wbGF0ZSh7IFJlc291cmNlczpcbiAgICAgIHsgTXlSb2xlRjQ4RkZFMDQ6XG4gICAgICAgICB7IFR5cGU6ICdBV1M6OklBTTo6Um9sZScsXG4gICAgICAgICBQcm9wZXJ0aWVzOlxuICAgICAgICAgIHsgQXNzdW1lUm9sZVBvbGljeURvY3VtZW50OlxuICAgICAgICAgICB7IFN0YXRlbWVudDpcbiAgICAgICAgICAgIFsgeyBBY3Rpb246ICdzdHM6QXNzdW1lUm9sZScsXG4gICAgICAgICAgICAgIEVmZmVjdDogJ0FsbG93JyxcbiAgICAgICAgICAgICAgUHJpbmNpcGFsOiB7IFNlcnZpY2U6ICdzbnMuYW1hem9uYXdzLmNvbScgfSB9IF0sXG4gICAgICAgICAgICAgIFZlcnNpb246ICcyMDEyLTEwLTE3JyB9fSB9IH0gfSk7XG4gIH0pO1xuXG4gIHRlc3QoJ2Rlc2NyaXB0aW9uIGNhbiBvbmx5IGJlIDEwMDAgY2hhcmFjdGVycyBsb25nJywgKCkgPT4ge1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgICBleHBlY3QoKCkgPT4ge1xuICAgICAgbmV3IFJvbGUoc3RhY2ssICdNeVJvbGUnLCB7XG4gICAgICAgIGFzc3VtZWRCeTogbmV3IFNlcnZpY2VQcmluY2lwYWwoJ3Nucy5hbWF6b25hd3MuY29tJyksXG4gICAgICAgIGRlc2NyaXB0aW9uOiBcIjEwMDArIGNoYXJhY3RlciBsb25nIGRlc2NyaXB0aW9uOiBMb3JlbSBpcHN1bSBkb2xvciBzaXQgYW1ldCwgY29uc2VjdGV0dWVyIGFkaXBpc2NpbmcgZWxpdC4gXFxcbiAgICAgICAgQWVuZWFuIGNvbW1vZG8gbGlndWxhIGVnZXQgZG9sb3IuIEFlbmVhbiBtYXNzYS4gQ3VtIHNvY2lpcyBuYXRvcXVlIHBlbmF0aWJ1cyBldCBtYWduaXMgZGlzIHBhcnR1cmllbnQgbW9udGVzLCBcXFxuICAgICAgICBuYXNjZXR1ciByaWRpY3VsdXMgbXVzLiBEb25lYyBxdWFtIGZlbGlzLCB1bHRyaWNpZXMgbmVjLCBwZWxsZW50ZXNxdWUgZXUsIHByZXRpdW0gcXVpcywgc2VtLiBOdWxsYSBjb25zZXF1YXQgXFxcbiAgICAgICAgbWFzc2EgcXVpcyBlbmltLiBEb25lYyBwZWRlIGp1c3RvLCBmcmluZ2lsbGEgdmVsLCBhbGlxdWV0IG5lYywgdnVscHV0YXRlIGVnZXQsIGFyY3UuIEluIGVuaW0ganVzdG8sIHJob25jdXMgdXQsIFxcXG4gICAgICAgIGltcGVyZGlldCBhLCB2ZW5lbmF0aXMgdml0YWUsIGp1c3RvLiBOdWxsYW0gZGljdHVtIGZlbGlzIGV1IHBlZGUgbW9sbGlzIHByZXRpdW0uIEludGVnZXIgdGluY2lkdW50LiBDcmFzIGRhcGlidXMuIFxcXG4gICAgICAgIFZpdmFtdXMgZWxlbWVudHVtIHNlbXBlciBuaXNpLiBBZW5lYW4gdnVscHV0YXRlIGVsZWlmZW5kIHRlbGx1cy4gQWVuZWFuIGxlbyBsaWd1bGEsIHBvcnR0aXRvciBldSwgY29uc2VxdWF0IHZpdGFlLCBcXFxuICAgICAgICBlbGVpZmVuZCBhYywgZW5pbS4gQWxpcXVhbSBsb3JlbSBhbnRlLCBkYXBpYnVzIGluLCB2aXZlcnJhIHF1aXMsIGZldWdpYXQgYSwgdGVsbHVzLiBQaGFzZWxsdXMgdml2ZXJyYSBudWxsYSB1dCBtZXR1cyBcXFxuICAgICAgICB2YXJpdXMgbGFvcmVldC4gUXVpc3F1ZSBydXRydW0uIEFlbmVhbiBpbXBlcmRpZXQuIEV0aWFtIHVsdHJpY2llcyBuaXNpIHZlbCBhdWd1ZS4gQ3VyYWJpdHVyIHVsbGFtY29ycGVyIHVsdHJpY2llcyBuaXNpLiBcXFxuICAgICAgICBOYW0gZWdldCBkdWkuIEV0aWFtIHJob25jdXMuIE1hZWNlbmFzIHRlbXB1cywgdGVsbHVzIGVnZXQgY29uZGltZW50dW0gcmhvbmN1cywgc2VtIHF1YW0gc2VtcGVyIGxpYmVybywgc2l0IGFtZXQgYWRpcGlzY2luZyBcXFxuICAgICAgICBzZW0gbmVxdWUgc2VkIGlwc3VtLlwiXG4gICAgICB9KTtcbiAgICB9KS50b1Rocm93KC9Sb2xlIGRlc2NyaXB0aW9uIG11c3QgYmUgbm8gbG9uZ2VyIHRoYW4gMTAwMCBjaGFyYWN0ZXJzLi8pO1xuICB9KTtcbn0pO1xuIl19