import itertools

import pandas as pd

from ._factor_metadata import AmbiguousAttributeError, Factor, get_factor_storage_and_materializer, resolve_ambiguous


class FormulaicContrasts:
    """Build contrasts for a model design specified as formulaic formula.

    Keeps track of factors and their metadata and provides the {func}`~FormulaicContrasts.cond` method
    for building contrasts.

    Parameters
    ----------
    data
        Metadata to be passed to formulaic
    design
        Formulaic formula of the model definition
    """

    def __init__(self, data: pd.DataFrame, design: str):
        self.factor_storage, self.variable_to_factors, materializer_class = get_factor_storage_and_materializer()
        self.data = data
        self.design_matrix = materializer_class(data, record_factor_metadata=True).get_model_matrix(design)

    @property
    def variables(self):
        """Get the names of the variables used in the model definition."""
        return self.design_matrix.model_spec.variables_by_source["data"]

    def cond(self, **kwargs):
        """
        Get a contrast vector representing a specific condition.

        The `kwargs` are key/value pairs where the key refers to a variable used in the
        design and the value represents a category of that variable. Variables not specified
        will be filled with their default/baseline value.

        The vectors generated by `.cond` can be combined using standard arithmetic operations
        to obtain the desired contrast, e.g.

        >>> contrast = model.cond(treatment="drugA") - model.cond(treatment="placebo")

        For more information on how to build contrasts, see :doc:`/contrasts`.

        Parameters
        ----------
        **kwargs
            column/value pairs.

        Returns
        -------
        A vector with one element per column in the design matrix,
        where the kwargs arguments are coded as in the design matrix.
        """
        cond_dict = kwargs
        if not set(cond_dict.keys()).issubset(self.variables):
            raise ValueError(
                "You specified a variable that is not part of the model. Available variables: "
                + ",".join(self.variables)
            )
        for var in self.variables:
            if var in cond_dict:
                self._check_category(var, cond_dict[var])
            else:
                cond_dict[var] = self._get_default_value(var)
        df = pd.DataFrame([kwargs])
        return self.design_matrix.model_spec.get_model_matrix(df).iloc[0]

    def contrast(self, column, baseline, group_to_compare):
        """
        Build a simple contrast for pairwise comparisons of a single variable.

        For more complex contrasts, please use construct a contrast vector using
        :func:`~formulaic_contrasts.FormulaicContrasts.cond`.

        Parameters
        ----------
        column
            column in adata.obs to test on.
        baseline
            baseline category (denominator).
        group_to_compare
            category to compare against baseline (nominator).

        Returns
        -------
            Numeric contrast vector.
        """
        return self.cond(**{column: group_to_compare}) - self.cond(**{column: baseline})

    def _get_default_value(self, var):
        factor_metadata = self._get_factor_metadata_for_variable(var)
        if resolve_ambiguous(factor_metadata, "kind") == Factor.Kind.CATEGORICAL:
            try:
                tmp_base = resolve_ambiguous(factor_metadata, "base")
            except AmbiguousAttributeError as e:
                raise ValueError(
                    f"Could not automatically resolve base category for variable {var}. Please specify it explicity in `model.cond`."
                ) from e
            return tmp_base if tmp_base is not None else "\0"
        else:
            return 0

    def _get_factor_metadata_for_variable(self, var):
        factors = self.variable_to_factors[var]
        return list(itertools.chain.from_iterable(self.factor_storage[f] for f in factors))

    def _check_category(self, var, value):
        factor_metadata = self._get_factor_metadata_for_variable(var)
        tmp_categories = resolve_ambiguous(factor_metadata, "categories")
        if resolve_ambiguous(factor_metadata, "kind") == Factor.Kind.CATEGORICAL and value not in tmp_categories:
            raise ValueError(
                f"You specified a non-existant category for {var}. Possible categories: {', '.join(tmp_categories)}"
            )
