
"""
SoIdea-update-python 进程锁与文件占用检测工具
支持跨平台 lock 文件、PID 检查、文件占用检测、强制关闭标记
"""

import logging
import os
import platform

import psutil

logger = logging.getLogger("SoIdea-update-python.lock")

def get_lock_pid(lock_file):
    """
    读取 lock 文件中的 PID。
    :param lock_file: lock 文件路径
    :return: PID 或 None
    """
    if not os.path.exists(lock_file):
        return None
    try:
        with open(lock_file, 'r') as f:
            pid = f.read().strip()
            return int(pid)
    except Exception:
        return None

def is_pid_alive(pid):
    """
    检查指定 PID 是否存活。
    :param pid: 进程号
    :return: bool
    """
    try:
        return psutil.pid_exists(int(pid))
    except Exception:
        return False

def create_lock(lock_file, pid=None):
    """
    创建 lock 文件，写入当前或指定 PID。
    :param lock_file: lock 文件路径
    :param pid: 可选，指定 PID
    """
    if pid is None:
        pid = os.getpid()
    lock_dir = os.path.dirname(lock_file)
    if lock_dir and not os.path.exists(lock_dir):
        os.makedirs(lock_dir, exist_ok=True)
    with open(lock_file, 'w') as f:
        f.write(str(pid))
    logger.info(f"创建 lock 文件: {lock_file} PID={pid}")

def remove_lock(lock_file):
    """
    删除 lock 文件。
    :param lock_file: lock 文件路径
    """
    if os.path.exists(lock_file):
        os.remove(lock_file)
        logger.info(f"删除 lock 文件: {lock_file}")


# 增强：跨平台文件占用检测和强制退出判断
def is_file_locked(filepath):
    """
    检测文件是否被占用，兼容 Windows/Linux/Mac。
    :param filepath: 文件路径
    :return: bool
    """
    sys_platform = platform.system().lower()
    if sys_platform == "windows":
        try:
            import msvcrt
            with open(filepath, 'a') as f:
                msvcrt.locking(f.fileno(), msvcrt.LK_NBLCK, 1)
                msvcrt.locking(f.fileno(), msvcrt.LK_UNLCK, 1)
            return False
        except Exception:
            return True
    else:
        try:
            import fcntl
            with open(filepath, 'a') as f:
                fcntl.flock(f, fcntl.LOCK_EX | fcntl.LOCK_NB)
                fcntl.flock(f, fcntl.LOCK_UN)
            return False
        except Exception:
            return True

def was_file_forced_closed(filepath, marker_path=None):
    """
    判断文件是否被强制关闭/异常退出（通过标记文件或日志）。
    :param filepath: 文件路径
    :param marker_path: 可选，标记文件路径
    :return: bool
    """
    marker_path = marker_path or (filepath + '.lock_marker')
    return os.path.exists(marker_path)

def set_forced_closed_marker(filepath, marker_path=None):
    """
    设置强制关闭标记。
    :param filepath: 文件路径
    :param marker_path: 可选，标记文件路径
    """
    marker_path = marker_path or (filepath + '.lock_marker')
    marker_dir = os.path.dirname(marker_path)
    if marker_dir and not os.path.exists(marker_dir):
        os.makedirs(marker_dir, exist_ok=True)
    with open(marker_path, 'w') as f:
        f.write('forced_closed')
