
"""
SoIdea-update-python 日志工具
支持日志文件、控制台、自动创建目录
"""

import logging
import sys
from logging.handlers import RotatingFileHandler


def setup_logger(
    logfile: str = "SoIdea-update-python.log",
    logdir: str = None,
    level: int = logging.INFO,
    callback: callable = None
) -> logging.Logger:
    """
    初始化日志，支持文件、控制台、可选回调，自动创建目录，强制utf-8编码。

    Args:
        logfile (str): 日志文件名或路径。
        logdir (str, optional): 日志目录（可选）。
        level (int): 日志级别。
        callback (callable, optional): 可选日志回调（每条日志都会调用 callback(str)）。
    Returns:
        logging.Logger: logger 对象。
    """
    import os
    if logdir:
        os.makedirs(logdir, exist_ok=True)
        logfile = os.path.join(logdir, logfile)
    else:
        logdir = os.path.dirname(logfile)
        if logdir and not os.path.exists(logdir):
            os.makedirs(logdir, exist_ok=True)
    logger = logging.getLogger("SoIdea-update-python")
    if not logger.handlers:
        file_handler = RotatingFileHandler(logfile, maxBytes=2*1024*1024, backupCount=5, encoding="utf-8")
        formatter = logging.Formatter('[%(asctime)s] %(levelname)s: %(message)s', datefmt='%Y-%m-%d %H:%M:%S')
        file_handler.setFormatter(formatter)
        logger.addHandler(file_handler)

        stream_handler = logging.StreamHandler()
        try:
            stream_handler.setStream(sys.stdout)
        except Exception:
            pass
        stream_handler.setFormatter(formatter)
        if hasattr(stream_handler.stream, 'reconfigure'):
            try:
                stream_handler.stream.reconfigure(encoding='utf-8')
            except Exception:
                pass
        logger.addHandler(stream_handler)
        logger.setLevel(level)
        # 可选日志回调
        if callback:
            class CallbackHandler(logging.Handler):
                def emit(self, record):
                    callback(self.format(record))
            cb_handler = CallbackHandler()
            cb_handler.setFormatter(formatter)
            logger.addHandler(cb_handler)
    return logger


class ProgressBar:
    """
    简易进度条（适用于非 tqdm 场景）。

    Args:
        total (int): 总数。
        desc (str): 描述。
        width (int): 进度条宽度。
    """
    def __init__(self, total: int, desc: str = "", width: int = 40):
        self.total = total
        self.desc = desc
        self.width = width
        self.current = 0

    def update(self, n: int) -> None:
        """进度条更新 n 步。"""
        self.current += n
        self.show()

    def show(self) -> None:
        """显示当前进度。"""
        percent = self.current / self.total if self.total else 0
        filled = int(self.width * percent)
        bar = "█" * filled + "-" * (self.width - filled)
        print(f"\r{self.desc} |{bar}| {percent:.0%}", end="")
        if self.current >= self.total:
            print()
