
# -*- coding: utf-8 -*-
import glob
import hashlib
import os
import shutil
import subprocess
import time
import zipfile

import pyminizip

from .signature import PUBLIC_KEY_FILE, verify_file


def create_exe_from_py(py_file: str, output_dir: str, onefile: bool = True, console: bool = True, build_dir: str = None, spec_dir: str = None) -> str:
    # 先定义 cmd，再追加 hidden-import
    cmd = [
        "pyinstaller",
        "--distpath", output_dir,
        "--workpath", build_dir,
        "--specpath", spec_dir,
        "--clean",
        "--paths", soidea_pkg_path
    ]
    hidden_modules = [
        'SoIdea_update_python',
        'SoIdea_update_python.backup',
        'SoIdea_update_python.config',
        'SoIdea_update_python.constants',
        'SoIdea_update_python.encoding_utils',
        'SoIdea_update_python.events',
        'SoIdea_update_python.git_utils',
        'SoIdea_update_python.lock_utils',
        'SoIdea_update_python.log_utils',
        'SoIdea_update_python.package_sources',
        'SoIdea_update_python.package',
        'SoIdea_update_python.param_utils',
        'SoIdea_update_python.path_utils',
        'SoIdea_update_python.platform_utils',
        'SoIdea_update_python.process',
        'SoIdea_update_python.process_utils',
        'SoIdea_update_python.self_update_utils',
        'SoIdea_update_python.signature',
        'SoIdea_update_python.update_manager',
        'SoIdea_update_python.upgrade_utils',
        'SoIdea_update_python.version_utils',
    ]
    for mod in hidden_modules:
        cmd.extend(['--hidden-import', mod])
    """
    用 PyInstaller 打包指定 py 文件为 EXE，产物输出到 output_dir。
    Args:
        py_file (str): 入口 py 文件路径。
        output_dir (str): EXE 输出目录。
        onefile (bool): 是否生成单文件 EXE。
        console (bool): 是否为控制台程序（False=GUI）。
    Returns:
        str: EXE 文件路径。
    """
    if not os.path.isfile(py_file):
        raise RuntimeError(f"入口文件不存在: {py_file}")
    # 自动清理旧 build/spec 目录
    if build_dir is None:
        build_dir = os.path.join(output_dir, "build")
    if spec_dir is None:
        spec_dir = os.path.join(output_dir, "spec")
    for d in [build_dir, spec_dir]:
        if os.path.exists(d):
            shutil.rmtree(d)
    os.makedirs(output_dir, exist_ok=True)
    # 自动添加 --paths 参数，指向项目根目录下 SoIdea_update_python
    project_root = os.path.abspath(os.path.join(os.path.dirname(__file__), '..', '..'))
    soidea_pkg_path = os.path.join(project_root, 'SoIdea_update_python')
    cmd = [
        "pyinstaller",
        "--distpath", output_dir,
        "--workpath", build_dir,
        "--specpath", spec_dir,
        "--clean",
        "--paths", soidea_pkg_path
    ]
    if onefile:
        cmd.append("--onefile")
    if not console:
        cmd.append("--noconsole")
    cmd.append(py_file)
    result = subprocess.run(cmd, capture_output=True, text=True)
    if result.returncode != 0:
        raise RuntimeError(f"PyInstaller 打包失败: {result.stderr}")
    exe_name = os.path.splitext(os.path.basename(py_file))[0] + ".exe"
    exe_path = os.path.join(output_dir, exe_name)
    if not os.path.isfile(exe_path):
        # 兼容 PyInstaller 可能输出到 dist 子目录
        dist_dir = os.path.join(output_dir, os.path.splitext(os.path.basename(py_file))[0])
        alt_exe = os.path.join(dist_dir, exe_name)
        if os.path.isfile(alt_exe):
            shutil.move(alt_exe, exe_path)
    return exe_path




# ----------------- 升级包生成主函数 -----------------
def create_update_package(config: dict, encrypt: bool = False) -> str:
    """
    根据配置收集主程序及相关文件，生成升级包（zip），可选加密。

    Args:
        config (dict): build_config.toml 加载结果。
        encrypt (bool, optional): 是否加密压缩（需 pyminizip）。
    Returns:
        str: 升级包 zip 路径。
    """
    base_dir = config.get('base_dir', os.getcwd())
    files = config.get('files', [])
    if not files:
        # 默认收集主程序和所有 dll/ini/txt
        files = [config.get('target_exe', 'main.exe'), '*.dll', '*.ini', '*.txt']
    file_list = []
    for pattern in files:
        abs_pattern = os.path.join(base_dir, pattern)
        file_list.extend(glob.glob(abs_pattern))
    if not file_list:
        raise RuntimeError('未找到任何待打包文件')
    out_dir = config.get('out_dir', base_dir)
    if not os.path.exists(out_dir):
        os.makedirs(out_dir, exist_ok=True)
    zip_name = config.get('update_zip', f'update_{time.strftime("%Y%m%d_%H%M%S")}.zip')
    zip_path = os.path.join(out_dir, zip_name)
    if encrypt:
        password = config.get('zip_password', 'SoIdea-update-python')
        pyminizip.compress_multiple(file_list, [], zip_path, password, 5)
    else:
        with zipfile.ZipFile(zip_path, 'w', zipfile.ZIP_DEFLATED) as zf:
            for f in file_list:
                arcname = os.path.relpath(f, base_dir)
                zf.write(f, arcname)
    return zip_path





def validate_upgrade_package(
    zip_path: str,
    required_files: list = None,
    hash_value: str = None,
    sig_path: str = None,
    public_key_path: str = None
) -> tuple[bool, str]:
    """
    校验升级包：
    1. zip 文件内容完整性（必须文件/目录）
    2. hash 校验（如提供）
    3. 签名校验（如提供）

    Args:
        zip_path (str): 升级包路径。
        required_files (list, optional): 必须包含的文件。
        hash_value (str, optional): 期望 hash。
        sig_path (str, optional): 签名文件路径。
        public_key_path (str, optional): 公钥路径。
    Returns:
        tuple[bool, str]: (校验结果, 原因/ok)。
    """
    if not os.path.isfile(zip_path):
        return False, f"升级包不存在: {zip_path}"
    # 1. hash 校验
    if hash_value:
        h = hashlib.sha256()
        with open(zip_path, 'rb') as f:
            while True:
                chunk = f.read(8192)
                if not chunk:
                    break
                h.update(chunk)
        actual_hash = h.hexdigest()
        if actual_hash.lower() != hash_value.lower():
            return False, f"升级包 hash 校验失败: {actual_hash} != {hash_value}"
    # 2. zip 内容校验
    try:
        with zipfile.ZipFile(zip_path, 'r') as zf:
            namelist = zf.namelist()
            if required_files:
                for req in required_files:
                    if req not in namelist:
                        return False, f"升级包缺少必须文件: {req}"
    except Exception as e:
        return False, f"升级包解压校验失败: {e}"
    # 3. 签名校验（如有）
    if sig_path and public_key_path:
        try:
            from .signature import verify_file
            if not verify_file(zip_path, sig_path, public_key_path):
                return False, "升级包签名校验失败"
        except Exception as e:
            return False, f"签名校验异常: {e}"
    return True, "ok"

# 依赖说明：需安装 core.signature，shutil 为标准库


def verify_and_extract(
    zip_path: str,
    extract_to: str,
    expected_hash: str = None,
    sig_check: bool = False,
    logger=None
) -> bool:
    """
    校验并解压升级包：
    1. 校验 hash（如有）
    2. 校验签名（如 sig_check=True）
    3. 解压到指定目录

    Args:
        zip_path (str): 升级包路径。
        extract_to (str): 解压目标目录。
        expected_hash (str, optional): 期望 hash。
        sig_check (bool, optional): 是否校验签名。
        logger (logging.Logger, optional): 日志记录器。
    Returns:
        bool: 是否校验并解压成功。
    """
    if not os.path.isfile(zip_path):
        if logger:
            logger.error(f"升级包不存在: {zip_path}")
        return False
    if not os.path.exists(extract_to):
        os.makedirs(extract_to, exist_ok=True)
    if expected_hash:
        sha256 = hashlib.sha256()
        with open(zip_path, 'rb') as f:
            for chunk in iter(lambda: f.read(8192), b''):
                sha256.update(chunk)
        file_hash = sha256.hexdigest()
        if file_hash.lower() != expected_hash.lower():
            if logger:
                logger.error(f"升级包 hash 校验失败: 期望 {expected_hash}, 实际 {file_hash}")
            return False
    if sig_check:
        sig_path = zip_path + '.sig'
        if not os.path.isfile(sig_path):
            if logger:
                logger.error(f"未找到升级包签名文件: {sig_path}")
            return False
        if not verify_file(zip_path, sig_path, public_key_path=PUBLIC_KEY_FILE):
            if logger:
                logger.error("升级包签名校验失败，拒绝升级！")
            return False
        if logger:
            logger.info("升级包签名校验通过")
    try:
        with zipfile.ZipFile(zip_path, 'r') as zip_ref:
            zip_ref.extractall(extract_to)
        if logger:
            logger.info(f"升级包已解压到: {extract_to}")
        return True
    except Exception as e:
        if logger:
            logger.error(f"升级包解压失败: {e}")
        return False

def overwrite_files(src_dir: str, dst_dir: str, logger=None) -> None:
    """
    覆盖主程序文件：将 src_dir 下所有文件/目录覆盖到 dst_dir。

    Args:
        src_dir (str): 源目录。
        dst_dir (str): 目标目录。
        logger (logging.Logger, optional): 日志记录器。
    """
    for item in os.listdir(src_dir):
        s = os.path.join(src_dir, item)
        d = os.path.join(dst_dir, item)
        if os.path.isdir(s):
            if os.path.exists(d):
                shutil.rmtree(d)
            shutil.copytree(s, d)
        else:
            parent = os.path.dirname(d)
            if parent and not os.path.exists(parent):
                os.makedirs(parent, exist_ok=True)
            shutil.copy2(s, d)
    if logger:
        logger.info(f"覆盖主程序文件: {src_dir} -> {dst_dir}")

