
"""
SoIdea-update-python 平台工具
支持平台检测、权限判断、路径标准化、可执行权限设置
"""

import logging
import os
import platform
import stat

logger = logging.getLogger("SoIdea-update-python.platform")

def get_platform() -> str:
    """
    返回当前平台标识。

    Returns:
        str: 平台字符串（windows/linux/darwin）。
    """
    return platform.system().lower()

def is_windows() -> bool:
    """
    判断是否为 Windows 平台。

    Returns:
        bool: 是否为 Windows。
    """
    return get_platform() == "windows"

def is_linux() -> bool:
    """
    判断是否为 Linux 平台。

    Returns:
        bool: 是否为 Linux。
    """
    return get_platform() == "linux"

def is_mac() -> bool:
    """
    判断是否为 Mac 平台。

    Returns:
        bool: 是否为 Mac。
    """
    return get_platform() == "darwin"

def set_executable_permission(filepath: str) -> None:
    """
    设置文件为可执行（跨平台）。

    Args:
        filepath (str): 文件路径。
    """
    sys_platform = platform.system().lower()
    if sys_platform == "windows":
        # Windows 下无需设置
        return
    else:
        if os.path.isfile(filepath):
            st = os.stat(filepath)
            os.chmod(filepath, st.st_mode | stat.S_IEXEC)
            logger.info(f"设置可执行权限: {filepath}")

def normalize_path(path: str) -> str:
    """
    路径标准化，兼容多平台。

    Args:
        path (str): 路径字符串。
    Returns:
        str: 标准化路径。
    """
    return os.path.normpath(path)

def is_admin() -> bool:
    """
    判断当前进程是否有管理员/超级用户权限。

    Returns:
        bool: 是否有管理员权限。
    """
    sys_platform = platform.system().lower()
    if sys_platform == "windows":
        import ctypes
        try:
            return ctypes.windll.shell32.IsUserAnAdmin() != 0
        except Exception:
            return False
    else:
        return os.geteuid() == 0


def is_new_version(current_version, latest_tag, logger=None):
    """
    判断是否有新版本。
    :param current_version: 当前版本号
    :param latest_tag: 最新版本号
    :param logger: 可选日志对象
    :return: bool
    """
    from packaging.version import Version
    try:
        cur = Version(str(current_version).lstrip('v'))
        latest = Version(str(latest_tag).lstrip('v'))
        result = latest > cur
    except Exception:
        if logger:
            logger.warning(f'版本号解析失败，回退为字符串比较: current={current_version}, latest={latest_tag}')
        result = str(latest_tag) > str(current_version)
    if logger:
        logger.info(f"当前版本: {current_version}，最新版本: {latest_tag}，需要升级: {result}")
    return result
