"""
Tests for identifier generation with PEPPOL precedence logic.

Tests the generate_all_with_precedence function which generates all
applicable party identifiers from VAT, registration, and PEPPOL IDs.
"""

import unittest
import warnings

from ubl.identifiers.generator import generate_all_with_precedence
from ubl.models.aggregate_components import PartyIdentification


class TestIdentifierGeneration(unittest.TestCase):
    """Test identifier generation with precedence logic."""

    def test_p4x_belgium_vat_only(self):
        """
        P4X SA (Belgium) - VAT only → 2 identifiers.

        Expected:
        1. BE:VAT:BE0867709540 (scheme_id format)
        2. 9925:be0867709540 (iso6523 format, lowercase)
        """
        identifiers = generate_all_with_precedence(
            country_code="BE", vat="BE0867709540", registration=None, peppol_participant_ids=[],
        )

        self.assertEqual(len(identifiers), 2)

        # Check scheme_id format
        scheme_id_format = [i for i in identifiers if i.schemeID == "BE:VAT"]
        self.assertEqual(len(scheme_id_format), 1)
        self.assertEqual(scheme_id_format[0].value, "BE0867709540")
        self.assertEqual(scheme_id_format[0].iso6523, "9925")

        # Check iso6523 format (lowercase)
        iso6523_format = [i for i in identifiers if i.schemeID == "9925"]
        self.assertEqual(len(iso6523_format), 1)
        self.assertEqual(iso6523_format[0].value, "be0867709540")

    def test_squads_netherlands_vat_only(self):
        """
        Squads B.V. (Netherlands) - VAT only → 2 identifiers.

        Netherlands registration schemes (KVK, OINO) have specific length requirements
        that don't match typical business numbers, so we only use VAT.

        Expected:
        1. NL:VAT:NL855934682B01 (scheme_id format)
        2. 9944:nl855934682b01 (iso6523 format)
        """
        identifiers = generate_all_with_precedence(
            country_code="NL",
            vat="NL855934682B01",
            registration=None,  # NL registration schemes have strict length requirements
            peppol_participant_ids=[],
        )

        # Only VAT generates identifiers
        self.assertEqual(len(identifiers), 2)

        # VAT scheme_id format
        vat_scheme_id = [i for i in identifiers if i.schemeID == "NL:VAT"]
        self.assertEqual(len(vat_scheme_id), 1)
        self.assertEqual(vat_scheme_id[0].value, "NL855934682B01")

        # VAT iso6523 format
        vat_iso = [i for i in identifiers if i.schemeID == "9944"]
        self.assertEqual(len(vat_iso), 1)
        self.assertEqual(vat_iso[0].value, "nl855934682b01")

    def test_levit_with_peppol_precedence(self):
        """
        LevIT SC (Belgium) - VAT + registration + peppol_ids → 4 identifiers.

        PEPPOL IDs take precedence for iso6523 format.

        Expected:
        1. BE:VAT:BE0597601756
        2. 9925:be0597601756 (from peppol, not generated)
        3. BE:EN:0597601756
        4. 0208:0597601756 (from peppol, not generated)
        """
        identifiers = generate_all_with_precedence(
            country_code="BE",
            vat="BE0597601756",
            registration="0597601756",
            peppol_participant_ids=["9925:be0597601756", "0208:0597601756"],
        )

        self.assertEqual(len(identifiers), 4)

        # VAT scheme_id format
        vat_scheme = [i for i in identifiers if i.schemeID == "BE:VAT"]
        self.assertEqual(len(vat_scheme), 1)
        self.assertEqual(vat_scheme[0].value, "BE0597601756")

        # VAT iso6523 format (from peppol)
        vat_iso = [i for i in identifiers if i.schemeID == "9925"]
        self.assertEqual(len(vat_iso), 1)
        self.assertEqual(vat_iso[0].value, "be0597601756")

        # Registration scheme_id format
        reg_scheme = [i for i in identifiers if i.schemeID == "BE:EN"]
        self.assertEqual(len(reg_scheme), 1)
        self.assertEqual(reg_scheme[0].value, "0597601756")

        # Registration iso6523 format (from peppol)
        reg_iso = [i for i in identifiers if i.schemeID == "0208"]
        self.assertEqual(len(reg_iso), 1)
        self.assertEqual(reg_iso[0].value, "0597601756")

    def test_peppol_precedence_mismatch_warning(self):
        """PEPPOL ID differs from generated → warning."""
        with warnings.catch_warnings(record=True) as w:
            warnings.simplefilter("always")

            identifiers = generate_all_with_precedence(
                country_code="BE",
                vat="BE0867709540",
                registration=None,
                peppol_participant_ids=["9925:be9999999999"],  # Different value
            )

            # Should warn about mismatch
            self.assertGreater(len(w), 0)
            warning_msg = str(w[0].message)
            self.assertIn("PEPPOL ID", warning_msg)
            self.assertIn("differs", warning_msg)

            # Should use PEPPOL value regardless
            iso_format = [i for i in identifiers if i.schemeID == "9925"]
            self.assertEqual(len(iso_format), 1)
            self.assertEqual(iso_format[0].value, "be9999999999")

    def test_peppol_precedence_match_no_warning(self):
        """PEPPOL ID matches generated → no warning."""
        with warnings.catch_warnings(record=True) as w:
            warnings.simplefilter("always")

            identifiers = generate_all_with_precedence(
                country_code="BE",
                vat="BE0867709540",
                registration=None,
                peppol_participant_ids=["9925:be0867709540"],  # Matches
            )

            # Should not warn
            self.assertEqual(len(w), 0)

    def test_deduplication(self):
        """Duplicate identifiers are deduplicated."""
        identifiers = generate_all_with_precedence(
            country_code="BE",
            vat="BE0867709540",
            registration=None,
            peppol_participant_ids=["9925:be0867709540"],  # Duplicate iso6523 format
        )

        # Should not have duplicates
        self.assertEqual(len(identifiers), 2)  # scheme_id + iso6523

        # Check using set
        id_tuples = [(i.schemeID, i.value.lower()) for i in identifiers]
        self.assertEqual(len(id_tuples), len(set(id_tuples)))

    def test_no_vat_no_registration(self):
        """No VAT or registration → empty list."""
        identifiers = generate_all_with_precedence(
            country_code="BE", vat=None, registration=None, peppol_participant_ids=[],
        )

        self.assertEqual(len(identifiers), 0)

    def test_only_peppol_ids(self):
        """Only PEPPOL IDs provided → returns them."""
        identifiers = generate_all_with_precedence(
            country_code="BE",
            vat=None,
            registration=None,
            peppol_participant_ids=["9925:be0867709540", "0208:0597601756"],
        )

        self.assertEqual(len(identifiers), 2)

        ids_dict = {i.schemeID: i.value for i in identifiers}
        self.assertEqual(ids_dict["9925"], "be0867709540")
        self.assertEqual(ids_dict["0208"], "0597601756")

    def test_invalid_peppol_format_ignored(self):
        """Invalid PEPPOL ID format (no colon) is ignored."""
        identifiers = generate_all_with_precedence(
            country_code="BE", vat="BE0867709540", registration=None, peppol_participant_ids=["invalid_format"],
        )

        # Should generate from VAT normally
        self.assertEqual(len(identifiers), 2)

    def test_multiple_registration_schemes(self):
        """Country with multiple registration schemes."""
        # Belgium has multiple registration schemes: BE:EN, BE:CBE, BE:KBON
        identifiers = generate_all_with_precedence(
            country_code="BE", vat=None, registration="0597601756", peppol_participant_ids=[],
        )

        # Should generate at least one registration identifier
        self.assertGreater(len(identifiers), 0)

        # Should have scheme_id format
        scheme_ids = [i.schemeID for i in identifiers]
        self.assertTrue(any("BE:" in s for s in scheme_ids))

    def test_identifier_equality(self):
        """PartyIdentification equality works correctly."""
        id1 = PartyIdentification(value="BE0867709540", schemeID="BE:VAT", iso6523="9925")
        id2 = PartyIdentification(value="be0867709540", schemeID="BE:VAT", iso6523="9925")
        id3 = PartyIdentification(value="be0867709540", schemeID="9925", iso6523="9925")

        # Same scheme, different case → equal
        self.assertEqual(id1, id2)

        # Different scheme_id, same iso6523 → equal
        self.assertEqual(id1, id3)

    def test_identifier_hash(self):
        """PartyIdentification hash works for deduplication."""
        id1 = PartyIdentification(value="BE0867709540", schemeID="BE:VAT", iso6523="9925")
        id2 = PartyIdentification(value="be0867709540", schemeID="BE:VAT", iso6523="9925")

        # Same hash despite case difference
        self.assertEqual(hash(id1), hash(id2))

        # Can use in set
        id_set = {id1, id2}
        self.assertEqual(len(id_set), 1)

    def test_germany_vat(self):
        """German VAT generation."""
        identifiers = generate_all_with_precedence(
            country_code="DE", vat="DE123456789", registration=None, peppol_participant_ids=[],
        )

        self.assertEqual(len(identifiers), 2)

        # Check DE:VAT scheme
        vat_scheme = [i for i in identifiers if i.schemeID == "DE:VAT"]
        self.assertEqual(len(vat_scheme), 1)

    def test_france_vat_and_sirene(self):
        """French VAT + SIRENE."""
        identifiers = generate_all_with_precedence(
            country_code="FR", vat="FR12345678901", registration="784301772", peppol_participant_ids=[],
        )

        self.assertGreaterEqual(len(identifiers), 4)

        # Should have FR:VAT
        vat_ids = [i for i in identifiers if "VAT" in i.schemeID]
        self.assertGreater(len(vat_ids), 0)

        # Should have FR:SIRENE
        sirene_ids = [i for i in identifiers if "SIRENE" in i.schemeID]
        self.assertGreater(len(sirene_ids), 0)

    def test_case_insensitive_scheme_lookup(self):
        """PEPPOL IDs with case variations."""
        identifiers = generate_all_with_precedence(
            country_code="BE", vat="be0867709540", registration=None, peppol_participant_ids=["9925:BE0867709540"],
        )

        # Should handle case differences
        self.assertEqual(len(identifiers), 2)

    def test_empty_peppol_list(self):
        """Empty PEPPOL list behaves same as None."""
        ids1 = generate_all_with_precedence(
            country_code="BE", vat="BE0867709540", registration=None, peppol_participant_ids=[],
        )

        ids2 = generate_all_with_precedence(
            country_code="BE", vat="BE0867709540", registration=None, peppol_participant_ids=[],
        )

        self.assertEqual(len(ids1), len(ids2))


if __name__ == "__main__":
    unittest.main()
