"""
Tests for Party component and related elements.

Tests the Party dataclass with automatic identifier generation,
as well as PartyIdentification, PartyTaxScheme, and PartyLegalEntity.
"""

import unittest

from ubl.exceptions import ValidationError
from ubl.models.aggregate_components import (
    Party,
    PartyIdentification,
    PartyLegalEntity,
    PartyTaxScheme,
    PostalAddress,
    TaxScheme,
)
from ubl.models.basic_components import CompanyID, Identifier


class TestPartyIdentification(unittest.TestCase):
    """Test PartyIdentification component."""

    def test_create_party_identification(self):
        """Create PartyIdentification with all fields."""
        party_id = PartyIdentification(value="BE0867709540", schemeID="BE:VAT", iso6523="9925")

        self.assertEqual(party_id.value, "BE0867709540")
        self.assertEqual(party_id.schemeID, "BE:VAT")
        self.assertEqual(party_id.iso6523, "9925")

    def test_party_identification_generates_id(self):
        """PartyIdentification generates cbc:ID child."""
        party_id = PartyIdentification(value="BE0867709540", schemeID="BE:VAT", iso6523="9925")

        self.assertIsInstance(party_id.id, Identifier)
        self.assertEqual(party_id.id.value, "BE0867709540")
        self.assertEqual(party_id.id.schemeID, "BE:VAT")

    def test_party_identification_equality(self):
        """PartyIdentification equality with case-insensitive comparison."""
        id1 = PartyIdentification(value="BE0867709540", schemeID="BE:VAT", iso6523="9925")
        id2 = PartyIdentification(value="be0867709540", schemeID="BE:VAT", iso6523="9925")

        self.assertEqual(id1, id2)

    def test_party_identification_hash(self):
        """PartyIdentification hash for set deduplication."""
        id1 = PartyIdentification(value="BE0867709540", schemeID="BE:VAT", iso6523="9925")
        id2 = PartyIdentification(value="be0867709540", schemeID="BE:VAT", iso6523="9925")

        self.assertEqual(hash(id1), hash(id2))

        # Can deduplicate in set
        id_set = {id1, id2}
        self.assertEqual(len(id_set), 1)


class TestPartyTaxScheme(unittest.TestCase):
    """Test PartyTaxScheme component."""

    def test_create_party_tax_scheme(self):
        """Create PartyTaxScheme with company ID."""
        company_id = CompanyID(value="BE0867709540", schemeID="BE:VAT")
        party_tax_scheme = PartyTaxScheme(company_id=company_id)

        self.assertEqual(party_tax_scheme.company_id.value, "BE0867709540")
        self.assertIsInstance(party_tax_scheme.tax_scheme, TaxScheme)
        self.assertEqual(party_tax_scheme.tax_scheme.id.value, "VAT")

    def test_party_tax_scheme_custom_tax_scheme(self):
        """PartyTaxScheme with custom tax scheme."""
        company_id = CompanyID(value="12345")
        tax_scheme = TaxScheme(id=Identifier(value="GST", schemeID="UN/ECE 5153"))
        party_tax_scheme = PartyTaxScheme(company_id=company_id, tax_scheme=tax_scheme)

        self.assertEqual(party_tax_scheme.tax_scheme.id.value, "GST")


class TestPartyLegalEntity(unittest.TestCase):
    """Test PartyLegalEntity component."""

    def test_create_party_legal_entity(self):
        """Create PartyLegalEntity with name and company ID."""
        company_id = CompanyID(value="0597601756", schemeID="BE:EN")
        entity = PartyLegalEntity(registration_name="LevIT SC", company_id=company_id)

        self.assertEqual(entity.registration_name.value, "LevIT SC")
        self.assertEqual(entity.company_id.value, "0597601756")

    def test_party_legal_entity_no_company_id(self):
        """PartyLegalEntity without company ID."""
        entity = PartyLegalEntity(registration_name="Test Company")

        self.assertEqual(entity.registration_name.value, "Test Company")
        self.assertIsNone(entity.company_id)


class TestPartyComponent(unittest.TestCase):
    """Test Party component with identifier generation."""

    def setUp(self):
        """Set up test fixtures."""
        self.address = PostalAddress(
            street_name="Rue de Test",
            city_name="Brussels",
            postal_zone="1000",
            country="BE",
        )

    def test_create_party_with_vat(self):
        """Create party with VAT → generates identifiers."""
        party = Party(
            name="P4X SA",
            country_code="BE",
            postal_address=self.address,
            vat="BE0867709540",
        )

        self.assertEqual(party.name, "P4X SA")
        self.assertEqual(party.country_code, "BE")
        self.assertEqual(party.vat, "BE0867709540")

        # Should generate 2 identifiers
        identifiers = party.all_identifiers
        self.assertEqual(len(identifiers), 2)

    def test_party_endpoint_id_property(self):
        """Party.endpoint_id returns first identifier."""
        from ubl.models.basic_components import EndpointID

        party = Party(
            name="P4X SA",
            country_code="BE",
            postal_address=self.address,
            vat="BE0867709540",
        )

        endpoint_id = party.endpoint_id
        self.assertIsInstance(endpoint_id, EndpointID)
        self.assertEqual(endpoint_id.value, "BE0867709540")
        self.assertEqual(endpoint_id.schemeID, "BE:VAT")

    def test_party_identification_list_property(self):
        """Party.party_identification_list returns all identifiers."""
        party = Party(
            name="P4X SA",
            country_code="BE",
            postal_address=self.address,
            vat="BE0867709540",
        )

        id_list = party.party_identification_list
        self.assertEqual(len(id_list), 2)
        self.assertIsInstance(id_list[0], PartyIdentification)

    def test_party_tax_scheme_property(self):
        """Party.party_tax_scheme returns tax registration."""
        party = Party(
            name="P4X SA",
            country_code="BE",
            postal_address=self.address,
            vat="BE0867709540",
        )

        tax_scheme = party.party_tax_scheme
        self.assertIsNotNone(tax_scheme)
        self.assertIsInstance(tax_scheme, PartyTaxScheme)
        self.assertEqual(tax_scheme.company_id.value, "BE0867709540")
        self.assertEqual(tax_scheme.company_id.schemeID, "BE:VAT")

    def test_party_tax_scheme_no_vat(self):
        """Party without VAT → party_tax_scheme is None."""
        party = Party(
            name="Test Company",
            country_code="BE",
            postal_address=self.address,
            registration="0597601756",  # Belgian enterprise number (valid)
        )

        self.assertIsNone(party.party_tax_scheme)

    def test_party_legal_entity_property(self):
        """Party.party_legal_entity returns legal registration."""
        party = Party(
            name="LevIT SC",
            country_code="BE",
            postal_address=self.address,
            vat="BE0597601756",
            registration="0597601756",
        )

        legal_entity = party.party_legal_entity
        self.assertIsNotNone(legal_entity)
        self.assertIsInstance(legal_entity, PartyLegalEntity)
        self.assertEqual(legal_entity.registration_name.value, "LevIT SC")
        self.assertIsNotNone(legal_entity.company_id)
        self.assertEqual(legal_entity.company_id.value, "0597601756")

    def test_party_with_vat_and_registration(self):
        """Party with both VAT and registration → 4 identifiers."""
        party = Party(
            name="LevIT SC",
            country_code="BE",
            postal_address=self.address,
            vat="BE0597601756",
            registration="0597601756",
        )

        identifiers = party.all_identifiers
        # Should have identifiers from both VAT and registration
        self.assertGreaterEqual(len(identifiers), 4)

    def test_party_with_peppol_ids(self):
        """Party with PEPPOL participant IDs."""
        party = Party(
            name="LevIT SC",
            country_code="BE",
            postal_address=self.address,
            vat="BE0597601756",
            registration="0597601756",
            peppol_participant_ids=["9925:be0597601756", "0208:0597601756"],
        )

        identifiers = party.all_identifiers
        self.assertEqual(len(identifiers), 4)

        # PEPPOL IDs should be in the list
        iso_values = {i.value for i in identifiers if i.schemeID in ["9925", "0208"]}
        self.assertIn("be0597601756", iso_values)
        self.assertIn("0597601756", iso_values)

    def test_party_with_contact(self):
        """Party with contact information."""
        from ubl.models.aggregate_components import Contact

        contact = Contact(electronic_mail='test@example.com', telephone='+32123456789')
        party = Party(
            name="P4X SA",
            country_code="BE",
            postal_address=self.address,
            vat="BE0867709540",
            contact=contact,
        )

        self.assertIsNotNone(party.contact)
        self.assertEqual(party.contact.electronic_mail.value, "test@example.com")

    def test_party_with_website(self):
        """Party with website URI."""
        party = Party(
            name="P4X SA",
            country_code="BE",
            postal_address=self.address,
            vat="BE0867709540",
            website_uri="https://www.p4x.be",
        )

        self.assertEqual(party.website_uri, "https://www.p4x.be")

    def test_party_netherlands_vat_only(self):
        """Netherlands party - only VAT (registration schemes have strict length requirements)."""
        address = PostalAddress(
            street_name="Test Street",
            city_name="Amsterdam",
            postal_zone="1000AA",
            country="NL",
        )

        party = Party(
            name="Squads B.V.",
            country_code="NL",
            postal_address=address,
            vat="NL855934682B01",
            registration=None,  # NL schemes (KVK, OINO) have strict length requirements
        )

        # Only VAT generates identifiers
        identifiers = party.all_identifiers
        self.assertEqual(len(identifiers), 2)

        # Should have VAT identifiers
        vat_ids = [i for i in identifiers if "VAT" in i.schemeID or i.schemeID == "9944"]
        self.assertGreater(len(vat_ids), 0)


if __name__ == "__main__":
    unittest.main()
