from starfyre import compile
from pathlib import Path
from starfyre.file_router import FileRouter
from starfyre.build_main_file_content import get_build_main_file_content

import sys
import os
import shutil
import subprocess
import click
import importlib.resources as pkg_resources


def write_js_file(path):
    dist_path = Path(path) / "dist"
    dist_path.mkdir(exist_ok=True)
    js_store = pkg_resources.path("starfyre.js", "store.js")
    shutil.copy(str(js_store), path + "/dist/store.js")


def prepare_html_and_main(generated_routes, path):
    """
    Build the HTML output files for each generated route and create the main execution file.

    Parameters:
    - generated_routes (list): List of generated route names.
    - path (str): Path to the project directory.

    This function gets the HTML output for each generated route based on the provided `generated_routes` list.
    It also creates the main execution file (__main__.py) that orchestrates rendering and writing HTML content to files.
    The rendered HTML content is generated by importing and utilizing the corresponding components for each route in the __main__ file.

    Args:
        generated_routes (list): A list of route names that have been generated.
        path (str): The path to the project directory.
    """
    write_js_file(path)

    main_file_path = path + "/build/__main__.py"
    init_file_path = path + "/build/__init__.py"
    main_file_content = get_build_main_file_content(user_routes=generated_routes, path=path)

    # create empty __init__.py file
    Path(init_file_path).touch()

    with open(main_file_path, "w") as f:
        f.write(main_file_content)


@click.command()
@click.option("--path", help="Path to the project")
@click.option("--build", is_flag=True, help="Compile and build package")
def main(path, build):
    """
    Command-line interface to compile and build a Starfyre project.

    Args:

        path (str): Path to the project directory.\n
        build (bool): Whether to start the build package.
    """
    if not path:
        click.echo(
            "Error: Please provide a valid path using the --path flag.\nUse --help for more details")
        return

    # Convert path to absolute path
    absolute_path = Path(path).resolve()
    print(f'Absolute path of the project = {absolute_path}')

    if build:
        # Compile and build project
        init_file_path = absolute_path / "__init__.py"
        # Note: The routes specified in the pages folder will have generated code in the build directory.
        compile(init_file_path.resolve())

        # At this point, the project has been compiled and the build directory has been created.
        # Now, initialize the Router object and use it to handle file-based routing.
        # Basically, get all the file names from the "pages" directory
        file_router = FileRouter(absolute_path / "pages")
        routes = file_router.populate_router()

        # We have to create the main file.
        # The main file will be used to generate the HTML output for all routes found by the FileRouter, index route inclusively.
        prepare_html_and_main(generated_routes=routes, path=str(absolute_path))

        # Start/run project
        result = subprocess.run(
            [sys.executable, "-m", "build"],
            cwd=path,
            stdout=subprocess.PIPE,
            stderr=None,
        )


if __name__ == "__main__":
    main()
