"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const policy_document_1 = require("./policy-document");
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
class Grant {
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options) {
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource
        });
        if (result.success) {
            return result;
        }
        const statement = new policy_document_1.PolicyStatement()
            .addActions(...options.actions)
            .addResources(...(options.resourceSelfArns || options.resourceArns))
            .addPrincipal(options.grantee.grantPrincipal);
        options.resource.addToResourcePolicy(statement);
        return new Grant({ resourceStatement: statement, options });
    }
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options) {
        const statement = new policy_document_1.PolicyStatement()
            .addActions(...options.actions)
            .addResources(...options.resourceArns);
        const addedToPrincipal = options.grantee.grantPrincipal.addToPolicy(statement);
        return new Grant({ principalStatement: addedToPrincipal ? statement : undefined, options });
    }
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the pricipal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options) {
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const statement = new policy_document_1.PolicyStatement()
            .addActions(...options.actions)
            .addResources(...(options.resourceSelfArns || options.resourceArns))
            .addPrincipal(options.grantee.grantPrincipal);
        options.resource.addToResourcePolicy(statement);
        return new Grant({ principalStatement: statement, resourceStatement: result.resourceStatement, options });
    }
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee, _intent) {
        return new Grant({
            options: { grantee, actions: [], resourceArns: [] }
        });
    }
    constructor(props) {
        this.options = props.options;
        this.principalStatement = props.principalStatement;
        this.resourceStatement = props.resourceStatement;
    }
    /**
     * Whether the grant operation was successful
     */
    get success() {
        return this.principalStatement !== undefined || this.resourceStatement !== undefined;
    }
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess() {
        if (!this.success) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`${describeGrant(this.options)} could not be added on either identity or resource policy.`);
        }
    }
}
exports.Grant = Grant;
function describeGrant(options) {
    return `Permissions for '${options.grantee}' to call '${options.actions}' on '${options.resourceArns}'`;
}
//# sourceMappingURL=data:application/json;base64,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