"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const policy_document_1 = require("./policy-document");
const util_1 = require("./util");
class GroupBase extends cdk_1.Resource {
    constructor() {
        super(...arguments);
        this.grantPrincipal = this;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.attachedPolicies = new util_1.AttachedPolicies();
    }
    get policyFragment() {
        return new policy_document_1.ArnPrincipal(this.groupArn).policyFragment;
    }
    /**
     * Attaches a policy to this group.
     * @param policy The policy to attach.
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToGroup(this);
    }
    attachManagedPolicy(_arn) {
        // drop
    }
    /**
     * Adds a user to this group.
     */
    addUser(user) {
        user.addToGroup(this);
    }
    /**
     * Adds an IAM statement to the default policy.
     */
    addToPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.defaultPolicy.attachToGroup(this);
        }
        this.defaultPolicy.addStatement(statement);
        return true;
    }
}
class Group extends GroupBase {
    /**
     * Imports a group from ARN
     * @param groupArn (e.g. `arn:aws:iam::account-id:group/group-name`)
     */
    static fromGroupArn(scope, id, groupArn) {
        const groupName = scope.node.stack.parseArn(groupArn).resourceName;
        class Import extends GroupBase {
            constructor() {
                super(...arguments);
                this.groupName = groupName;
                this.groupArn = groupArn;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.managedPolicies = props.managedPolicyArns || [];
        const group = new iam_generated_1.CfnGroup(this, 'Resource', {
            groupName: props.groupName,
            managedPolicyArns: util_1.undefinedIfEmpty(() => this.managedPolicies),
            path: props.path,
        });
        this.groupName = group.groupName;
        this.groupArn = group.groupArn;
    }
    /**
     * Attaches a managed policy to this group.
     * @param arn The ARN of the managed policy to attach.
     */
    attachManagedPolicy(arn) {
        this.managedPolicies.push(arn);
    }
}
exports.Group = Group;
//# sourceMappingURL=data:application/json;base64,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