import cdk = require('@aws-cdk/cdk');
import { IPrincipal } from './principals';
export declare class PolicyDocument extends cdk.Token implements cdk.IResolvedValuePostProcessor {
    private readonly baseDocument;
    private statements;
    private _autoAssignSids;
    /**
     * Creates a new IAM policy document.
     * @param defaultDocument An IAM policy document to use as an initial
     * policy. All statements of this document will be copied in.
     */
    constructor(baseDocument?: any);
    /**
     * Will automatically assign a unique SID to each statement, unless an SID is provided.
     */
    autoAssignSids(): void;
    resolve(_context: cdk.IResolveContext): any;
    /**
     * Removes duplicate statements
     */
    postProcess(input: any, _context: cdk.IResolveContext): any;
    readonly isEmpty: boolean;
    /**
     * The number of statements already added to this policy.
     * Can be used, for example, to generate uniuqe "sid"s within the policy.
     */
    readonly statementCount: number;
    /**
     * Adds a statement to the policy document.
     *
     * @param statement the statement to add.
     */
    addStatement(statement: PolicyStatement): PolicyDocument;
}
/**
 * Base class for policy principals
 */
export declare abstract class PrincipalBase implements IPrincipal {
    readonly grantPrincipal: IPrincipal;
    /**
     * Return the policy fragment that identifies this principal in a Policy.
     */
    abstract readonly policyFragment: PrincipalPolicyFragment;
    /**
     * When this Principal is used in an AssumeRole policy, the action to use.
     */
    readonly assumeRoleAction: string;
    addToPolicy(_statement: PolicyStatement): boolean;
    toString(): string;
    toJSON(): {
        [key: string]: string[];
    };
}
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
export declare class PrincipalPolicyFragment {
    readonly principalJson: {
        [key: string]: string[];
    };
    readonly conditions: {
        [key: string]: any;
    };
    constructor(principalJson: {
        [key: string]: string[];
    }, conditions?: {
        [key: string]: any;
    });
}
export declare class ArnPrincipal extends PrincipalBase {
    readonly arn: string;
    constructor(arn: string);
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
export declare class AccountPrincipal extends ArnPrincipal {
    readonly accountId: any;
    constructor(accountId: any);
    toString(): string;
}
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
export declare class ServicePrincipal extends PrincipalBase {
    readonly service: string;
    private readonly opts;
    constructor(service: string, opts?: ServicePrincipalOpts);
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
/**
 * A principal that represents an AWS Organization
 */
export declare class OrganizationPrincipal extends PrincipalBase {
    readonly organizationId: string;
    constructor(organizationId: string);
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
/**
 * A policy prinicipal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
export declare class CanonicalUserPrincipal extends PrincipalBase {
    readonly canonicalUserId: string;
    constructor(canonicalUserId: string);
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
export declare class FederatedPrincipal extends PrincipalBase {
    readonly federated: string;
    readonly conditions: {
        [key: string]: any;
    };
    readonly assumeRoleAction: string;
    constructor(federated: string, conditions: {
        [key: string]: any;
    }, assumeRoleAction?: string);
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
export declare class AccountRootPrincipal extends AccountPrincipal {
    constructor();
    toString(): string;
}
/**
 * A principal representing all identities in all accounts
 */
export declare class AnyPrincipal extends ArnPrincipal {
    constructor();
    toString(): string;
}
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
export declare class Anyone extends AnyPrincipal {
}
export declare class CompositePrincipal extends PrincipalBase {
    readonly assumeRoleAction: string;
    private readonly principals;
    constructor(...principals: PrincipalBase[]);
    addPrincipals(...principals: PrincipalBase[]): this;
    readonly policyFragment: PrincipalPolicyFragment;
    toString(): string;
}
/**
 * Represents a statement in an IAM policy document.
 */
export declare class PolicyStatement extends cdk.Token {
    sid?: string;
    private action;
    private principal;
    private resource;
    private condition;
    private effect?;
    constructor(effect?: PolicyStatementEffect);
    addAction(action: string): PolicyStatement;
    addActions(...actions: string[]): PolicyStatement;
    /**
     * Indicates if this permission has a "Principal" section.
     */
    readonly hasPrincipal: boolean;
    addPrincipal(principal: IPrincipal): this;
    addAwsPrincipal(arn: string): this;
    addAwsAccountPrincipal(accountId: string): this;
    addArnPrincipal(arn: string): this;
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts    options for adding the service principal (such as specifying a principal in a different region)
     */
    addServicePrincipal(service: string, opts?: ServicePrincipalOpts): this;
    addFederatedPrincipal(federated: any, conditions: {
        [key: string]: any;
    }): this;
    addAccountRootPrincipal(): this;
    addCanonicalUserPrincipal(canonicalUserId: string): this;
    addAnyPrincipal(): this;
    addResource(arn: string): PolicyStatement;
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources(): PolicyStatement;
    addResources(...arns: string[]): PolicyStatement;
    /**
     * Indicates if this permission as at least one resource associated with it.
     */
    readonly hasResource: boolean;
    /**
     * @deprecated Use `statement.sid = value`
     */
    describe(sid: string): PolicyStatement;
    /**
     * Sets the permission effect to allow access to resources.
     */
    allow(): PolicyStatement;
    /**
     * Sets the permission effect to deny access to resources.
     */
    deny(): PolicyStatement;
    /**
     * Add a condition to the Policy
     */
    addCondition(key: string, value: any): PolicyStatement;
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions: {
        [key: string]: any;
    }): PolicyStatement;
    /**
     * Add a condition to the Policy.
     *
     * @deprecated For backwards compatibility. Use addCondition() instead.
     */
    setCondition(key: string, value: any): PolicyStatement;
    limitToAccount(accountId: string): PolicyStatement;
    resolve(_context: cdk.IResolveContext): any;
    toJson(): any;
}
export declare enum PolicyStatementEffect {
    Allow = "Allow",
    Deny = "Deny"
}
/**
 * Options for a service principal.
 */
export interface ServicePrincipalOpts {
    /**
     * The region in which the service is operating.
     *
     * @default the current Stack's region.
     */
    readonly region?: string;
}
