"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/cdk");
const region_info_1 = require("@aws-cdk/region-info");
const util_1 = require("./util");
class PolicyDocument extends cdk.Token {
    /**
     * Creates a new IAM policy document.
     * @param defaultDocument An IAM policy document to use as an initial
     * policy. All statements of this document will be copied in.
     */
    constructor(baseDocument = {}) {
        super();
        this.baseDocument = baseDocument;
        this.statements = new Array();
        this._autoAssignSids = false;
    }
    /**
     * Will automatically assign a unique SID to each statement, unless an SID is provided.
     */
    autoAssignSids() {
        this._autoAssignSids = true;
    }
    resolve(_context) {
        if (this.isEmpty) {
            return undefined;
        }
        const doc = {
            ...this.baseDocument,
            Statement: (this.baseDocument.Statement || []).concat(this.statements),
            Version: this.baseDocument.Version || '2012-10-17'
        };
        return doc;
    }
    /**
     * Removes duplicate statements
     */
    postProcess(input, _context) {
        if (!input || !input.Statement) {
            return input;
        }
        const jsonStatements = new Set();
        const uniqueStatements = [];
        for (const statement of input.Statement) {
            const jsonStatement = JSON.stringify(statement);
            if (!jsonStatements.has(jsonStatement)) {
                uniqueStatements.push(statement);
                jsonStatements.add(jsonStatement);
            }
        }
        // assign unique SIDs (the statement index) if `autoAssignSids` is enabled
        const statements = uniqueStatements.map((s, i) => {
            if (this._autoAssignSids && !s.Sid) {
                s.Sid = i.toString();
            }
            return s;
        });
        return {
            ...input,
            Statement: statements
        };
    }
    get isEmpty() {
        return this.statements.length === 0;
    }
    /**
     * The number of statements already added to this policy.
     * Can be used, for example, to generate uniuqe "sid"s within the policy.
     */
    get statementCount() {
        return this.statements.length;
    }
    /**
     * Adds a statement to the policy document.
     *
     * @param statement the statement to add.
     */
    addStatement(statement) {
        this.statements.push(statement);
        return this;
    }
}
exports.PolicyDocument = PolicyDocument;
/**
 * Base class for policy principals
 */
class PrincipalBase {
    constructor() {
        this.grantPrincipal = this;
        /**
         * When this Principal is used in an AssumeRole policy, the action to use.
         */
        this.assumeRoleAction = 'sts:AssumeRole';
    }
    addToPolicy(_statement) {
        // This base class is used for non-identity principals. None of them
        // have a PolicyDocument to add to.
        return false;
    }
    toString() {
        // This is a first pass to make the object readable. Descendant principals
        // should return something nicer.
        return JSON.stringify(this.policyFragment.principalJson);
    }
    toJSON() {
        // Have to implement toJSON() because the default will lead to infinite recursion.
        return this.policyFragment.principalJson;
    }
}
exports.PrincipalBase = PrincipalBase;
/**
 * A collection of the fields in a PolicyStatement that can be used to identify a principal.
 *
 * This consists of the JSON used in the "Principal" field, and optionally a
 * set of "Condition"s that need to be applied to the policy.
 */
class PrincipalPolicyFragment {
    constructor(principalJson, conditions = {}) {
        this.principalJson = principalJson;
        this.conditions = conditions;
    }
}
exports.PrincipalPolicyFragment = PrincipalPolicyFragment;
class ArnPrincipal extends PrincipalBase {
    constructor(arn) {
        super();
        this.arn = arn;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: [this.arn] });
    }
    toString() {
        return `ArnPrincipal(${this.arn})`;
    }
}
exports.ArnPrincipal = ArnPrincipal;
class AccountPrincipal extends ArnPrincipal {
    constructor(accountId) {
        super(new StackDependentToken(stack => `arn:${stack.partition}:iam::${accountId}:root`).toString());
        this.accountId = accountId;
    }
    toString() {
        return `AccountPrincipal(${this.accountId})`;
    }
}
exports.AccountPrincipal = AccountPrincipal;
/**
 * An IAM principal that represents an AWS service (i.e. sqs.amazonaws.com).
 */
class ServicePrincipal extends PrincipalBase {
    constructor(service, opts = {}) {
        super();
        this.service = service;
        this.opts = opts;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({
            Service: [
                new ServicePrincipalToken(this.service, this.opts).toString()
            ]
        });
    }
    toString() {
        return `ServicePrincipal(${this.service})`;
    }
}
exports.ServicePrincipal = ServicePrincipal;
/**
 * A principal that represents an AWS Organization
 */
class OrganizationPrincipal extends PrincipalBase {
    constructor(organizationId) {
        super();
        this.organizationId = organizationId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ AWS: ['*'] }, { StringEquals: { 'aws:PrincipalOrgID': this.organizationId } });
    }
    toString() {
        return `OrganizationPrincipal(${this.organizationId})`;
    }
}
exports.OrganizationPrincipal = OrganizationPrincipal;
/**
 * A policy prinicipal for canonicalUserIds - useful for S3 bucket policies that use
 * Origin Access identities.
 *
 * See https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html
 *
 * and
 *
 * https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/private-content-restricting-access-to-s3.html
 *
 * for more details.
 *
 */
class CanonicalUserPrincipal extends PrincipalBase {
    constructor(canonicalUserId) {
        super();
        this.canonicalUserId = canonicalUserId;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ CanonicalUser: [this.canonicalUserId] });
    }
    toString() {
        return `CanonicalUserPrincipal(${this.canonicalUserId})`;
    }
}
exports.CanonicalUserPrincipal = CanonicalUserPrincipal;
class FederatedPrincipal extends PrincipalBase {
    constructor(federated, conditions, assumeRoleAction = 'sts:AssumeRole') {
        super();
        this.federated = federated;
        this.conditions = conditions;
        this.assumeRoleAction = assumeRoleAction;
    }
    get policyFragment() {
        return new PrincipalPolicyFragment({ Federated: [this.federated] }, this.conditions);
    }
    toString() {
        return `FederatedPrincipal(${this.federated})`;
    }
}
exports.FederatedPrincipal = FederatedPrincipal;
class AccountRootPrincipal extends AccountPrincipal {
    constructor() {
        super(new StackDependentToken(stack => stack.accountId).toString());
    }
    toString() {
        return `AccountRootPrincipal()`;
    }
}
exports.AccountRootPrincipal = AccountRootPrincipal;
/**
 * A principal representing all identities in all accounts
 */
class AnyPrincipal extends ArnPrincipal {
    constructor() {
        super('*');
    }
    toString() {
        return `AnyPrincipal()`;
    }
}
exports.AnyPrincipal = AnyPrincipal;
/**
 * A principal representing all identities in all accounts
 * @deprecated use `AnyPrincipal`
 */
class Anyone extends AnyPrincipal {
}
exports.Anyone = Anyone;
class CompositePrincipal extends PrincipalBase {
    constructor(...principals) {
        super();
        this.principals = new Array();
        if (principals.length === 0) {
            throw new Error('CompositePrincipals must be constructed with at least 1 Principal but none were passed.');
        }
        this.assumeRoleAction = principals[0].assumeRoleAction;
        this.addPrincipals(...principals);
    }
    addPrincipals(...principals) {
        for (const p of principals) {
            if (p.assumeRoleAction !== this.assumeRoleAction) {
                throw new Error(`Cannot add multiple principals with different "assumeRoleAction". ` +
                    `Expecting "${this.assumeRoleAction}", got "${p.assumeRoleAction}"`);
            }
            const fragment = p.policyFragment;
            if (fragment.conditions && Object.keys(fragment.conditions).length > 0) {
                throw new Error(`Components of a CompositePrincipal must not have conditions. ` +
                    `Tried to add the following fragment: ${JSON.stringify(fragment)}`);
            }
            this.principals.push(p);
        }
        return this;
    }
    get policyFragment() {
        const principalJson = {};
        for (const p of this.principals) {
            util_1.mergePrincipal(principalJson, p.policyFragment.principalJson);
        }
        return new PrincipalPolicyFragment(principalJson);
    }
    toString() {
        return `CompositePrincipal(${this.principals})`;
    }
}
exports.CompositePrincipal = CompositePrincipal;
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement extends cdk.Token {
    constructor(effect = PolicyStatementEffect.Allow) {
        super();
        this.action = new Array();
        this.principal = {};
        this.resource = new Array();
        this.condition = {};
        this.effect = effect;
    }
    //
    // Actions
    //
    addAction(action) {
        this.action.push(action);
        return this;
    }
    addActions(...actions) {
        actions.forEach(action => this.addAction(action));
        return this;
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0;
    }
    addPrincipal(principal) {
        const fragment = principal.policyFragment;
        util_1.mergePrincipal(this.principal, fragment.principalJson);
        this.addConditions(fragment.conditions);
        return this;
    }
    addAwsPrincipal(arn) {
        return this.addPrincipal(new ArnPrincipal(arn));
    }
    addAwsAccountPrincipal(accountId) {
        return this.addPrincipal(new AccountPrincipal(accountId));
    }
    addArnPrincipal(arn) {
        return this.addAwsPrincipal(arn);
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts    options for adding the service principal (such as specifying a principal in a different region)
     */
    addServicePrincipal(service, opts) {
        return this.addPrincipal(new ServicePrincipal(service, opts));
    }
    addFederatedPrincipal(federated, conditions) {
        return this.addPrincipal(new FederatedPrincipal(federated, conditions));
    }
    addAccountRootPrincipal() {
        return this.addPrincipal(new AccountRootPrincipal());
    }
    addCanonicalUserPrincipal(canonicalUserId) {
        return this.addPrincipal(new CanonicalUserPrincipal(canonicalUserId));
    }
    addAnyPrincipal() {
        return this.addPrincipal(new Anyone());
    }
    //
    // Resources
    //
    addResource(arn) {
        this.resource.push(arn);
        return this;
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources() {
        return this.addResource('*');
    }
    addResources(...arns) {
        arns.forEach(r => this.addResource(r));
        return this;
    }
    /**
     * Indicates if this permission as at least one resource associated with it.
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    /**
     * @deprecated Use `statement.sid = value`
     */
    describe(sid) {
        this.sid = sid;
        return this;
    }
    //
    // Effect
    //
    /**
     * Sets the permission effect to allow access to resources.
     */
    allow() {
        this.effect = PolicyStatementEffect.Allow;
        return this;
    }
    /**
     * Sets the permission effect to deny access to resources.
     */
    deny() {
        this.effect = PolicyStatementEffect.Deny;
        return this;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy
     */
    addCondition(key, value) {
        this.condition[key] = value;
        return this;
    }
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
        return this;
    }
    /**
     * Add a condition to the Policy.
     *
     * @deprecated For backwards compatibility. Use addCondition() instead.
     */
    setCondition(key, value) {
        return this.addCondition(key, value);
    }
    limitToAccount(accountId) {
        return this.addCondition('StringEquals', new cdk.Token(() => {
            return { 'sts:ExternalId': accountId };
        }));
    }
    //
    // Serialization
    //
    resolve(_context) {
        return this.toJson();
    }
    toJson() {
        return {
            Action: _norm(this.action),
            Condition: _norm(this.condition),
            Effect: _norm(this.effect),
            Principal: _normPrincipal(this.principal),
            Resource: _norm(this.resource),
            Sid: _norm(this.sid),
        };
        function _norm(values) {
            if (typeof (values) === 'undefined') {
                return undefined;
            }
            if (cdk.Token.isToken(values)) {
                return values;
            }
            if (Array.isArray(values)) {
                if (!values || values.length === 0) {
                    return undefined;
                }
                if (values.length === 1) {
                    return values[0];
                }
                return values;
            }
            if (typeof (values) === 'object') {
                if (Object.keys(values).length === 0) {
                    return undefined;
                }
            }
            return values;
        }
        function _normPrincipal(principal) {
            const keys = Object.keys(principal);
            if (keys.length === 0) {
                return undefined;
            }
            const result = {};
            for (const key of keys) {
                const normVal = _norm(principal[key]);
                if (normVal) {
                    result[key] = normVal;
                }
            }
            if (Object.keys(result).length === 1 && result.AWS === '*') {
                return '*';
            }
            return result;
        }
    }
}
exports.PolicyStatement = PolicyStatement;
var PolicyStatementEffect;
(function (PolicyStatementEffect) {
    PolicyStatementEffect["Allow"] = "Allow";
    PolicyStatementEffect["Deny"] = "Deny";
})(PolicyStatementEffect = exports.PolicyStatementEffect || (exports.PolicyStatementEffect = {}));
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken extends cdk.Token {
    constructor(fn) {
        super();
        this.fn = fn;
    }
    resolve(context) {
        return this.fn(context.scope.node.stack);
    }
}
class ServicePrincipalToken extends cdk.Token {
    constructor(service, opts) {
        super();
        this.service = service;
        this.opts = opts;
    }
    resolve(ctx) {
        const region = this.opts.region || ctx.scope.node.stack.region;
        const fact = region_info_1.RegionInfo.get(region).servicePrincipal(this.service);
        return fact || region_info_1.Default.servicePrincipal(this.service, region, ctx.scope.node.stack.urlSuffix);
    }
}
//# sourceMappingURL=data:application/json;base64,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