"""
This module defines the `PerturbationOcclusion` class, which implements a generator composed of
modular perturbation and occlusion-based algorithms
"""

from __future__ import annotations

from collections.abc import Sequence
from typing import Any, TypeVar

import numpy as np
from smqtk_core.configuration import from_config_dict, make_default_config, to_config_dict
from smqtk_descriptors.interfaces.image_descriptor_generator import ImageDescriptorGenerator
from typing_extensions import Self

from xaitk_saliency import GenerateDescriptorSimilaritySaliency, GenerateImageSimilarityBlackboxSaliency, PerturbImage
from xaitk_saliency.utils.masking import occlude_image_batch

C = TypeVar("C", bound="PerturbationOcclusion")


class PerturbationOcclusion(GenerateImageSimilarityBlackboxSaliency):
    """
    Image similarity saliency generator composed of modular perturbation and
    occlusion-based algorithms.

    This implementation exposes its `fill` attribute as public.
    This allows it to be set during runtime as this is most often driven by the
    black-box algorithm used, if at all.
    """

    def __init__(
        self,
        perturber: PerturbImage,
        generator: GenerateDescriptorSimilaritySaliency,
        fill: int | Sequence[int] | np.ndarray | None = None,
        threads: int | None = None,
    ) -> None:
        """
        Image similarity saliency generator composed of modular perturbation and
        occlusion-based algorithms.

        :param perturber: `PerturbImage` implementation instance for generating
            occlusion masks.
        :param generator: `GenerateDescriptorSimilaritySaliency` implementation
            instance for generating saliency masks given occlusion masks and
            image feature vector generator outputs.
        :param fill: Optional fill for alpha-blending the occluded regions based on
            the masks generated by the given perturber.
            Can be a scalar value, a per-channel sequence or a shape-matched image.
        :param threads: Optional number threads to use to enable parallelism in
            applying perturbation masks to an input image.
            If 0, a negative value, or `None`, work will be performed on the
            main-thread in-line.
        """
        self._perturber = perturber
        self._generator = generator
        self._threads = threads
        self.fill = fill

    def _generate(
        self,
        ref_image: np.ndarray,
        query_images: Sequence[np.ndarray],
        blackbox: ImageDescriptorGenerator,
    ) -> np.ndarray:
        ref_feat = np.array(list(blackbox.generate_arrays_from_images([ref_image]))[0])
        query_feats = np.array(list(blackbox.generate_arrays_from_images(query_images)))

        pert_masks = self._perturber(ref_image)

        pert_imgs = occlude_image_batch(ref_image, pert_masks, fill=self.fill, threads=self._threads)

        pert_feats = np.array(list(blackbox.generate_arrays_from_images(pert_imgs)))

        return self._generator(ref_feat, query_feats, pert_feats, pert_masks)

    @classmethod
    def get_default_config(cls) -> dict[str, Any]:
        """
        Returns the default configuration for the PerturbationOcclusion.

        This method provides a default configuration dictionary, specifying default
        values for key parameters in the factory. It can be used to create an instance
        of the factory with preset configurations.

        Returns:
            dict[str, Any]: A dictionary containing default configuration parameters.
        """
        cfg = super().get_default_config()
        cfg["perturber"] = make_default_config(PerturbImage.get_impls())
        cfg["generator"] = make_default_config(GenerateDescriptorSimilaritySaliency.get_impls())
        return cfg

    @classmethod
    def from_config(cls, config_dict: dict, merge_default: bool = True) -> Self:
        """
        Create a PerturbationOcclusion instance from a configuration dictionary.

        Args:
            config_dict (dict): Configuration dictionary with perturber details.
            merge_default (bool): Whether to merge with the default configuration.

        Returns:
            PerturbationOcclusion: An instance of PerturbationOcclusion.
        """
        config_dict = dict(config_dict)  # shallow-copy
        config_dict["perturber"] = from_config_dict(config_dict["perturber"], PerturbImage.get_impls())
        config_dict["generator"] = from_config_dict(
            config_dict["generator"],
            GenerateDescriptorSimilaritySaliency.get_impls(),
        )
        return super().from_config(config_dict, merge_default=merge_default)

    def get_config(self) -> dict[str, Any]:
        """
        Get the configuration dictionary of the PerturbationOcclusion instance.

        Returns:
            dict[str, Any]: Configuration dictionary.
        """
        return {
            "perturber": to_config_dict(self._perturber),
            "generator": to_config_dict(self._generator),
            "fill": self.fill,
            "threads": self._threads,
        }
