#################################################
# IMPORTS
#################################################
from __future__ import annotations

import inspect
from pathlib import Path
from typing import Any

from InquirerPy import inquirer  # type: ignore
from InquirerPy.validator import EmptyInputValidator  # type: ignore
from click import Command, Option

from ..utils.cli import clear, confirm
from .custom_group import CustomGroup
from .menu import Menus

#################################################
# CODE
#################################################
dicts = dict[str, Any]


class Builder(CustomGroup):

    def __init__(self) -> None:
        super().__init__()

    def create(self) -> Command:
        help = "Create all files for the containerization."
        options = [Option(["--network"], is_flag=True, default=False)]

        def callback(network: bool = False) -> None:
            clear(0)

            services: set[dicts] = set([])
            networks: set[str] = set([])
            envs: set[dicts] = set([])

            if self.cwd.joinpath("data.json").exists():
                if not confirm(
                    msg="A data.json file was already found, want to continue? ",
                    default=False,
                ):
                    return

            if not network:
                menu = Menus()
                service, env = self.__get_data(menu)
                services.add(service)
                envs.add(env)

            else:
                network_name = self.__get_name(
                    message="Enter the name of the network: "
                )
                networks.add(network_name)

                menu = Menus(network=network_name)

                while True:
                    menu.ports = {}

                    service, env = self.__get_data(menu)
                    services.add(service)
                    envs.add(env)

                    clear(0.5)

                    if not confirm(
                        msg=f"Want to continue adding services? (Count: {len(services)})"
                    ):
                        break

            clear(0)
            self.file_manager.save_files(
                data={
                    "compose": {
                        "services": services,
                        "networks": networks,
                    },
                    "envs": envs,
                }
            )
            clear(0)
            print("Files saved!")

        return Command(
            name=inspect.currentframe().f_code.co_name,  # type: ignore
            help=help,
            callback=callback,
            params=options,  # type: ignore
        )

    def update(self) -> Command:
        help = "Update the contents of the containers."
        options = [
            Option(["--service"], type=self.service_type, default=None),
            Option(["--add"], is_flag=True, default=False),
            Option(["--remove"], is_flag=True, default=False),
        ]

        def callback(
            service: str | None = None, add: bool = False, remove: bool = False
        ) -> None:
            clear(0)

            path: Path = self.cwd.joinpath("data.json")

            if not path.exists():
                exit(
                    "ERROR: Missing JSON file for services. Use 'create' first."
                )

            data: dicts = self.file_manager.read_json(path) or {}
            compose: dicts = data.get("compose", {}) or {}

            services: list[dicts] = compose.get("services", []) or []
            networks: list[str] = compose.get("networks", []) or []
            envs: list[dicts] = data.get("envs", []) or []

            if not services:
                exit("ERROR: No services found. Use 'create' first.")

            def find_index_by_name(name: str) -> int | None:
                for i, s in enumerate(services):
                    if s.get("name") == name:
                        return i
                return None

            if remove:
                target = service
                if not target:
                    names = [s.get("name") for s in services if s.get("name")]
                    if not names:
                        exit("ERROR: No services found.")

                    target = inquirer.select(  # type: ignore
                        message="Select a service to remove: ", choices=names
                    ).execute()

                idx = find_index_by_name(target)  # type: ignore
                if idx is None:
                    exit(f"ERROR: Service '{target}' not found.")

                if confirm(msg=f"Remove service '{target}'", default=False):
                    services.pop(idx)
                    envs = [
                        e for e in envs if e.get("CONTAINER_NAME") != target
                    ]
                    compose["services"] = services
                    compose["networks"] = networks
                    data["compose"] = compose
                    data["envs"] = envs
                    self.file_manager.save_files(data)
                    print(f"Service '{target}' removed and files updated.")

            elif add:
                name = service
                if not name:
                    name = self.__get_name("Enter the name of the service: ")
                if find_index_by_name(name):
                    if not confirm(
                        msg=f"Service '{name}' already exists. Overwrite? "
                    ):
                        exit("ERROR: Add cancelled.")

                    services = [s for s in services if s.get("name") != name]
                    envs = [e for e in envs if e.get("CONTAINER_NAME") != name]

                network = None
                if networks:
                    network = inquirer.select(  # type: ignore
                        message="Select a network: ", choices=networks
                    ).execute()
                menu = Menus(network=network)

                service_obj, env_obj = self.__get_data(menu, name)
                service_obj["name"] = name
                env_obj["CONTAINER_NAME"] = name

                services.append(service_obj)
                envs.append(env_obj)

                if confirm(msg=f"Add/Update service '{name}'"):
                    compose["services"] = services
                    compose["networks"] = networks
                    data["compose"] = compose
                    data["envs"] = envs
                    self.file_manager.save_files(data)
                    print(f"Service '{name}' removed and files updated.")

            else:
                print("Use --add or --remove flag.")
                print("Use --services [service] for faster output.")
                for s in services:
                    print(f" - {s.get('name')}")

        return Command(
            name=inspect.currentframe().f_code.co_name,  # type: ignore
            help=help,
            callback=callback,
            params=options,  # type: ignore
        )

    def build(self) -> Command:
        help = "Build the files for the containerization."
        options: list[Option] = []

        def callback() -> None:
            clear(0)

            path: Path = self.cwd.joinpath("data.json")

            if not path.exists():
                exit(
                    "ERROR: Missing JSON file for services. Use 'create' first."
                )

            data: dicts = self.file_manager.read_json(path) or {}

            if not data:
                exit("ERROR: JSON file is empty. Use 'create' first.")

            clear(0)
            self.file_manager.save_files(data, build=True)
            clear(0)
            print("Files saved!")

        return Command(
            name=inspect.currentframe().f_code.co_name,  # type: ignore
            help=help,
            callback=callback,
            params=options,  # type: ignore
        )

    def __get_data(
        self, menu: Menus, name: str | None = None
    ) -> tuple[dicts, dicts]:
        clear(0.5)

        if not name:
            name = self.__get_name(message="Enter the name of the service: ")

        service = menu.service(name=name)
        env = menu.env(name=name)

        return (service, env)

    def __get_name(self, message: str, network: bool = False) -> str:
        while True:
            clear(0.5)
            name: str = inquirer.text(  # type: ignore
                message=message, validate=EmptyInputValidator()
            ).execute()

            if confirm(
                msg=f"Want to name this {"network" if network else "service"} '{name}'? ",
                default=True,
            ):
                break

        return name
