"""
Top-level API of vmware-reporter library.
"""
from __future__ import annotations

import json
import logging
import os
import re
from collections.abc import Callable, Iterator
from configparser import ConfigParser
from contextlib import nullcontext
from datetime import date
from inspect import signature
from io import IOBase
from pathlib import Path
from types import (BuiltinFunctionType, BuiltinMethodType, FunctionType,
                   MethodType)
from typing import Any, Literal, TypeVar, overload
from uuid import UUID

from pyVim.connect import Disconnect, SmartConnect
from pyVmomi import vim, vmodl
from pyVmomi.VmomiSupport import _managedDefMap
from zut import (ExtendedJSONEncoder, Filters, MessageError,
                 iter_dicts_from_csv, resolve_host)
from zut.excel import ExcelWorkbook, is_excel_path, split_excel_path

from .settings import CONFIG, CONFIG_SECTION

__prog__ = 'vmware-reporter'

try:
    # Version generated by setuptools_scm during build
    from ._version import __version__, __version_tuple__
except ImportError:
    __version__ = None
    __version_tuple__ = None

_logger = logging.getLogger(__name__)

T_Obj = TypeVar('T_Obj', bound=vim.ManagedEntity)


class VCenterClient:
    """
    Main entry point of the library to retrieve VMWare managed objects and interact with them. 
    """
    def __init__(self, env: str = None, *, host: str = None, user: str = None, password: str = None, no_ssl_verify: bool = None, config: ConfigParser = None, section: str = None):
        """
        Create a new vCenter client.

        If `host`, `user`, `password` or `no_ssl_verify` options are not provided, they are read from configuration file
        in section `[vmware-reporter]` (or `[vmware-reporter:{name}]` if `name` is given).

        :param env: An optional name to distinguish between several vCenters.
        :param host: Host name of the vCenter.
        :param user: Name of the vCenter user having access to the API.
        :param password: Password of the vCenter user having access to the API.
        """        
        if not config:
            config = CONFIG
        if not section:
            section = CONFIG_SECTION
        
        if not env:
            envs = VCenterClient.get_configured_envs(config=config, section=section)
            if len(envs) > 1:
                raise MessageError(f"Name of the environment / VCenter to use must be provided. Available: {', '.join(envs) if envs else 'none'}.")
            elif len(envs) == 1:
                env = envs[0]
            else:
                raise MessageError(f"No VCenter client configured.")
        self.env = env or 'default'

        full_section = section + ('' if env == 'default' else f':{env}')
        self.host = host if host is not None else config.get(full_section, 'host')
        self.user = user if user is not None else config.get(full_section, 'user')
        self.password = password if password is not None else config.get(full_section, 'password')
        self.no_ssl_verify = no_ssl_verify if no_ssl_verify is not None else config.getboolean(full_section, 'no_ssl_verify', fallback=False)
        
        self.logger = logging.getLogger(f'{self.__class__.__module__}.{self.__class__.__qualname__}' + ('' if env == 'default' else f'.{self.env}'))


    #region Enter/connect and exit/close

    def __enter__(self):
        self.connect()
        return self


    def __exit__(self, exc_type = None, exc_value = None, exc_traceback = None):
        self.close()


    def connect(self):
        addrs = resolve_host(self.host, timeout=2.0)
        if not addrs:
            raise ValueError(f"Cannot resolve host name \"{self.host}\"")
        
        addr = addrs[0]
        self.logger.debug(f"Connect to {addr} ({self.host}) with user {self.user}")

        options = {}
        if 'httpConnectionTimeout' in signature(SmartConnect).parameters:
            # Introduced in pyVmomi 8.0.0.1 (see https://github.com/vmware/pyvmomi/issues/627)
            options['httpConnectionTimeout'] = 5.0

        self._service_instance = SmartConnect(host=self.host, user=self.user, pwd=self.password, disableSslCertValidation=self.no_ssl_verify, **options)


    def close(self):
        try:
            Disconnect(self._service_instance)
        except AttributeError:
            pass
    

    @property
    def service_instance(self) -> vim.ServiceInstance:
        try:
            return self._service_instance
        except AttributeError:
            pass

        self.connect()
        return self._service_instance


    @property
    def service_content(self) -> vim.ServiceInstanceContent:
        try:
            return self._service_content
        except AttributeError:
            pass

        self._service_content = self.service_instance.RetrieveContent()
        return self._service_content

    #endregion


    #region Retrieve managed objects

    @property
    def datacenter(self):
        try:
            return self._datacenter
        except AttributeError:
            pass

        datacenters = self.get_objs(vim.Datacenter)
        if not datacenters:
            raise ValueError(f"Datacenter not found")
        if len(datacenters) > 1:
            raise ValueError(f"Several datacenter found")
        self._datacenter = datacenters[0]
        return self._datacenter


    def get_obj(self, type: type[T_Obj], search: list[str|re.Pattern]|str|re.Pattern|UUID, *, normalize: bool = False, key: Literal['name', 'ref', 'uuid', 'bios_uuid'] = 'name') -> T_Obj:
        """
        Find a single VMWare managed object.

        Raise KeyError if not found or several found.
        """
        if key in ['uuid', 'bios_uuid']:
            if not isinstance(search, (UUID,str)):
                raise TypeError(f"specs must be UUID or str for key {key}, got {type(search).__name__}")
            
            if isinstance(search, UUID):
                uuid = search
            else:
                uuid = UUID(search)

            obj = None
            
            if key == 'bios_uuid':
                # NOTE: uuid is "BIOS UUID". Seems to match the end of `sudo cat /sys/class/dmi/id/product_uuid`.
                if type == vim.VirtualMachine:
                    obj = self._find_by_uuid(uuid, for_vm=True, instance_uuid=False)
                else:
                    raise ValueError(f"key '{key}' can be used only for virtual machines")
                
            else:
                if type == vim.VirtualMachine:
                    obj = self._find_by_uuid(uuid, for_vm=True, instance_uuid=True)
                elif type == vim.HostSystem:
                    obj = self._find_by_uuid(uuid, for_vm=False, instance_uuid=False)
                else:
                    raise ValueError(f"key '{key}' can be used only for virtual machines or host systems")

            if obj:
                return obj
            else:
                raise KeyError(f"Not found: {search} (type: {type.__name__})")

        else:
            iterator = self.iter_objs(types=type, search=search, normalize=normalize, key=key)
            try:
                found = next(iterator)
            except StopIteration:
                raise KeyError(f"Not found: {search} (type: {type.__name__})")
            
            try:
                next(iterator)
                raise KeyError(f"Several found: {search} (type: {type.__name__})")
            except StopIteration:
                pass
            return found
            

    def _find_by_uuid(self, uuid: UUID|str, for_vm: bool, instance_uuid: bool):
        if isinstance(uuid, UUID):
            uuid = str(uuid)
        
        for datacenter in self.iter_objs(vim.Datacenter):
            obj = self.service_content.searchIndex.FindByUuid(datacenter, uuid, vmSearch=for_vm, instanceUuid=instance_uuid)
            if obj:
                return obj


    @overload
    def get_objs(self, types: type[T_Obj], search: list[str|Path|re.Pattern]|str|Path|re.Pattern = None, *, normalize: bool = None, key: Literal['name', 'ref'] = 'name', sort_key: str|list[str]|Callable = None) -> list[T_Obj]:
        ...

    def get_objs(self, types: list[type|str]|type|str = None, search: list[str|Path|re.Pattern]|str|Path|re.Pattern = None, *, normalize: bool = None, key: Literal['name', 'ref'] = 'name', sort_key: str|list[str]|Callable = None):        
        """
        List VMWare managed objects matching the given search.
        """
        objs = [obj for obj in self.iter_objs(types, search, normalize=normalize, key=key)]

        if sort_key:
            if isinstance(sort_key, str):
                sort_key = [sort_key]

            if isinstance(sort_key, list):
                sort_func = lambda obj: [getattr(obj, attr) for attr in sort_key]
            else:
                sort_func = sort_key

            objs.sort(key=sort_func)

        return objs


    @overload
    def iter_objs(self, types: type[T_Obj], search: list[str|Path|re.Pattern]|str|Path|re.Pattern = None, *, normalize: bool = None, key: Literal['name', 'ref'] = 'name') -> Iterator[T_Obj]:
        ...

    def iter_objs(self, types: list[type|str]|type|str = None, search: list[str|Path|re.Pattern]|str|Path|re.Pattern = None, *, normalize: bool = None, key: Literal['name', 'ref'] = 'name'):
        """
        Iterate over VMWare managed objects matching the given search.
        """
        # Expand search from CSV or Excel files if search ends with '.csv' or '.xlsx'
        search = _expand_search_from_files(search, key=key)

        # Prepare value filter
        filters = Filters(search, normalize=normalize)

        # Prepare types
        if not types:
            types = []
        elif isinstance(types, (str,type)):
            types = [types]
        
        types = [self.parse_obj_type(_type) for _type in types]

        # Search using a container view
        view = None
        try:
            view = self.service_content.viewManager.CreateContainerView(self.service_content.rootFolder, types, recursive=True)

            for obj in view.view:
                if self._obj_matches(obj, key, filters):
                    yield obj
        finally:
            if view:
                view.Destroy()


    def _obj_matches(self, obj: vim.ManagedEntity, key: Literal['name', 'ref'], filters: Filters):
        if not filters:
            return True
        
        if key == 'name':
            try:
                value = obj.name
            except vim.fault.NoPermission:
                return False
            
        elif key == 'ref':
            value = get_obj_ref(obj)
            
        else:
            raise ValueError(f"key not supported: {key}")
        
        return filters.matches(value)

    #endregion


    #region Instance helpers

    @property
    def cookie(self) -> dict:
        try:
            return self._cookie
        except AttributeError:
            pass
    
        # Get the cookie built from the current session
        client_cookie = self.service_instance._stub.cookie

        # Break apart the cookie into it's component parts
        cookie_name = client_cookie.split("=", 1)[0]
        cookie_value = client_cookie.split("=", 1)[1].split(";", 1)[0]
        cookie_path = client_cookie.split("=", 1)[1].split(";", 1)[1].split(
            ";", 1)[0].lstrip()
        cookie_text = " " + cookie_value + "; $" + cookie_path

        # Make a cookie
        self._cookie = dict()
        self._cookie[cookie_name] = cookie_text
        return self._cookie


    def wait_for_task(self, tasks: vim.Task|list[vim.Task]|dict[vim.Task,Any]):
        """
        Given a service instance and tasks, return after all the tasks are complete.
        - `tasks`: a task, a list of tasks, or a dict associating task to log prefixes.
        """
        task_list: list[vim.Task] = []
        log_prefixes: dict[vim.Task,Any] = {}
        if isinstance(tasks, dict):
            log_prefixes = tasks
            for task in tasks.keys():
                if not isinstance(task, vim.Task):
                    raise TypeError(f"Invalid dict key: {task} (type {type(task).__name__}, expected vim.Task)")
                task_list.append(task)
        elif isinstance(tasks, list):
            for task in tasks:
                if not isinstance(task, vim.Task):
                    raise TypeError(f"Invalid list element: {task} (type {type(task).__name__}, expected vim.Task)")
                task_list.append(task)
        elif isinstance(tasks, vim.Task):
            task_list.append(tasks)
        else:
            raise TypeError(f"Invalid argument: {task} (type {type(task).__name__}, expected vim.Task)")
        
        # Create filter
        obj_specs = [vmodl.query.PropertyCollector.ObjectSpec(obj=task) for task in task_list]   
        property_spec = vmodl.query.PropertyCollector.PropertySpec(type=vim.Task, pathSet=[], all=True)
        filter_spec = vmodl.query.PropertyCollector.FilterSpec(objectSet=obj_specs, propSet=[property_spec])
        pc = self.service_instance.content.propertyCollector
        pc_filter = pc.CreateFilter(filter_spec, True)

        task_failures = 0
        try:
            remaining_task_strs = [str(task) for task in task_list]
            version = None
            state = None
            # Loop looking for updates till the state moves to a completed state.
            # (tasks are removed from task_list one by one when they reach a completed)
            while remaining_task_strs:
                update = pc.WaitForUpdates(version)
                for filter_set in update.filterSet:
                    for obj_set in filter_set.objectSet:
                        task: vim.Task = obj_set.obj
                        for change in obj_set.changeSet:
                            if change.name == 'info':
                                state = change.val.state
                            elif change.name == 'info.state':
                                state = change.val
                            else:
                                continue

                            if not str(task) in remaining_task_strs:
                                continue

                            if state == vim.TaskInfo.State.success:
                                remaining_task_strs.remove(str(task))
                                if log_prefix := log_prefixes.get(task):
                                    self.logger.info(f"{log_prefix}: success")
                            elif state == vim.TaskInfo.State.error:
                                if len(task_list) > 1:
                                    remaining_task_strs.remove(str(task))
                                    log_prefix = log_prefixes.get(task)
                                    self.logger.error(f"{f'{log_prefix}: ' if log_prefix else ''}{task.info.error}", exc_info=task.info.error)
                                    task_failures += 1
                                else:
                                    raise task.info.error
                # Move to next version
                version = update.version
        finally:
            if pc_filter:
                pc_filter.Destroy()

        if task_failures > 0:
            raise MessageError(f"{task_failures} task{'s' if task_failures > 1 else ''} failed (see previous logs)")

 
    def get_out_dir(self):
        return Path('data' if self.env == 'default' else f'data/{self.env}')

    #endregion


    #region Class helpers
    
    @classmethod
    def get_configured_envs(cls, *, config: ConfigParser = None, section: str = None):
        if not config:
            config = CONFIG
        if not section:
            section = CONFIG_SECTION
        
        envs: list[str] = []
        for _section in config.sections():
            if m := re.match(r'^' + re.escape(section) + r'(?:\:(.+))?', _section):
                env = m[1]
                if env == 'default':
                    raise ValueError(f"Invalid configuration section \"{_section}\": name \"default\" is reserved")
                if not env:
                    env = 'default'
                envs.append(env)
        return envs
        
    @classmethod
    def parse_obj_type(cls, value: str|type|vim.ManagedEntity) -> type[vim.ManagedEntity]:
        if not value:
            raise ValueError(f"name cannot be blank")
        
        elif isinstance(value, type):
            if not issubclass(value, vim.ManagedEntity):
                raise TypeError(f"type {value} is not a subclass of vim.ManagedEntity")
            
            return value
        
        elif isinstance(value, vim.ManagedEntity):
            return type(value)
        
        elif not isinstance(value, str):
            raise TypeError(f"invalid type for name: {value}")
        
        else:
            lower = value.lower()

            # Search in types
            if lower in cls.OBJ_TYPES:
                return cls.OBJ_TYPES[lower]

            # Handle aliases            
            if lower == 'vm':
                return vim.VirtualMachine
            if lower == 'host':
                return vim.HostSystem
            if lower == 'net':
                return vim.Network
            if lower == 'dvs':
                return vim.DistributedVirtualSwitch
            if lower == 'dvp':
                return vim.dvs.DistributedVirtualPortgroup
            if lower == 'ds':
                return vim.Datastore
            if lower == 'dc':
                return vim.Datacenter
            if lower == 'cluster':
                return vim.ClusterComputeResource

            raise KeyError(f"vim managed object type not found for name {value}")

    def _build_obj_types() -> dict[str,type[vim.ManagedEntity]]:
        types = {}

        for key in _managedDefMap.keys():
            if not key.startswith('vim.'):
                continue

            attr = key[len('vim.'):]
            _type = getattr(vim, attr)
            if not issubclass(_type, vim.ManagedEntity):
                continue

            lower = attr.lower()
            if lower in types:
                continue

            types[lower] = _type

        return types
            
    OBJ_TYPES = _build_obj_types()
    
    #endregion


    #region Retrieve network objects by key
    
    def get_portgroup_by_key(self, key: str) -> vim.dvs.DistributedVirtualPortgroup:
        if key is None:
            return None
        
        try:
            by_key = self._portgroups_by_key
        except AttributeError:
            by_key = {}
            for obj in self.iter_objs(vim.dvs.DistributedVirtualPortgroup):
                by_key[obj.key] = obj
            self._portgroups_by_key = by_key

        return by_key.get(key)
    
    def get_switch_by_uuid(self, uuid: str) -> vim.DistributedVirtualSwitch:
        if uuid is None:
            return None
        
        try:
            by_uuid = self._switchs_by_uuid
        except AttributeError:
            by_uuid = {}
            for obj in self.iter_objs(vim.DistributedVirtualSwitch):
                by_uuid[obj.uuid] = obj
            self._switchs_by_uuid = by_uuid

        return by_uuid.get(uuid)

    #endregion
    

def _expand_search_from_files(search: list[str|Path|re.Pattern]|str|Path|re.Pattern|None, *, key: Literal['name', 'ref'] = 'name') -> list[str|re.Pattern]|None:
    """
    Expand search from CSV or Excel files if search ends with '.csv' or '.xlsx'.
    """
    if search is None:
        return None
    elif isinstance(search, (str,Path,re.Pattern)):
        search = [search]
    
    def expand_from_xlsx_file(path: Path, table_name: str = None):
        workbook = ExcelWorkbook(path)
        table = workbook.get_table(table_name)
        if not key in table.column_names:
            raise ValueError(f"Column \"{key}\" not found in {path}")
        for row in table:
            if row[key]:
                yield row[key]

    def expand_from_csv_file(path: Path):
        for row in iter_dicts_from_csv(path):
            if not key in row:
                raise ValueError(f"Column \"{key}\" not found in {path}")
            if row[key]:
                yield row[key]
    
    result = []
    for elem in search:
        if isinstance(elem, Path):
            if elem.suffix.lower() == '.xlsx':
                for file_elem in expand_from_xlsx_file(elem):
                    result.append(file_elem)
            elif elem.suffix.lower() == '.csv':
                for file_elem in expand_from_csv_file(elem):
                    result.append(file_elem)
            else:
                raise ValueError(f"Invalid search path extension: {elem}")
        elif isinstance(elem, str):
            if is_excel_path(elem, accept_table_suffix=True):
                path, table_name = split_excel_path(elem)
                for file_elem in expand_from_xlsx_file(path, table_name):
                    result.append(file_elem)
            elif elem.lower().endswith('.csv'):
                for file_elem in expand_from_csv_file(Path(elem)):
                    result.append(file_elem)
            else:
                result.append(elem)
        else:
            result.append(elem)

    return result


def get_obj_ref(obj: vim.ManagedObject) -> str:
    """
    Get the value of the Managed Object Reference (MOR) of the given object.

    See: https://vdc-repo.vmware.com/vmwb-repository/dcr-public/1ef6c336-7bef-477d-b9bb-caa1767d7e30/82521f49-9d9a-42b7-b19b-9e6cd9b30db1/mo-types-landing.html
    """
    text = str(obj)
    m = re.match(r"^'(.*)\:(.*)'$", text)
    if not m:
        raise ValueError(f'Invalid object identifier: {text}')
    
    expected_type = type(obj).__name__
    if m.group(1) != expected_type:
        raise ValueError(f'Invalid type for object identifier: {text}, expected: {expected_type}')
    return m.group(2)


def get_obj_path(obj: vim.ManagedEntity, full: bool = False) -> str:
    """ Return the path of the given vim managed entity. """
    if not obj:
        return None
    if isinstance(obj, vim.Datacenter):
        return obj.name
    if not full:
        if obj.parent:
            if isinstance(obj.parent, vim.Datacenter):
                return None            
            super_parent = obj.parent.parent
            if isinstance(super_parent, vim.Datacenter):
                return obj.name
                
    if not obj.parent:
        return obj.name
    elif not full and isinstance(obj, vim.ResourcePool) and obj.name == 'Resources':
        return get_obj_path(obj.parent, full=full)
    else:        
        return get_obj_path(obj.parent, full=full) + "/" + obj.name


def identify_obj(obj: vim.ManagedObject) -> dict:
    if obj is None:
        return None

    if not isinstance(obj, vim.ManagedObject):
        raise ValueError(f'invalid type: {type(obj)}')

    data = {
        "_type": type(obj).__name__, # managed object type
        "ref": get_obj_ref(obj),
    }

    try:
        if name := getattr(obj, 'name', None):
            data["name"] = name            
    except vim.fault.NoPermission:
        data["name"] = '!error:no_permission'

    if 'name' in data and data["name"] == "Resources" and isinstance(obj, vim.ResourcePool) and hasattr(obj, 'parent') and isinstance(obj.parent, vim.ClusterComputeResource):
        # root resource pool of a cluster (named 'Resources'): let's prepend cluster name
        data["name"] = obj.parent.name + "/" + data["name"]

    return data
    

def dictify_value(data: list|str):
    """
    Return a dict if data is a list of OptionValue, StringValue or SystemIdentificationInfo objects, or if it is a string like guestOS.detailed.data.
    Otherwise leave as is.
    """
    def allinstance(enumerable_instance, element_type):
        any = False
        for element in enumerable_instance:
            any = True
            if not isinstance(element, element_type):
                return False
        return any
        
    if isinstance(data, list):
        if allinstance(data, vim.option.OptionValue) or allinstance(data, vim.CustomFieldsManager.StringValue): #example for vm: config.extraConfig, summary.config.customValue
            result = {}
            for ov in data:
                key = ov.key
                value = ov.value
                if key == "guestOS.detailed.data":
                    value = dictify_value(value)
                result[key] = value
            return result

        elif allinstance(data, vim.host.SystemIdentificationInfo): #example for host: summary.hardware.otherIdentifyingInfo
            result = {}
            for ov in data:
                key = ov.identifierType.key
                value = ov.identifierValue
                result[key] = value
            return result

        else:
            return data
    
    if isinstance(data, str):
        if matches := re.findall(r"([a-zA-Z0-9]+)='([^']+)'", data): # example: guestOS.detailed.data
            result = {}
            for m in matches:
                key = m[0]
                value = m[1]
                if key == 'bitness' and re.match(r'^\d+$', value):
                    value = int(value)
                result[key] = value
            return result

        else:
            return data       
        
    else:
        return data


def dictify_obj(obj: vim.ManagedEntity, *, object_types=False, exclude_keys=[], max_depth=None) -> dict:
    """
    Export all available information about the given VMWare managed object to a dictionnary.
    """    
    for key in ['dynamicProperty', 'recentTask']:
        if not key in exclude_keys:
            exclude_keys.append(key)
    exclude_keys_containing = ['capability', 'alarm']
    keypath = []

    def keypath_str():
        s = ''
        for key in keypath:
            s += ('.' if s and not isinstance(key, int) else '') + (f"[{key}]" if isinstance(key, int) else key)
        return s

    def forward(key: str):
        keypath.append(key)

    def backward():
        del keypath[-1]

    def handle_object(obj: object):
        if method := getattr(obj.__class__, 'to_dict', None):
            value = method(obj)
            if value and object_types:
                    return { '_type': type(obj).__name__, **value }
            return value

        result = { '_type': type(obj).__name__ } if object_types else {}
        any = False
        for key in dir(obj):
            ignore = False
            if key.startswith('_') or key in exclude_keys:
                ignore = True
            else:
                for containing in exclude_keys_containing:
                    if containing in key.lower():
                        ignore = True
                        break

            if ignore:
                continue

            forward(key)
            
            try:
                value = getattr(obj, key)
            except: # problem getting the data (e.g. invalid/not-supported accessor)
                _key = keypath_str()
                if _key not in ['configManagerEnabled', 'environmentBrowser']:
                    _logger.error('Cannot read attribute: %s', _key)
                value = "!error:cannot_read"
            
            value = handle_any(value)

            if value is not None:
                result[key] = value
                any = True

            backward()

        if any:
            return result

    def handle_dict(data: dict):
        result = {}
        any = False
        for key in data:
            forward(key)
            value = handle_any(data[key])
            if value is not None:
                result[key] = value
                any = True
            backward()

        if any:
            return result

    def handle_list(data: list):
        result = dictify_value(data)
        if isinstance(result, dict):
            return result

        # general case
        result = []
        any = False
        for i, value in enumerate(data):
            forward(i)
            extracted = handle_any(value)
            if extracted is not None:
                result.append(extracted)
                any = True
            backward()

        if any:
            return result

    def handle_any(data):
        if data is None or isinstance(data, (type, FunctionType, MethodType, BuiltinMethodType, BuiltinFunctionType)):
            return None
        
        elif isinstance(data, (str, int, float, complex)):
            return data

        elif isinstance(data, date):
            if data.year == 1970 and data.month == 1 and data.day == 1:
                return None
            return data

        elif isinstance(data, vim.ManagedObject):
            if not keypath: # depth == 0
                result = identify_obj(data)
                for key, value in handle_object(data).items():
                    result[key] = value
                return result
            else:
                return identify_obj(data)

        elif max_depth and len(keypath) >= max_depth:
            _logger.error('Reached max depth: %s', type(data).__name__)
            return f"!error:max_depth({type(data).__name__})"

        elif isinstance(data, dict):
            return handle_dict(data)

        elif isinstance(data, list):
            return handle_list(data)
            
        else:
            return handle_object(data)

    return handle_any(obj)


def dump_obj(obj: vim.ManagedObject, obj_out: os.PathLike|IOBase, *, title: str = None):
    if not title:
        title = getattr(obj, 'name', None)
        if not title:
            title = type(obj).__name__

    if isinstance(obj_out, IOBase):
        out_name = getattr(obj_out, 'name', '<io>')
    else:
        out_name = str(obj_out)

    data = dictify_obj(obj)

    _logger.info(f"Export {title} to {out_name}")
    with nullcontext(obj_out) if isinstance(obj_out, IOBase) else open(obj_out, 'w', encoding='utf-8') as fp:
        json.dump(data, fp=fp, indent=4, cls=ExtendedJSONEncoder, ensure_ascii=False)


# For docs
__all__ = (
    '__prog__', '__version__', '__version_tuple__',
    'VCenterClient',
    'get_obj_ref', 'get_obj_path', 'identify_obj', 'dictify_value', 'dictify_obj', 'dump_obj',
)
