import pandas as pd
import datetime
import re
import os
import matplotlib.pyplot as plt

from subseasonal_data.data_loaders import load_combined_data
from subseasonal_toolkit.models.localboosting.attributes import get_submodel_name
from subseasonal_toolkit.utils.eval_util import get_target_dates
from subseasonal_toolkit.utils.general_util import dt_to_string
from subseasonal_toolkit.utils.models_util import get_forecast_filename


def get_best_features(gt_id, horizon, n_features, feature_list="localboosting"):
    """Returns the n best features from a selected txt file.
    Args:
        n_features (int): Number of best features to select.
        feature_list (str): Identifier for file that contains a list of best features
            per line, best first.

    Returns:
        list: List of all features or "n" best features.

    Raises:
        ValueError: If the name of a list is not valid or n_features greater than
            the number of total features.
    """

    if n_features == "all":
        return None

    else:
        if feature_list == "localboosting":
            features = [
                line.rstrip("\n")
                for line in open(
                    f"models/localboosting/model_analysis/{gt_id}_{horizon}/"
                    "list-feature_importance-predictions_values_change.txt"
                )
            ]
        else:
            raise NotImplementedError("The only model list avaiable is 'localboosting'.")

        short_gt_id = gt_id.split("_")[1]
        best_features = features[: int(n_features)] + [
            "lat",
            "lon",
            "start_date",
            short_gt_id,
        ]
        return best_features


def subset_data(
    data_full,
    gt_id,
    lat,
    lon,
    region_extension,
    horizon,
    target_date,
    n_features,
    margin_of_days,
    fillna_value=-999,
    number_of_validation_years=1,
):
    data = data_full[
        (lat - region_extension <= data_full.lat)
        & (data_full.lat <= lat + region_extension)
        & (lon - region_extension <= data_full.lon)
        & (data_full.lon <= lon + region_extension)
    ]

    dates_in_data = [date for date in data.start_date.dt.date]
    train_dates_to_use = [
        (change_year(target_date, y) + datetime.timedelta(days=d)).date()
        for y in range(1980, target_date.year - number_of_validation_years)
        for d in range(-margin_of_days, margin_of_days + 1)
        # if change_year(target_date, y) + datetime.timedelta(days=d) in dates_in_data
    ]
    # print(dates_in_data)
    # print(train_dates_to_use)
    train_dates_to_use = sorted(
        set(dates_in_data).intersection(set(train_dates_to_use))
    )
    # print(train_dates_to_use)
    val_dates_to_use = [
        (change_year(target_date, y) + datetime.timedelta(days=d)).date()
        for y in range(target_date.year - number_of_validation_years, target_date.year)
        for d in range(-margin_of_days, margin_of_days + 1)
    ]
    val_dates_to_use = sorted(set(dates_in_data).intersection(set(val_dates_to_use)))
    test_dates_to_use = [target_date]
    dates_to_use = train_dates_to_use + val_dates_to_use + test_dates_to_use

    data = data[data.start_date.isin(dates_to_use)]
    data = data.fillna(fillna_value)

    if "tmp2m" in gt_id:
        X = data.drop(
            [
                "tmp2m",
                "tmp2m_sqd",
                "tmp2m_std",
                "tmp2m_anom",
                "tmp2m_anom_inv_std",
                "tmp2m_shift365_clim",
                "climate_region",
            ],
            axis=1,
            errors="ignore",
        )
        y = data[["start_date", "lat", "lon", "tmp2m"]]
    else:
        X = data.drop(
            [
                "precip",
                "precip_anom",
                "precip_anom_inv_std",
                "precip_shift365_clim",
                "climate_region",
            ],
            axis=1,
            errors="ignore",
        )
        y = data[["start_date", "lat", "lon", "precip"]]

    y = y[(y.lat == lat) & (y.lon == lon)]
    X = X.set_index(["start_date", "lat", "lon"]).unstack(["lat", "lon"])
    y = y.set_index(["start_date", "lat", "lon"]).unstack(["lat", "lon"])

    X_train = X.loc[train_dates_to_use]
    y_train = y.loc[train_dates_to_use]
    X_val = X.loc[val_dates_to_use]
    y_val = y.loc[val_dates_to_use]
    X_test = X.loc[test_dates_to_use]
    y_test = y.loc[test_dates_to_use]

    return X_train, y_train, X_val, y_val, X_test, y_test


def change_year(date, year):
    """Replaces year in date with provided argument. Equivalent to
    date.replace(year=year), except it avoids issues with leap years by subtracting
    one day when February 29th is encountered.

    Args:
        date (datetime): Date whose year is to be changed.
        year (int): Year to use in date.

    Returns:
        datetime: date with new year.
    """
    try:
        return date.replace(year=year)
    except ValueError:
        # This happens when target_date is February 29th, so use February 28th
        return (date - datetime.timedelta(days=1)).replace(year=year)


def camel_case_to_snake_case(s):
    """Convert string in CamelCase to snake_case.
    Args:
        s (str): String in CamelCase.
    Returns:
        str: String in snake_case.
    """
    _underscorer1 = re.compile(r"(.)([A-Z][a-z]+)")
    _underscorer2 = re.compile("([a-z0-9])([A-Z])")
    subbed = _underscorer1.sub(r"\1_\2", s)
    return _underscorer2.sub(r"\1_\2", subbed).lower()


def get_feature_importance_plot(
    feature_importance, method_name, X_train, output_folder, n_features=10
):
    """Creates feature importance plots.
    Args:
        feature_importance (np.array): Array with feature importance values.
        method_name (str): Name of feature importance method.
        n_features (str): how many features to plot.
    """
    method_name = camel_case_to_snake_case(method_name)
    feature_score = pd.DataFrame(
        list(zip(X_train.dtypes.index, feature_importance)),
        columns=["Feature", "Score"],
    )
    feature_score = feature_score.sort_values(
        by="Score",
        ascending=False,
        inplace=False,
        kind="quicksort",
        na_position="last",
    )
    feature_score = feature_score.iloc[:n_features]

    plt.rcParams["figure.figsize"] = (12, 7)
    ax = feature_score.plot("Feature", "Score", kind="bar", color="c")
    ax.set_title("Feature Importance using {}".format(method_name), fontsize=14)
    ax.set_xlabel(None)
    plt.savefig(
        os.path.join(output_folder,f"feature_importance-{method_name}-{n_features}.png"),
        bbox_inches="tight",
        dpi=150,
    )
    plt.close()


def load_train_data(
    gt_id,
    horizon,
    first_train_date,
    last_train_date,
    number_of_validation_years,
    n_features,
    fillna_value=-999,
):
    features_to_use = get_best_features(gt_id, horizon, n_features)
    data = load_combined_data("all_data", gt_id,
                              horizon, columns=features_to_use)
    data = data[
        (data.start_date > first_train_date) & (data.start_date < last_train_date)
    ]
    data = data.fillna(fillna_value)

    first_validation_date = last_train_date.replace(
        year=last_train_date.year - number_of_validation_years
    ) + datetime.timedelta(days=1)

    if "tmp2m" in gt_id:
        X = data.drop(
            [
                "tmp2m",
                "tmp2m_sqd",
                "tmp2m_std",
                "tmp2m_anom",
                "tmp2m_anom_inv_std",
                "tmp2m_shift365_clim",
                "climate_region",
            ],
            axis=1,
            errors="ignore",
        )
        y = data[["start_date", "lat", "lon", "tmp2m"]]
    else:
        X = data.drop(
            [
                "precip",
                "precip_anom",
                "precip_anom_inv_std",
                "precip_shift365_clim",
                "climate_region",
            ],
            axis=1,
            errors="ignore",
        )
        y = data[["start_date", "lat", "lon", "precip"]]

    X_train = X[X.start_date < first_validation_date].drop(
        ["lat", "lon", "start_date"], axis=1
    )
    y_train = y[y.start_date < first_validation_date].drop(
        ["lat", "lon", "start_date"], axis=1
    )
    X_val = X[X.start_date >= first_validation_date].drop(
        ["lat", "lon", "start_date"], axis=1
    )
    y_val = y[y.start_date >= first_validation_date].drop(
        ["lat", "lon", "start_date"], axis=1
    )

    return X_train, y_train, X_val, y_val


def check_forecasts_exist(gt_id, horizon, region_extension=3, target_dates_str="std_paper_eval", n_features=10, margin_of_days=56, iterations=50, depth=2, learning_rate=0.17):

    model_name = "localboosting"
    target_dates = get_target_dates(date_str=target_dates_str)
    submodel_name = get_submodel_name(
        region_extension, n_features, margin_of_days, iterations, depth, learning_rate,
    )
    for target_date in target_dates:
        target_date_str = dt_to_string(target_date)
        forecast_file = get_forecast_filename(
            model=model_name,
            submodel=submodel_name,
            gt_id=gt_id,
            horizon=horizon,
            target_date_str=target_date_str,
        )
        if not os.path.isfile(forecast_file):
            print(f"File not found: {forecast_file}")


CONTEST_LATLON = [
    (27.0, 261.0),
    (27.0, 262.0),
    (28.0, 261.0),
    (28.0, 262.0),
    (28.0, 263.0),
    (29.0, 260.0),
    (29.0, 261.0),
    (29.0, 262.0),
    (29.0, 263.0),
    (29.0, 264.0),
    (30.0, 256.0),
    (30.0, 257.0),
    (30.0, 258.0),
    (30.0, 259.0),
    (30.0, 260.0),
    (30.0, 261.0),
    (30.0, 262.0),
    (30.0, 263.0),
    (30.0, 264.0),
    (30.0, 265.0),
    (30.0, 266.0),
    (31.0, 255.0),
    (31.0, 256.0),
    (31.0, 257.0),
    (31.0, 258.0),
    (31.0, 259.0),
    (31.0, 260.0),
    (31.0, 261.0),
    (31.0, 262.0),
    (31.0, 263.0),
    (31.0, 264.0),
    (31.0, 265.0),
    (31.0, 266.0),
    (32.0, 247.0),
    (32.0, 248.0),
    (32.0, 249.0),
    (32.0, 250.0),
    (32.0, 251.0),
    (32.0, 252.0),
    (32.0, 253.0),
    (32.0, 254.0),
    (32.0, 255.0),
    (32.0, 256.0),
    (32.0, 257.0),
    (32.0, 258.0),
    (32.0, 259.0),
    (32.0, 260.0),
    (32.0, 261.0),
    (32.0, 262.0),
    (32.0, 263.0),
    (32.0, 264.0),
    (32.0, 265.0),
    (33.0, 243.0),
    (33.0, 244.0),
    (33.0, 245.0),
    (33.0, 246.0),
    (33.0, 247.0),
    (33.0, 248.0),
    (33.0, 249.0),
    (33.0, 250.0),
    (33.0, 251.0),
    (33.0, 252.0),
    (33.0, 253.0),
    (33.0, 254.0),
    (33.0, 255.0),
    (33.0, 256.0),
    (33.0, 257.0),
    (33.0, 258.0),
    (33.0, 259.0),
    (33.0, 260.0),
    (33.0, 261.0),
    (33.0, 262.0),
    (33.0, 263.0),
    (33.0, 264.0),
    (33.0, 265.0),
    (34.0, 242.0),
    (34.0, 243.0),
    (34.0, 244.0),
    (34.0, 245.0),
    (34.0, 246.0),
    (34.0, 247.0),
    (34.0, 248.0),
    (34.0, 249.0),
    (34.0, 250.0),
    (34.0, 251.0),
    (34.0, 252.0),
    (34.0, 253.0),
    (34.0, 254.0),
    (34.0, 255.0),
    (34.0, 256.0),
    (34.0, 257.0),
    (34.0, 258.0),
    (34.0, 259.0),
    (34.0, 260.0),
    (34.0, 261.0),
    (34.0, 262.0),
    (34.0, 263.0),
    (34.0, 264.0),
    (34.0, 265.0),
    (35.0, 240.0),
    (35.0, 241.0),
    (35.0, 242.0),
    (35.0, 243.0),
    (35.0, 244.0),
    (35.0, 245.0),
    (35.0, 246.0),
    (35.0, 247.0),
    (35.0, 248.0),
    (35.0, 249.0),
    (35.0, 250.0),
    (35.0, 251.0),
    (35.0, 252.0),
    (35.0, 253.0),
    (35.0, 254.0),
    (35.0, 255.0),
    (35.0, 256.0),
    (35.0, 257.0),
    (35.0, 258.0),
    (35.0, 259.0),
    (35.0, 260.0),
    (35.0, 261.0),
    (35.0, 262.0),
    (35.0, 263.0),
    (35.0, 264.0),
    (35.0, 265.0),
    (36.0, 239.0),
    (36.0, 240.0),
    (36.0, 241.0),
    (36.0, 242.0),
    (36.0, 243.0),
    (36.0, 244.0),
    (36.0, 245.0),
    (36.0, 246.0),
    (36.0, 247.0),
    (36.0, 248.0),
    (36.0, 249.0),
    (36.0, 250.0),
    (36.0, 251.0),
    (36.0, 252.0),
    (36.0, 253.0),
    (36.0, 254.0),
    (36.0, 255.0),
    (36.0, 256.0),
    (36.0, 257.0),
    (36.0, 258.0),
    (36.0, 259.0),
    (36.0, 260.0),
    (36.0, 261.0),
    (36.0, 262.0),
    (36.0, 263.0),
    (36.0, 264.0),
    (36.0, 265.0),
    (37.0, 238.0),
    (37.0, 239.0),
    (37.0, 240.0),
    (37.0, 241.0),
    (37.0, 242.0),
    (37.0, 243.0),
    (37.0, 244.0),
    (37.0, 245.0),
    (37.0, 246.0),
    (37.0, 247.0),
    (37.0, 248.0),
    (37.0, 249.0),
    (37.0, 250.0),
    (37.0, 251.0),
    (37.0, 252.0),
    (37.0, 253.0),
    (37.0, 254.0),
    (37.0, 255.0),
    (37.0, 256.0),
    (37.0, 257.0),
    (37.0, 258.0),
    (37.0, 259.0),
    (37.0, 260.0),
    (37.0, 261.0),
    (37.0, 262.0),
    (37.0, 263.0),
    (37.0, 264.0),
    (37.0, 265.0),
    (38.0, 238.0),
    (38.0, 239.0),
    (38.0, 240.0),
    (38.0, 241.0),
    (38.0, 242.0),
    (38.0, 243.0),
    (38.0, 244.0),
    (38.0, 245.0),
    (38.0, 246.0),
    (38.0, 247.0),
    (38.0, 248.0),
    (38.0, 249.0),
    (38.0, 250.0),
    (38.0, 251.0),
    (38.0, 252.0),
    (38.0, 253.0),
    (38.0, 254.0),
    (38.0, 255.0),
    (38.0, 256.0),
    (38.0, 257.0),
    (38.0, 258.0),
    (38.0, 259.0),
    (38.0, 260.0),
    (38.0, 261.0),
    (38.0, 262.0),
    (38.0, 263.0),
    (38.0, 264.0),
    (38.0, 265.0),
    (39.0, 237.0),
    (39.0, 238.0),
    (39.0, 239.0),
    (39.0, 240.0),
    (39.0, 241.0),
    (39.0, 242.0),
    (39.0, 243.0),
    (39.0, 244.0),
    (39.0, 245.0),
    (39.0, 246.0),
    (39.0, 247.0),
    (39.0, 248.0),
    (39.0, 249.0),
    (39.0, 250.0),
    (39.0, 251.0),
    (39.0, 252.0),
    (39.0, 253.0),
    (39.0, 254.0),
    (39.0, 255.0),
    (39.0, 256.0),
    (39.0, 257.0),
    (39.0, 258.0),
    (39.0, 259.0),
    (39.0, 260.0),
    (39.0, 261.0),
    (39.0, 262.0),
    (39.0, 263.0),
    (39.0, 264.0),
    (39.0, 265.0),
    (40.0, 236.0),
    (40.0, 237.0),
    (40.0, 238.0),
    (40.0, 239.0),
    (40.0, 240.0),
    (40.0, 241.0),
    (40.0, 242.0),
    (40.0, 243.0),
    (40.0, 244.0),
    (40.0, 245.0),
    (40.0, 246.0),
    (40.0, 247.0),
    (40.0, 248.0),
    (40.0, 249.0),
    (40.0, 250.0),
    (40.0, 251.0),
    (40.0, 252.0),
    (40.0, 253.0),
    (40.0, 254.0),
    (40.0, 255.0),
    (40.0, 256.0),
    (40.0, 257.0),
    (40.0, 258.0),
    (40.0, 259.0),
    (40.0, 260.0),
    (40.0, 261.0),
    (40.0, 262.0),
    (40.0, 263.0),
    (40.0, 264.0),
    (41.0, 236.0),
    (41.0, 237.0),
    (41.0, 238.0),
    (41.0, 239.0),
    (41.0, 240.0),
    (41.0, 241.0),
    (41.0, 242.0),
    (41.0, 243.0),
    (41.0, 244.0),
    (41.0, 245.0),
    (41.0, 246.0),
    (41.0, 247.0),
    (41.0, 248.0),
    (41.0, 249.0),
    (41.0, 250.0),
    (41.0, 251.0),
    (41.0, 252.0),
    (41.0, 253.0),
    (41.0, 254.0),
    (41.0, 255.0),
    (41.0, 256.0),
    (41.0, 257.0),
    (41.0, 258.0),
    (41.0, 259.0),
    (41.0, 260.0),
    (41.0, 261.0),
    (41.0, 262.0),
    (41.0, 263.0),
    (41.0, 264.0),
    (42.0, 236.0),
    (42.0, 237.0),
    (42.0, 238.0),
    (42.0, 239.0),
    (42.0, 240.0),
    (42.0, 241.0),
    (42.0, 242.0),
    (42.0, 243.0),
    (42.0, 244.0),
    (42.0, 245.0),
    (42.0, 246.0),
    (42.0, 247.0),
    (42.0, 248.0),
    (42.0, 249.0),
    (42.0, 250.0),
    (42.0, 251.0),
    (42.0, 252.0),
    (42.0, 253.0),
    (42.0, 254.0),
    (42.0, 255.0),
    (42.0, 256.0),
    (42.0, 257.0),
    (42.0, 258.0),
    (42.0, 259.0),
    (42.0, 260.0),
    (42.0, 261.0),
    (42.0, 262.0),
    (42.0, 263.0),
    (43.0, 236.0),
    (43.0, 237.0),
    (43.0, 238.0),
    (43.0, 239.0),
    (43.0, 240.0),
    (43.0, 241.0),
    (43.0, 242.0),
    (43.0, 243.0),
    (43.0, 244.0),
    (43.0, 245.0),
    (43.0, 246.0),
    (43.0, 247.0),
    (43.0, 248.0),
    (43.0, 249.0),
    (43.0, 250.0),
    (43.0, 251.0),
    (43.0, 252.0),
    (43.0, 253.0),
    (43.0, 254.0),
    (43.0, 255.0),
    (43.0, 256.0),
    (43.0, 257.0),
    (43.0, 258.0),
    (43.0, 259.0),
    (43.0, 260.0),
    (43.0, 261.0),
    (43.0, 262.0),
    (43.0, 263.0),
    (44.0, 236.0),
    (44.0, 237.0),
    (44.0, 238.0),
    (44.0, 239.0),
    (44.0, 240.0),
    (44.0, 241.0),
    (44.0, 242.0),
    (44.0, 243.0),
    (44.0, 244.0),
    (44.0, 245.0),
    (44.0, 246.0),
    (44.0, 247.0),
    (44.0, 248.0),
    (44.0, 249.0),
    (44.0, 250.0),
    (44.0, 251.0),
    (44.0, 252.0),
    (44.0, 253.0),
    (44.0, 254.0),
    (44.0, 255.0),
    (44.0, 256.0),
    (44.0, 257.0),
    (44.0, 258.0),
    (44.0, 259.0),
    (44.0, 260.0),
    (44.0, 261.0),
    (44.0, 262.0),
    (44.0, 263.0),
    (45.0, 236.0),
    (45.0, 237.0),
    (45.0, 238.0),
    (45.0, 239.0),
    (45.0, 240.0),
    (45.0, 241.0),
    (45.0, 242.0),
    (45.0, 243.0),
    (45.0, 244.0),
    (45.0, 245.0),
    (45.0, 246.0),
    (45.0, 247.0),
    (45.0, 248.0),
    (45.0, 249.0),
    (45.0, 250.0),
    (45.0, 251.0),
    (45.0, 252.0),
    (45.0, 253.0),
    (45.0, 254.0),
    (45.0, 255.0),
    (45.0, 256.0),
    (45.0, 257.0),
    (45.0, 258.0),
    (45.0, 259.0),
    (45.0, 260.0),
    (45.0, 261.0),
    (45.0, 262.0),
    (45.0, 263.0),
    (46.0, 237.0),
    (46.0, 238.0),
    (46.0, 239.0),
    (46.0, 240.0),
    (46.0, 241.0),
    (46.0, 242.0),
    (46.0, 243.0),
    (46.0, 244.0),
    (46.0, 245.0),
    (46.0, 246.0),
    (46.0, 247.0),
    (46.0, 248.0),
    (46.0, 249.0),
    (46.0, 250.0),
    (46.0, 251.0),
    (46.0, 252.0),
    (46.0, 253.0),
    (46.0, 254.0),
    (46.0, 255.0),
    (46.0, 256.0),
    (46.0, 257.0),
    (46.0, 258.0),
    (46.0, 259.0),
    (46.0, 260.0),
    (46.0, 261.0),
    (46.0, 262.0),
    (46.0, 263.0),
    (47.0, 236.0),
    (47.0, 237.0),
    (47.0, 238.0),
    (47.0, 239.0),
    (47.0, 240.0),
    (47.0, 241.0),
    (47.0, 242.0),
    (47.0, 243.0),
    (47.0, 244.0),
    (47.0, 245.0),
    (47.0, 246.0),
    (47.0, 247.0),
    (47.0, 248.0),
    (47.0, 249.0),
    (47.0, 250.0),
    (47.0, 251.0),
    (47.0, 252.0),
    (47.0, 253.0),
    (47.0, 254.0),
    (47.0, 255.0),
    (47.0, 256.0),
    (47.0, 257.0),
    (47.0, 258.0),
    (47.0, 259.0),
    (47.0, 260.0),
    (47.0, 261.0),
    (47.0, 262.0),
    (47.0, 263.0),
    (48.0, 236.0),
    (48.0, 237.0),
    (48.0, 238.0),
    (48.0, 239.0),
    (48.0, 240.0),
    (48.0, 241.0),
    (48.0, 242.0),
    (48.0, 243.0),
    (48.0, 244.0),
    (48.0, 245.0),
    (48.0, 246.0),
    (48.0, 247.0),
    (48.0, 248.0),
    (48.0, 249.0),
    (48.0, 250.0),
    (48.0, 251.0),
    (48.0, 252.0),
    (48.0, 253.0),
    (48.0, 254.0),
    (48.0, 255.0),
    (48.0, 256.0),
    (48.0, 257.0),
    (48.0, 258.0),
    (48.0, 259.0),
    (48.0, 260.0),
    (48.0, 261.0),
    (48.0, 262.0),
    (49.0, 238.0),
    (49.0, 239.0),
    (49.0, 240.0),
    (49.0, 241.0),
    (49.0, 242.0),
    (49.0, 243.0),
    (49.0, 244.0),
    (49.0, 245.0),
    (49.0, 246.0),
    (49.0, 247.0),
    (49.0, 248.0),
    (49.0, 249.0),
    (49.0, 250.0),
    (49.0, 251.0),
    (49.0, 252.0),
    (49.0, 253.0),
    (49.0, 254.0),
    (49.0, 255.0),
    (49.0, 256.0),
    (49.0, 257.0),
    (49.0, 258.0),
    (49.0, 259.0),
    (49.0, 260.0),
    (49.0, 261.0),
    (49.0, 262.0),
]

US_LATLON = [
    (26.0, 279.0),
    (26.0, 280.0),
    (27.0, 261.0),
    (27.0, 262.0),
    (27.0, 278.0),
    (27.0, 279.0),
    (27.0, 280.0),
    (28.0, 261.0),
    (28.0, 262.0),
    (28.0, 263.0),
    (28.0, 278.0),
    (28.0, 279.0),
    (29.0, 260.0),
    (29.0, 261.0),
    (29.0, 262.0),
    (29.0, 263.0),
    (29.0, 264.0),
    (29.0, 278.0),
    (29.0, 279.0),
    (30.0, 256.0),
    (30.0, 257.0),
    (30.0, 258.0),
    (30.0, 259.0),
    (30.0, 260.0),
    (30.0, 261.0),
    (30.0, 262.0),
    (30.0, 263.0),
    (30.0, 264.0),
    (30.0, 265.0),
    (30.0, 266.0),
    (30.0, 267.0),
    (30.0, 268.0),
    (30.0, 269.0),
    (30.0, 270.0),
    (30.0, 275.0),
    (30.0, 276.0),
    (30.0, 277.0),
    (30.0, 278.0),
    (31.0, 255.0),
    (31.0, 256.0),
    (31.0, 257.0),
    (31.0, 258.0),
    (31.0, 259.0),
    (31.0, 260.0),
    (31.0, 261.0),
    (31.0, 262.0),
    (31.0, 263.0),
    (31.0, 264.0),
    (31.0, 265.0),
    (31.0, 266.0),
    (31.0, 267.0),
    (31.0, 268.0),
    (31.0, 269.0),
    (31.0, 270.0),
    (31.0, 271.0),
    (31.0, 272.0),
    (31.0, 273.0),
    (31.0, 274.0),
    (31.0, 275.0),
    (31.0, 276.0),
    (31.0, 277.0),
    (31.0, 278.0),
    (32.0, 247.0),
    (32.0, 248.0),
    (32.0, 249.0),
    (32.0, 250.0),
    (32.0, 251.0),
    (32.0, 252.0),
    (32.0, 253.0),
    (32.0, 254.0),
    (32.0, 255.0),
    (32.0, 256.0),
    (32.0, 257.0),
    (32.0, 258.0),
    (32.0, 259.0),
    (32.0, 260.0),
    (32.0, 261.0),
    (32.0, 262.0),
    (32.0, 263.0),
    (32.0, 264.0),
    (32.0, 265.0),
    (32.0, 266.0),
    (32.0, 267.0),
    (32.0, 268.0),
    (32.0, 269.0),
    (32.0, 270.0),
    (32.0, 271.0),
    (32.0, 272.0),
    (32.0, 273.0),
    (32.0, 274.0),
    (32.0, 275.0),
    (32.0, 276.0),
    (32.0, 277.0),
    (32.0, 278.0),
    (32.0, 279.0),
    (33.0, 243.0),
    (33.0, 244.0),
    (33.0, 245.0),
    (33.0, 246.0),
    (33.0, 247.0),
    (33.0, 248.0),
    (33.0, 249.0),
    (33.0, 250.0),
    (33.0, 251.0),
    (33.0, 252.0),
    (33.0, 253.0),
    (33.0, 254.0),
    (33.0, 255.0),
    (33.0, 256.0),
    (33.0, 257.0),
    (33.0, 258.0),
    (33.0, 259.0),
    (33.0, 260.0),
    (33.0, 261.0),
    (33.0, 262.0),
    (33.0, 263.0),
    (33.0, 264.0),
    (33.0, 265.0),
    (33.0, 266.0),
    (33.0, 267.0),
    (33.0, 268.0),
    (33.0, 269.0),
    (33.0, 270.0),
    (33.0, 271.0),
    (33.0, 272.0),
    (33.0, 273.0),
    (33.0, 274.0),
    (33.0, 275.0),
    (33.0, 276.0),
    (33.0, 277.0),
    (33.0, 278.0),
    (33.0, 279.0),
    (33.0, 280.0),
    (34.0, 242.0),
    (34.0, 243.0),
    (34.0, 244.0),
    (34.0, 245.0),
    (34.0, 246.0),
    (34.0, 247.0),
    (34.0, 248.0),
    (34.0, 249.0),
    (34.0, 250.0),
    (34.0, 251.0),
    (34.0, 252.0),
    (34.0, 253.0),
    (34.0, 254.0),
    (34.0, 255.0),
    (34.0, 256.0),
    (34.0, 257.0),
    (34.0, 258.0),
    (34.0, 259.0),
    (34.0, 260.0),
    (34.0, 261.0),
    (34.0, 262.0),
    (34.0, 263.0),
    (34.0, 264.0),
    (34.0, 265.0),
    (34.0, 266.0),
    (34.0, 267.0),
    (34.0, 268.0),
    (34.0, 269.0),
    (34.0, 270.0),
    (34.0, 271.0),
    (34.0, 272.0),
    (34.0, 273.0),
    (34.0, 274.0),
    (34.0, 275.0),
    (34.0, 276.0),
    (34.0, 277.0),
    (34.0, 278.0),
    (34.0, 279.0),
    (34.0, 280.0),
    (34.0, 281.0),
    (34.0, 282.0),
    (35.0, 240.0),
    (35.0, 241.0),
    (35.0, 242.0),
    (35.0, 243.0),
    (35.0, 244.0),
    (35.0, 245.0),
    (35.0, 246.0),
    (35.0, 247.0),
    (35.0, 248.0),
    (35.0, 249.0),
    (35.0, 250.0),
    (35.0, 251.0),
    (35.0, 252.0),
    (35.0, 253.0),
    (35.0, 254.0),
    (35.0, 255.0),
    (35.0, 256.0),
    (35.0, 257.0),
    (35.0, 258.0),
    (35.0, 259.0),
    (35.0, 260.0),
    (35.0, 261.0),
    (35.0, 262.0),
    (35.0, 263.0),
    (35.0, 264.0),
    (35.0, 265.0),
    (35.0, 266.0),
    (35.0, 267.0),
    (35.0, 268.0),
    (35.0, 269.0),
    (35.0, 270.0),
    (35.0, 271.0),
    (35.0, 272.0),
    (35.0, 273.0),
    (35.0, 274.0),
    (35.0, 275.0),
    (35.0, 276.0),
    (35.0, 277.0),
    (35.0, 278.0),
    (35.0, 279.0),
    (35.0, 280.0),
    (35.0, 281.0),
    (35.0, 282.0),
    (35.0, 283.0),
    (36.0, 239.0),
    (36.0, 240.0),
    (36.0, 241.0),
    (36.0, 242.0),
    (36.0, 243.0),
    (36.0, 244.0),
    (36.0, 245.0),
    (36.0, 246.0),
    (36.0, 247.0),
    (36.0, 248.0),
    (36.0, 249.0),
    (36.0, 250.0),
    (36.0, 251.0),
    (36.0, 252.0),
    (36.0, 253.0),
    (36.0, 254.0),
    (36.0, 255.0),
    (36.0, 256.0),
    (36.0, 257.0),
    (36.0, 258.0),
    (36.0, 259.0),
    (36.0, 260.0),
    (36.0, 261.0),
    (36.0, 262.0),
    (36.0, 263.0),
    (36.0, 264.0),
    (36.0, 265.0),
    (36.0, 266.0),
    (36.0, 267.0),
    (36.0, 268.0),
    (36.0, 269.0),
    (36.0, 270.0),
    (36.0, 271.0),
    (36.0, 272.0),
    (36.0, 273.0),
    (36.0, 274.0),
    (36.0, 275.0),
    (36.0, 276.0),
    (36.0, 277.0),
    (36.0, 278.0),
    (36.0, 279.0),
    (36.0, 280.0),
    (36.0, 281.0),
    (36.0, 282.0),
    (36.0, 283.0),
    (36.0, 284.0),
    (37.0, 238.0),
    (37.0, 239.0),
    (37.0, 240.0),
    (37.0, 241.0),
    (37.0, 242.0),
    (37.0, 243.0),
    (37.0, 244.0),
    (37.0, 245.0),
    (37.0, 246.0),
    (37.0, 247.0),
    (37.0, 248.0),
    (37.0, 249.0),
    (37.0, 250.0),
    (37.0, 251.0),
    (37.0, 252.0),
    (37.0, 253.0),
    (37.0, 254.0),
    (37.0, 255.0),
    (37.0, 256.0),
    (37.0, 257.0),
    (37.0, 258.0),
    (37.0, 259.0),
    (37.0, 260.0),
    (37.0, 261.0),
    (37.0, 262.0),
    (37.0, 263.0),
    (37.0, 264.0),
    (37.0, 265.0),
    (37.0, 266.0),
    (37.0, 267.0),
    (37.0, 268.0),
    (37.0, 269.0),
    (37.0, 270.0),
    (37.0, 271.0),
    (37.0, 272.0),
    (37.0, 273.0),
    (37.0, 274.0),
    (37.0, 275.0),
    (37.0, 276.0),
    (37.0, 277.0),
    (37.0, 278.0),
    (37.0, 279.0),
    (37.0, 280.0),
    (37.0, 281.0),
    (37.0, 282.0),
    (37.0, 283.0),
    (38.0, 238.0),
    (38.0, 239.0),
    (38.0, 240.0),
    (38.0, 241.0),
    (38.0, 242.0),
    (38.0, 243.0),
    (38.0, 244.0),
    (38.0, 245.0),
    (38.0, 246.0),
    (38.0, 247.0),
    (38.0, 248.0),
    (38.0, 249.0),
    (38.0, 250.0),
    (38.0, 251.0),
    (38.0, 252.0),
    (38.0, 253.0),
    (38.0, 254.0),
    (38.0, 255.0),
    (38.0, 256.0),
    (38.0, 257.0),
    (38.0, 258.0),
    (38.0, 259.0),
    (38.0, 260.0),
    (38.0, 261.0),
    (38.0, 262.0),
    (38.0, 263.0),
    (38.0, 264.0),
    (38.0, 265.0),
    (38.0, 266.0),
    (38.0, 267.0),
    (38.0, 268.0),
    (38.0, 269.0),
    (38.0, 270.0),
    (38.0, 271.0),
    (38.0, 272.0),
    (38.0, 273.0),
    (38.0, 274.0),
    (38.0, 275.0),
    (38.0, 276.0),
    (38.0, 277.0),
    (38.0, 278.0),
    (38.0, 279.0),
    (38.0, 280.0),
    (38.0, 281.0),
    (38.0, 282.0),
    (38.0, 283.0),
    (38.0, 284.0),
    (39.0, 237.0),
    (39.0, 238.0),
    (39.0, 239.0),
    (39.0, 240.0),
    (39.0, 241.0),
    (39.0, 242.0),
    (39.0, 243.0),
    (39.0, 244.0),
    (39.0, 245.0),
    (39.0, 246.0),
    (39.0, 247.0),
    (39.0, 248.0),
    (39.0, 249.0),
    (39.0, 250.0),
    (39.0, 251.0),
    (39.0, 252.0),
    (39.0, 253.0),
    (39.0, 254.0),
    (39.0, 255.0),
    (39.0, 256.0),
    (39.0, 257.0),
    (39.0, 258.0),
    (39.0, 259.0),
    (39.0, 260.0),
    (39.0, 261.0),
    (39.0, 262.0),
    (39.0, 263.0),
    (39.0, 264.0),
    (39.0, 265.0),
    (39.0, 266.0),
    (39.0, 267.0),
    (39.0, 268.0),
    (39.0, 269.0),
    (39.0, 270.0),
    (39.0, 271.0),
    (39.0, 272.0),
    (39.0, 273.0),
    (39.0, 274.0),
    (39.0, 275.0),
    (39.0, 276.0),
    (39.0, 277.0),
    (39.0, 278.0),
    (39.0, 279.0),
    (39.0, 280.0),
    (39.0, 281.0),
    (39.0, 282.0),
    (39.0, 283.0),
    (39.0, 284.0),
    (39.0, 285.0),
    (40.0, 236.0),
    (40.0, 237.0),
    (40.0, 238.0),
    (40.0, 239.0),
    (40.0, 240.0),
    (40.0, 241.0),
    (40.0, 242.0),
    (40.0, 243.0),
    (40.0, 244.0),
    (40.0, 245.0),
    (40.0, 246.0),
    (40.0, 247.0),
    (40.0, 248.0),
    (40.0, 249.0),
    (40.0, 250.0),
    (40.0, 251.0),
    (40.0, 252.0),
    (40.0, 253.0),
    (40.0, 254.0),
    (40.0, 255.0),
    (40.0, 256.0),
    (40.0, 257.0),
    (40.0, 258.0),
    (40.0, 259.0),
    (40.0, 260.0),
    (40.0, 261.0),
    (40.0, 262.0),
    (40.0, 263.0),
    (40.0, 264.0),
    (40.0, 265.0),
    (40.0, 266.0),
    (40.0, 267.0),
    (40.0, 268.0),
    (40.0, 269.0),
    (40.0, 270.0),
    (40.0, 271.0),
    (40.0, 272.0),
    (40.0, 273.0),
    (40.0, 274.0),
    (40.0, 275.0),
    (40.0, 276.0),
    (40.0, 277.0),
    (40.0, 278.0),
    (40.0, 279.0),
    (40.0, 280.0),
    (40.0, 281.0),
    (40.0, 282.0),
    (40.0, 283.0),
    (40.0, 284.0),
    (40.0, 285.0),
    (40.0, 286.0),
    (41.0, 236.0),
    (41.0, 237.0),
    (41.0, 238.0),
    (41.0, 239.0),
    (41.0, 240.0),
    (41.0, 241.0),
    (41.0, 242.0),
    (41.0, 243.0),
    (41.0, 244.0),
    (41.0, 245.0),
    (41.0, 246.0),
    (41.0, 247.0),
    (41.0, 248.0),
    (41.0, 249.0),
    (41.0, 250.0),
    (41.0, 251.0),
    (41.0, 252.0),
    (41.0, 253.0),
    (41.0, 254.0),
    (41.0, 255.0),
    (41.0, 256.0),
    (41.0, 257.0),
    (41.0, 258.0),
    (41.0, 259.0),
    (41.0, 260.0),
    (41.0, 261.0),
    (41.0, 262.0),
    (41.0, 263.0),
    (41.0, 264.0),
    (41.0, 265.0),
    (41.0, 266.0),
    (41.0, 267.0),
    (41.0, 268.0),
    (41.0, 269.0),
    (41.0, 270.0),
    (41.0, 271.0),
    (41.0, 272.0),
    (41.0, 273.0),
    (41.0, 274.0),
    (41.0, 275.0),
    (41.0, 276.0),
    (41.0, 277.0),
    (41.0, 278.0),
    (41.0, 279.0),
    (41.0, 280.0),
    (41.0, 281.0),
    (41.0, 282.0),
    (41.0, 283.0),
    (41.0, 284.0),
    (41.0, 285.0),
    (41.0, 286.0),
    (41.0, 287.0),
    (42.0, 236.0),
    (42.0, 237.0),
    (42.0, 238.0),
    (42.0, 239.0),
    (42.0, 240.0),
    (42.0, 241.0),
    (42.0, 242.0),
    (42.0, 243.0),
    (42.0, 244.0),
    (42.0, 245.0),
    (42.0, 246.0),
    (42.0, 247.0),
    (42.0, 248.0),
    (42.0, 249.0),
    (42.0, 250.0),
    (42.0, 251.0),
    (42.0, 252.0),
    (42.0, 253.0),
    (42.0, 254.0),
    (42.0, 255.0),
    (42.0, 256.0),
    (42.0, 257.0),
    (42.0, 258.0),
    (42.0, 259.0),
    (42.0, 260.0),
    (42.0, 261.0),
    (42.0, 262.0),
    (42.0, 263.0),
    (42.0, 264.0),
    (42.0, 265.0),
    (42.0, 266.0),
    (42.0, 267.0),
    (42.0, 268.0),
    (42.0, 269.0),
    (42.0, 270.0),
    (42.0, 271.0),
    (42.0, 272.0),
    (42.0, 273.0),
    (42.0, 274.0),
    (42.0, 275.0),
    (42.0, 276.0),
    (42.0, 277.0),
    (42.0, 279.0),
    (42.0, 280.0),
    (42.0, 281.0),
    (42.0, 282.0),
    (42.0, 283.0),
    (42.0, 284.0),
    (42.0, 285.0),
    (42.0, 286.0),
    (42.0, 287.0),
    (42.0, 288.0),
    (42.0, 289.0),
    (43.0, 236.0),
    (43.0, 237.0),
    (43.0, 238.0),
    (43.0, 239.0),
    (43.0, 240.0),
    (43.0, 241.0),
    (43.0, 242.0),
    (43.0, 243.0),
    (43.0, 244.0),
    (43.0, 245.0),
    (43.0, 246.0),
    (43.0, 247.0),
    (43.0, 248.0),
    (43.0, 249.0),
    (43.0, 250.0),
    (43.0, 251.0),
    (43.0, 252.0),
    (43.0, 253.0),
    (43.0, 254.0),
    (43.0, 255.0),
    (43.0, 256.0),
    (43.0, 257.0),
    (43.0, 258.0),
    (43.0, 259.0),
    (43.0, 260.0),
    (43.0, 261.0),
    (43.0, 262.0),
    (43.0, 263.0),
    (43.0, 264.0),
    (43.0, 265.0),
    (43.0, 266.0),
    (43.0, 267.0),
    (43.0, 268.0),
    (43.0, 269.0),
    (43.0, 270.0),
    (43.0, 271.0),
    (43.0, 272.0),
    (43.0, 273.0),
    (43.0, 274.0),
    (43.0, 275.0),
    (43.0, 276.0),
    (43.0, 277.0),
    (43.0, 281.0),
    (43.0, 282.0),
    (43.0, 283.0),
    (43.0, 284.0),
    (43.0, 285.0),
    (43.0, 286.0),
    (43.0, 287.0),
    (43.0, 288.0),
    (43.0, 289.0),
    (44.0, 236.0),
    (44.0, 237.0),
    (44.0, 238.0),
    (44.0, 239.0),
    (44.0, 240.0),
    (44.0, 241.0),
    (44.0, 242.0),
    (44.0, 243.0),
    (44.0, 244.0),
    (44.0, 245.0),
    (44.0, 246.0),
    (44.0, 247.0),
    (44.0, 248.0),
    (44.0, 249.0),
    (44.0, 250.0),
    (44.0, 251.0),
    (44.0, 252.0),
    (44.0, 253.0),
    (44.0, 254.0),
    (44.0, 255.0),
    (44.0, 256.0),
    (44.0, 257.0),
    (44.0, 258.0),
    (44.0, 259.0),
    (44.0, 260.0),
    (44.0, 261.0),
    (44.0, 262.0),
    (44.0, 263.0),
    (44.0, 264.0),
    (44.0, 265.0),
    (44.0, 266.0),
    (44.0, 267.0),
    (44.0, 268.0),
    (44.0, 269.0),
    (44.0, 270.0),
    (44.0, 271.0),
    (44.0, 272.0),
    (44.0, 273.0),
    (44.0, 274.0),
    (44.0, 275.0),
    (44.0, 276.0),
    (44.0, 277.0),
    (44.0, 284.0),
    (44.0, 285.0),
    (44.0, 286.0),
    (44.0, 287.0),
    (44.0, 288.0),
    (44.0, 289.0),
    (44.0, 290.0),
    (44.0, 291.0),
    (45.0, 236.0),
    (45.0, 237.0),
    (45.0, 238.0),
    (45.0, 239.0),
    (45.0, 240.0),
    (45.0, 241.0),
    (45.0, 242.0),
    (45.0, 243.0),
    (45.0, 244.0),
    (45.0, 245.0),
    (45.0, 246.0),
    (45.0, 247.0),
    (45.0, 248.0),
    (45.0, 249.0),
    (45.0, 250.0),
    (45.0, 251.0),
    (45.0, 252.0),
    (45.0, 253.0),
    (45.0, 254.0),
    (45.0, 255.0),
    (45.0, 256.0),
    (45.0, 257.0),
    (45.0, 258.0),
    (45.0, 259.0),
    (45.0, 260.0),
    (45.0, 261.0),
    (45.0, 262.0),
    (45.0, 263.0),
    (45.0, 264.0),
    (45.0, 265.0),
    (45.0, 266.0),
    (45.0, 267.0),
    (45.0, 268.0),
    (45.0, 269.0),
    (45.0, 270.0),
    (45.0, 271.0),
    (45.0, 272.0),
    (45.0, 273.0),
    (45.0, 274.0),
    (45.0, 275.0),
    (45.0, 276.0),
    (45.0, 277.0),
    (45.0, 286.0),
    (45.0, 287.0),
    (45.0, 288.0),
    (45.0, 289.0),
    (45.0, 290.0),
    (45.0, 291.0),
    (45.0, 292.0),
    (46.0, 236.0),
    (46.0, 237.0),
    (46.0, 238.0),
    (46.0, 239.0),
    (46.0, 240.0),
    (46.0, 241.0),
    (46.0, 242.0),
    (46.0, 243.0),
    (46.0, 244.0),
    (46.0, 245.0),
    (46.0, 246.0),
    (46.0, 247.0),
    (46.0, 248.0),
    (46.0, 249.0),
    (46.0, 250.0),
    (46.0, 251.0),
    (46.0, 252.0),
    (46.0, 253.0),
    (46.0, 254.0),
    (46.0, 255.0),
    (46.0, 256.0),
    (46.0, 257.0),
    (46.0, 258.0),
    (46.0, 259.0),
    (46.0, 260.0),
    (46.0, 261.0),
    (46.0, 262.0),
    (46.0, 263.0),
    (46.0, 264.0),
    (46.0, 265.0),
    (46.0, 266.0),
    (46.0, 267.0),
    (46.0, 268.0),
    (46.0, 269.0),
    (46.0, 270.0),
    (46.0, 271.0),
    (46.0, 272.0),
    (46.0, 273.0),
    (46.0, 274.0),
    (46.0, 275.0),
    (46.0, 276.0),
    (46.0, 290.0),
    (46.0, 291.0),
    (46.0, 292.0),
    (47.0, 236.0),
    (47.0, 237.0),
    (47.0, 238.0),
    (47.0, 239.0),
    (47.0, 240.0),
    (47.0, 241.0),
    (47.0, 242.0),
    (47.0, 243.0),
    (47.0, 244.0),
    (47.0, 245.0),
    (47.0, 246.0),
    (47.0, 247.0),
    (47.0, 248.0),
    (47.0, 249.0),
    (47.0, 250.0),
    (47.0, 251.0),
    (47.0, 252.0),
    (47.0, 253.0),
    (47.0, 254.0),
    (47.0, 255.0),
    (47.0, 256.0),
    (47.0, 257.0),
    (47.0, 258.0),
    (47.0, 259.0),
    (47.0, 260.0),
    (47.0, 261.0),
    (47.0, 262.0),
    (47.0, 263.0),
    (47.0, 264.0),
    (47.0, 265.0),
    (47.0, 266.0),
    (47.0, 267.0),
    (47.0, 268.0),
    (47.0, 269.0),
    (47.0, 270.0),
    (47.0, 271.0),
    (47.0, 272.0),
    (47.0, 273.0),
    (47.0, 274.0),
    (47.0, 291.0),
    (47.0, 292.0),
    (48.0, 236.0),
    (48.0, 237.0),
    (48.0, 238.0),
    (48.0, 239.0),
    (48.0, 240.0),
    (48.0, 241.0),
    (48.0, 242.0),
    (48.0, 243.0),
    (48.0, 244.0),
    (48.0, 245.0),
    (48.0, 246.0),
    (48.0, 247.0),
    (48.0, 248.0),
    (48.0, 249.0),
    (48.0, 250.0),
    (48.0, 251.0),
    (48.0, 252.0),
    (48.0, 253.0),
    (48.0, 254.0),
    (48.0, 255.0),
    (48.0, 256.0),
    (48.0, 257.0),
    (48.0, 258.0),
    (48.0, 259.0),
    (48.0, 260.0),
    (48.0, 261.0),
    (48.0, 262.0),
    (48.0, 263.0),
    (48.0, 264.0),
    (48.0, 265.0),
    (48.0, 266.0),
    (48.0, 267.0),
    (48.0, 268.0),
    (48.0, 269.0),
    (48.0, 270.0),
    (48.0, 271.0),
    (48.0, 272.0),
    (49.0, 238.0),
    (49.0, 239.0),
    (49.0, 240.0),
    (49.0, 241.0),
    (49.0, 242.0),
    (49.0, 243.0),
    (49.0, 244.0),
    (49.0, 245.0),
    (49.0, 246.0),
    (49.0, 247.0),
    (49.0, 248.0),
    (49.0, 249.0),
    (49.0, 250.0),
    (49.0, 251.0),
    (49.0, 252.0),
    (49.0, 253.0),
    (49.0, 254.0),
    (49.0, 255.0),
    (49.0, 256.0),
    (49.0, 257.0),
    (49.0, 258.0),
    (49.0, 259.0),
    (49.0, 260.0),
    (49.0, 261.0),
    (49.0, 262.0),
    (49.0, 263.0),
    (49.0, 264.0),
    (49.0, 265.0),
]
