"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const bundling_1 = require("./bundling");
const package_manager_1 = require("./package-manager");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using esbuild
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_lambda_nodejs_NodejsFunctionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, NodejsFunction);
            }
            throw error;
        }
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        const handler = props.handler ?? 'handler';
        const runtime = props.runtime ?? lambda.Runtime.NODEJS_14_X;
        const architecture = props.architecture ?? aws_lambda_1.Architecture.X86_64;
        const depsLockFilePath = findLockFile(props.depsLockFilePath);
        const projectRoot = props.projectRoot ?? path.dirname(depsLockFilePath);
        super(scope, id, {
            ...props,
            runtime,
            code: bundling_1.Bundling.bundle({
                ...props.bundling ?? {},
                entry,
                runtime,
                architecture,
                depsLockFilePath,
                projectRoot,
            }),
            handler: `index.${handler}`,
        });
        // Enable connection reuse for aws-sdk
        if (props.awsSdkConnectionReuse ?? true) {
            this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
        }
    }
}
exports.NodejsFunction = NodejsFunction;
_a = JSII_RTTI_SYMBOL_1;
NodejsFunction[_a] = { fqn: "@aws-cdk/aws-lambda-nodejs.NodejsFunction", version: "1.183.0" };
/**
 * Checks given lock file or searches for a lock file
 */
function findLockFile(depsLockFilePath) {
    if (depsLockFilePath) {
        if (!fs.existsSync(depsLockFilePath)) {
            throw new Error(`Lock file at ${depsLockFilePath} doesn't exist`);
        }
        if (!fs.statSync(depsLockFilePath).isFile()) {
            throw new Error('`depsLockFilePath` should point to a file');
        }
        return path.resolve(depsLockFilePath);
    }
    const lockFiles = util_1.findUpMultiple([
        package_manager_1.LockFile.PNPM,
        package_manager_1.LockFile.YARN,
        package_manager_1.LockFile.NPM,
    ]);
    if (lockFiles.length === 0) {
        throw new Error('Cannot find a package lock file (`pnpm-lock.yaml`, `yarn.lock` or `package-lock.json`). Please specify it with `depsLockFilePath`.');
    }
    if (lockFiles.length > 1) {
        throw new Error(`Multiple package lock files found: ${lockFiles.join(', ')}. Please specify the desired one with \`depsLockFilePath\`.`);
    }
    return lockFiles[0];
}
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 * 4. A .mjs file name as the defining file with id as suffix (defining-file.id.mjs)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?|mjs)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    const mjsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.mjs`);
    if (fs.existsSync(mjsHandlerFile)) {
        return mjsHandlerFile;
    }
    throw new Error(`Cannot find handler file ${tsHandlerFile}, ${jsHandlerFile} or ${mjsHandlerFile}`);
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    let definingIndex;
    const sites = util_1.callsites();
    for (const [index, site] of sites.entries()) {
        if (site.getFunctionName() === 'NodejsFunction') {
            // The next site is the site where the NodejsFunction was created
            definingIndex = index + 1;
            break;
        }
    }
    if (!definingIndex || !sites[definingIndex]) {
        throw new Error('Cannot find defining file.');
    }
    return sites[definingIndex].getFileName();
}
//# sourceMappingURL=data:application/json;base64,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