"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PolicyStatementWithPrincipal = exports.PrincipalType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const _5_effect_1 = require("./5-effect");
var PrincipalType;
(function (PrincipalType) {
    PrincipalType["AWS"] = "AWS";
    PrincipalType["FEDERATED"] = "Federated";
    PrincipalType["CANONICAL_USER"] = "CanonicalUser";
    PrincipalType["SERVICE"] = "Service";
})(PrincipalType = exports.PrincipalType || (exports.PrincipalType = {}));
/**
 * Adds "principal" functionality to the Policy Statement
 */
class PolicyStatementWithPrincipal extends _5_effect_1.PolicyStatementWithEffect {
    constructor() {
        super(...arguments);
        this.useNotPrincipals = false;
        this.principals = {};
    }
    /**
     * Injects principals into the statement.
     *
     * Only relevant for the main package. In CDK mode this only calls super.
     */
    toJSON() {
        // @ts-ignore only available after swapping 1-base
        if (typeof this.addResources == 'function') {
            this.cdkApplyPrincipals();
            return super.toJSON();
        }
        const mode = this.useNotPrincipals ? 'NotPrincipal' : 'Principal';
        const statement = super.toJSON();
        if (this.hasPrincipals()) {
            statement[mode] = this.principals;
        }
        return statement;
    }
    toStatementJson() {
        this.cdkApplyPrincipals();
        // @ts-ignore only available after swapping 1-base
        return super.toStatementJson();
    }
    cdkApplyPrincipals() { }
    /**
     * Switches the statement to use [`notPrincipal`](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_notprincipal.html).
     */
    notPrincipals() {
        this.useNotPrincipals = true;
        return this;
    }
    /**
     * Checks weather a principal was applied to the policy
     */
    hasPrincipals() {
        return Object.keys(this.principals).length > 0;
    }
    /**
     * Adds a principal to the statement
     *
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service**
     * @param principal The principal string
     */
    addPrincipal(prefix, principal) {
        this.skipAutoResource = true;
        if (!(prefix in this.principals)) {
            this.principals[prefix] = [];
        }
        this.principals[prefix].push(principal);
        return this;
    }
    /**
     * Adds any principal to the statement
     *
     * @param arn The ARN of the principal
     * @param prefix One of **AWS**, **Federated**, **CanonicalUser** or **Service** - Default: **AWS**
     */
    for(arn, prefix) {
        return this.addPrincipal(prefix || PrincipalType.AWS, arn);
    }
    /**
     * Adds an account principal to the statement
     *
     * @param account ID of the AWS account
     */
    forAccount(account) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:root`);
    }
    /**
     * Adds a [federated](https://aws.amazon.com/identity/federation/) (web identity) principal to the statement
     *
     * @param provider ID of the AWS account
     */
    forFederated(provider) {
        return this.addPrincipal(PrincipalType.FEDERATED, provider);
    }
    /**
     * Adds a federated [AWS Cognito](https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc_cognito.html) principal to the statement
     */
    forFederatedCognito() {
        return this.forFederated('cognito-identity.amazonaws.com');
    }
    /**
     * Adds a federated [Amazon](https://login.amazon.com/) principal to the statement
     */
    forFederatedAmazon() {
        return this.forFederated('www.amazon.com');
    }
    /**
     * Adds a federated [Facebook](https://developers.facebook.com/docs/facebook-login) principal to the statement
     */
    forFederatedFacebook() {
        return this.forFederated('graph.facebook.com');
    }
    /**
     * Adds a federated [Google](https://developers.google.com/identity/protocols/oauth2/openid-connect) principal to the statement
     */
    forFederatedGoogle() {
        return this.forFederated('accounts.google.com');
    }
    /**
     * Adds a canonical user principal to the statement
     *
     * @param userID The user ID
     *
     * You can [find the canonical user ID](https://docs.aws.amazon.com/general/latest/gr/acct-identifiers.html#FindingCanonicalId) for your AWS account in the AWS Management Console. The canonical user ID for an AWS account is specific to the account. You can retrieve the canonical user ID for your AWS account as either the root user or an IAM user.
     */
    forCanonicalUser(userID) {
        return this.addPrincipal(PrincipalType.CANONICAL_USER, userID);
    }
    /**
     * Adds federated SAML principal to the statement
     *
     * @param account ID of the AWS account
     * @param providerName Name of the SAML provider
     */
    forSaml(account, providerName) {
        return this.forFederated(`arn:aws:iam::${account}:saml-provider/${providerName}`);
    }
    /**
     * Adds an IAM user principal to the statement
     *
     * @param account ID of the AWS account
     * @param user Name of the IAM user
     */
    forUser(account, user) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:user/${user}`);
    }
    /**
     * Adds an IAM role principal to the statement
     *
     * @param account ID of the AWS account
     * @param role Name of the IAM role
     */
    forRole(account, role) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:iam::${account}:role/${role}`);
    }
    /**
     * Adds a specific assumed role session principal to the statement
     *
     * @param account ID of the AWS account
     * @param roleName Name of the IAM role
     * @param sessionName Name of the session. You cannot use a wildcard (`*`) to mean *all sessions*. Principals must always name a specific session
     */
    forAssumedRoleSession(account, roleName, sessionName) {
        return this.addPrincipal(PrincipalType.AWS, `arn:aws:sts::${account}:assumed-role/${roleName}/${sessionName}`);
    }
    /**
     * Adds a service principal to the statement
     *
     * @param service Long version of the service name. Usually in the format: `long_service-name.amazonaws.com`
     *
     * The service principal is defined by the service. To learn the service principal for a service, see the documentation for that service. For some services, see [AWS Services That Work with IAM](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_aws-services-that-work-with-iam.html) and look for the services that have **Yes** in the **Service-Linked Role** column. Choose a **Yes** with a link to view the service-linked role documentation for that service. View the **Service-Linked Role Permissions** section for that service to view the service principal.
     */
    forService(service) {
        return this.addPrincipal(PrincipalType.SERVICE, service);
    }
    /**
     * Grants public asses
     *
     * **EVERYONE IN THE WORLD HAS ACCESS**
     *
     * We strongly recommend that you do not use a wildcard in the Principal element in a role's trust policy unless you otherwise restrict access through a Condition element in the policy. Otherwise, any IAM user in any account in your partition can access the role.
     */
    forPublic() {
        return this.addPrincipal(PrincipalType.AWS, '*');
    }
}
exports.PolicyStatementWithPrincipal = PolicyStatementWithPrincipal;
_a = JSII_RTTI_SYMBOL_1;
PolicyStatementWithPrincipal[_a] = { fqn: "cdk-iam-floyd.PolicyStatementWithPrincipal", version: "0.342.0" };
//# sourceMappingURL=data:application/json;base64,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