"use strict";
const cxapi = require("@aws-cdk/cx-api");
const lib_1 = require("../lib");
function createTree(context) {
    const root = new lib_1.App();
    if (context) {
        Object.keys(context).forEach(key => root.node.setContext(key, context[key]));
    }
    const child1 = new lib_1.Construct(root, 'Child1');
    const child2 = new lib_1.Construct(root, 'Child2');
    const child1_1 = new lib_1.Construct(child1, 'Child11');
    const child1_2 = new lib_1.Construct(child1, 'Child12');
    const child1_1_1 = new lib_1.Construct(child1_1, 'Child111');
    const child2_1 = new lib_1.Construct(child2, 'Child21');
    return {
        root, child1, child2, child1_1, child1_2, child1_1_1, child2_1
    };
}
class MyBeautifulConstruct extends lib_1.Construct {
    constructor(scope, id) {
        super(scope, id);
    }
}
/**
 * Returns a string with a tree representation of this construct and it's children.
 */
function toTreeString(node, depth = 0) {
    let out = '';
    for (let i = 0; i < depth; ++i) {
        out += '  ';
    }
    const name = node.node.id || '';
    out += `${node.constructor.name}${name.length > 0 ? ' [' + name + ']' : ''}\n`;
    for (const child of node.node.children) {
        out += toTreeString(child, depth + 1);
    }
    return out;
}
module.exports = {
    'the "Root" construct is a special construct which can be used as the root of the tree'(test) {
        const root = new lib_1.App();
        test.equal(root.node.id, '', 'if not specified, name of a root construct is an empty string');
        test.ok(!root.node.scope, 'no parent');
        test.equal(root.node.children.length, 0, 'a construct is created without children'); // no children
        test.done();
    },
    'constructs cannot be created with an empty name unless they are root'(test) {
        const root = new lib_1.App();
        test.throws(() => new lib_1.Construct(root, ''));
        test.done();
    },
    'construct.name returns the name of the construct'(test) {
        const t = createTree();
        test.equal(t.child1.node.id, 'Child1');
        test.equal(t.child2.node.id, 'Child2');
        test.equal(t.child1_1.node.id, 'Child11');
        test.equal(t.child1_2.node.id, 'Child12');
        test.equal(t.child1_1_1.node.id, 'Child111');
        test.equal(t.child2_1.node.id, 'Child21');
        test.done();
    },
    'construct id can use any character except the path separator'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'valid');
        new lib_1.Construct(root, 'ValiD');
        new lib_1.Construct(root, 'Va123lid');
        new lib_1.Construct(root, 'v');
        new lib_1.Construct(root, '  invalid');
        new lib_1.Construct(root, 'invalid   ');
        new lib_1.Construct(root, '123invalid');
        new lib_1.Construct(root, 'in valid');
        new lib_1.Construct(root, 'in_Valid');
        new lib_1.Construct(root, 'in-Valid');
        new lib_1.Construct(root, 'in\\Valid');
        new lib_1.Construct(root, 'in.Valid');
        test.done();
    },
    'if construct id contains path seperators, they will be replaced by double-dash'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Boom/Boom/Bam');
        test.deepEqual(c.node.id, 'Boom--Boom--Bam');
        test.done();
    },
    'if "undefined" is forcefully used as an "id", it will be treated as an empty string'(test) {
        const c = new lib_1.Construct(undefined, undefined);
        test.deepEqual(c.node.id, '');
        test.done();
    },
    "dont allow unresolved tokens to be used in construct IDs"(test) {
        // GIVEN
        const root = new lib_1.App();
        const token = lib_1.Lazy.stringValue({ produce: () => 'lazy' });
        // WHEN + THEN
        test.throws(() => new lib_1.Construct(root, `MyID: ${token}`), /Cannot use tokens in construct ID: MyID: \${Token/);
        test.done();
    },
    'construct.uniqueId returns a tree-unique alphanumeric id of this construct'(test) {
        const root = new lib_1.App();
        const child1 = new lib_1.Construct(root, 'This is the first child');
        const child2 = new lib_1.Construct(child1, 'Second level');
        const c1 = new lib_1.Construct(child2, 'My construct');
        const c2 = new lib_1.Construct(child1, 'My construct');
        test.deepEqual(c1.node.path, 'This is the first child/Second level/My construct');
        test.deepEqual(c2.node.path, 'This is the first child/My construct');
        test.deepEqual(c1.node.uniqueId, 'ThisisthefirstchildSecondlevelMyconstruct202131E0');
        test.deepEqual(c2.node.uniqueId, 'ThisisthefirstchildMyconstruct8C288DF9');
        test.done();
    },
    'cannot calculate uniqueId if the construct path is ["Default"]'(test) {
        const root = new lib_1.App();
        const c = new lib_1.Construct(root, 'Default');
        test.throws(() => c.node.uniqueId, /Unable to calculate a unique id for an empty set of components/);
        test.done();
    },
    'construct.getChildren() returns an array of all children'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Child1');
        new lib_1.Construct(root, 'Child2');
        test.equal(child.node.children.length, 0, 'no children');
        test.equal(root.node.children.length, 2, 'two children are expected');
        test.done();
    },
    'construct.findChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.tryFindChild(child.node.id), child, 'findChild(name) can be used to retrieve the child from a parent');
        test.ok(!root.node.tryFindChild('NotFound'), 'findChild(name) returns undefined if the child is not found');
        test.done();
    },
    'construct.getChild(name) can be used to retrieve a child from a parent'(test) {
        const root = new lib_1.App();
        const child = new lib_1.Construct(root, 'Contruct');
        test.strictEqual(root.node.findChild(child.node.id), child, 'getChild(name) can be used to retrieve the child from a parent');
        test.throws(() => {
            root.node.findChild('NotFound');
        }, '', 'getChild(name) returns undefined if the child is not found');
        test.done();
    },
    'construct.toString() and construct.toTreeString() can be used for diagnostics'(test) {
        const t = createTree();
        test.equal(t.root.toString(), '<root>');
        test.equal(t.child1_1_1.toString(), 'Child1/Child11/Child111');
        test.equal(t.child2.toString(), 'Child2');
        test.equal(toTreeString(t.root), 'App\n  Construct [Child1]\n    Construct [Child11]\n      Construct [Child111]\n    Construct [Child12]\n  Construct [Child2]\n    Construct [Child21]\n');
        test.done();
    },
    'construct.getContext(key) can be used to read a value from context defined at the root level'(test) {
        const context = {
            ctx1: 12,
            ctx2: 'hello'
        };
        const t = createTree(context);
        test.equal(t.root.node.tryGetContext('ctx1'), 12);
        test.equal(t.child1_1_1.node.tryGetContext('ctx2'), 'hello');
        test.done();
    },
    'construct.setContext(k,v) sets context at some level and construct.getContext(key) will return the lowermost value defined in the stack'(test) {
        const root = new lib_1.App();
        root.node.setContext('c1', 'root');
        root.node.setContext('c2', 'root');
        const child1 = new lib_1.Construct(root, 'child1');
        child1.node.setContext('c2', 'child1');
        child1.node.setContext('c3', 'child1');
        const child2 = new lib_1.Construct(root, 'child2');
        const child3 = new lib_1.Construct(child1, 'child1child1');
        child3.node.setContext('c1', 'child3');
        child3.node.setContext('c4', 'child3');
        test.equal(root.node.tryGetContext('c1'), 'root');
        test.equal(root.node.tryGetContext('c2'), 'root');
        test.equal(root.node.tryGetContext('c3'), undefined);
        test.equal(child1.node.tryGetContext('c1'), 'root');
        test.equal(child1.node.tryGetContext('c2'), 'child1');
        test.equal(child1.node.tryGetContext('c3'), 'child1');
        test.equal(child2.node.tryGetContext('c1'), 'root');
        test.equal(child2.node.tryGetContext('c2'), 'root');
        test.equal(child2.node.tryGetContext('c3'), undefined);
        test.equal(child3.node.tryGetContext('c1'), 'child3');
        test.equal(child3.node.tryGetContext('c2'), 'child1');
        test.equal(child3.node.tryGetContext('c3'), 'child1');
        test.equal(child3.node.tryGetContext('c4'), 'child3');
        test.done();
    },
    'construct.setContext(key, value) can only be called before adding any children'(test) {
        const root = new lib_1.App();
        new lib_1.Construct(root, 'child1');
        test.throws(() => root.node.setContext('k', 'v'));
        test.done();
    },
    'fails if context key contains unresolved tokens'(test) {
        const root = new lib_1.App();
        test.throws(() => root.node.setContext(`my-${lib_1.Aws.REGION}`, 'foo'), /Invalid context key/);
        test.throws(() => root.node.tryGetContext(lib_1.Aws.REGION), /Invalid context key/);
        test.done();
    },
    'construct.pathParts returns an array of strings of all names from root to node'(test) {
        const tree = createTree();
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'Child1/Child11/Child111');
        test.deepEqual(tree.child2.node.path, 'Child2');
        test.done();
    },
    'if a root construct has a name, it should be included in the path'(test) {
        const tree = createTree({});
        test.deepEqual(tree.root.node.path, '');
        test.deepEqual(tree.child1_1_1.node.path, 'Child1/Child11/Child111');
        test.done();
    },
    'construct can not be created with the name of a sibling'(test) {
        const root = new lib_1.App();
        // WHEN
        new lib_1.Construct(root, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(root, 'SameName');
        }, /There is already a Construct with name 'SameName' in App/);
        // WHEN
        const c0 = new lib_1.Construct(root, 'c0');
        new lib_1.Construct(c0, 'SameName');
        // THEN: They have different paths
        test.throws(() => {
            new lib_1.Construct(c0, 'SameName');
        }, /There is already a Construct with name 'SameName' in Construct \[c0\]/);
        test.done();
    },
    'addMetadata(type, data) can be used to attach metadata to constructs FIND_ME'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        test.deepEqual(con.node.metadata, [], 'starts empty');
        con.node.addMetadata('key', 'value');
        con.node.addMetadata('number', 103);
        con.node.addMetadata('array', [123, 456]);
        test.deepEqual(con.node.metadata[0].type, 'key');
        test.deepEqual(con.node.metadata[0].data, 'value');
        test.deepEqual(con.node.metadata[1].data, 103);
        test.deepEqual(con.node.metadata[2].data, [123, 456]);
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace[0].indexOf('FIND_ME') !== -1, 'First stack line should include this function\s name');
        test.done();
    },
    'addMetadata(type, undefined/null) is ignored'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'Foo');
        con.node.addMetadata('Null', null);
        con.node.addMetadata('Undefined', undefined);
        con.node.addMetadata('True', true);
        con.node.addMetadata('False', false);
        con.node.addMetadata('Empty', '');
        const exists = (key) => con.node.metadata.find(x => x.type === key);
        test.ok(!exists('Null'));
        test.ok(!exists('Undefined'));
        test.ok(exists('True'));
        test.ok(exists('False'));
        test.ok(exists('Empty'));
        test.done();
    },
    'addWarning(message) can be used to add a "WARNING" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addWarning('This construct is deprecated, use the other one instead');
        test.deepEqual(con.node.metadata[0].type, cxapi.WARNING_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'This construct is deprecated, use the other one instead');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addError(message) can be used to add a "ERROR" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addError('Stop!');
        test.deepEqual(con.node.metadata[0].type, cxapi.ERROR_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'Stop!');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'addInfo(message) can be used to add an "INFO" message entry to the construct'(test) {
        const root = new lib_1.App();
        const con = new lib_1.Construct(root, 'MyConstruct');
        con.node.addInfo('Hey there, how do you do?');
        test.deepEqual(con.node.metadata[0].type, cxapi.INFO_METADATA_KEY);
        test.deepEqual(con.node.metadata[0].data, 'Hey there, how do you do?');
        test.ok(con.node.metadata[0].trace && con.node.metadata[0].trace.length > 0);
        test.done();
    },
    'multiple children of the same type, with explicit names are welcome'(test) {
        const root = new lib_1.App();
        new MyBeautifulConstruct(root, 'mbc1');
        new MyBeautifulConstruct(root, 'mbc2');
        new MyBeautifulConstruct(root, 'mbc3');
        new MyBeautifulConstruct(root, 'mbc4');
        test.equal(root.node.children.length, 4);
        test.done();
    },
    // tslint:disable-next-line:max-line-length
    'construct.validate() can be implemented to perform validation, ConstructNode.validate(construct.node) will return all errors from the subtree (DFS)'(test) {
        class MyConstruct extends lib_1.Construct {
            validate() {
                return ['my-error1', 'my-error2'];
            }
        }
        class YourConstruct extends lib_1.Construct {
            validate() {
                return ['your-error1'];
            }
        }
        class TheirConstruct extends lib_1.Construct {
            constructor(scope, id) {
                super(scope, id);
                new YourConstruct(this, 'YourConstruct');
            }
            validate() {
                return ['their-error'];
            }
        }
        class TestStack extends lib_1.App {
            constructor() {
                super();
                new MyConstruct(this, 'MyConstruct');
                new TheirConstruct(this, 'TheirConstruct');
            }
            validate() {
                return ['stack-error'];
            }
        }
        const stack = new TestStack();
        const errors = lib_1.ConstructNode.validate(stack.node).map((v) => ({ path: v.source.node.path, message: v.message }));
        // validate DFS
        test.deepEqual(errors, [
            { path: 'MyConstruct', message: 'my-error1' },
            { path: 'MyConstruct', message: 'my-error2' },
            { path: 'TheirConstruct/YourConstruct', message: 'your-error1' },
            { path: 'TheirConstruct', message: 'their-error' },
            { path: '', message: 'stack-error' }
        ]);
        test.done();
    },
    'construct.lock() protects against adding children anywhere under this construct (direct or indirect)'(test) {
        class LockableConstruct extends lib_1.Construct {
            lockMe() {
                this.node._lock();
            }
            unlockMe() {
                this.node._unlock();
            }
        }
        const stack = new lib_1.App();
        const c0a = new LockableConstruct(stack, 'c0a');
        const c0b = new lib_1.Construct(stack, 'c0b');
        const c1a = new lib_1.Construct(c0a, 'c1a');
        const c1b = new lib_1.Construct(c0a, 'c1b');
        c0a.lockMe();
        // now we should still be able to add children to c0b, but not to c0a or any its children
        new lib_1.Construct(c0b, 'c1a');
        test.throws(() => new lib_1.Construct(c0a, 'fail1'), /Cannot add children to "c0a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1a, 'fail2'), /Cannot add children to "c0a\/c1a" during synthesis/);
        test.throws(() => new lib_1.Construct(c1b, 'fail3'), /Cannot add children to "c0a\/c1b" during synthesis/);
        c0a.unlockMe();
        new lib_1.Construct(c0a, 'c0aZ');
        new lib_1.Construct(c1a, 'c1aZ');
        new lib_1.Construct(c1b, 'c1bZ');
        test.done();
    },
    'findAll returns a list of all children in either DFS or BFS'(test) {
        // GIVEN
        const c1 = new lib_1.Construct(undefined, '1');
        const c2 = new lib_1.Construct(c1, '2');
        new lib_1.Construct(c1, '3');
        new lib_1.Construct(c2, '4');
        new lib_1.Construct(c2, '5');
        // THEN
        test.deepEqual(c1.node.findAll().map(x => x.node.id), c1.node.findAll(lib_1.ConstructOrder.PREORDER).map(x => x.node.id)); // default is PreOrder
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.PREORDER).map(x => x.node.id), ['1', '2', '4', '5', '3']);
        test.deepEqual(c1.node.findAll(lib_1.ConstructOrder.POSTORDER).map(x => x.node.id), ['4', '5', '2', '3', '1']);
        test.done();
    },
    'ancestors returns a list of parents up to root'(test) {
        const { child1_1_1 } = createTree();
        test.deepEqual(child1_1_1.node.scopes.map(x => x.node.id), ['', 'Child1', 'Child11', 'Child111']);
        test.done();
    },
    '"root" returns the root construct'(test) {
        const { child1, child2, child1_1_1, root } = createTree();
        test.ok(child1.node.root === root);
        test.ok(child2.node.root === root);
        test.ok(child1_1_1.node.root === root);
        test.done();
    },
    'defaultChild': {
        'returns the child with id "Resource"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            const defaultChild = new lib_1.Construct(root, 'Resource');
            new lib_1.Construct(root, 'child2');
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'returns the child with id "Default"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            const defaultChild = new lib_1.Construct(root, 'Default');
            new lib_1.Construct(root, 'child2');
            test.same(root.node.defaultChild, defaultChild);
            test.done();
        },
        'returns "undefined" if there is no default'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            new lib_1.Construct(root, 'child2');
            test.equal(root.node.defaultChild, undefined);
            test.done();
        },
        'fails if there are both "Resource" and "Default"'(test) {
            const root = new lib_1.App();
            new lib_1.Construct(root, 'child1');
            new lib_1.Construct(root, 'Default');
            new lib_1.Construct(root, 'child2');
            new lib_1.Construct(root, 'Resource');
            test.throws(() => root.node.defaultChild, /Cannot determine default child for . There is both a child with id "Resource" and id "Default"/);
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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