"use strict";
const lib_1 = require("../lib");
class TaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.STANDARD, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
class AsgTaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.AUTOSCALING_GROUP, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
class MapTaggableResource extends lib_1.CfnResource {
    constructor(scope, id, props) {
        super(scope, id, props);
        const tags = props.properties === undefined ? undefined : props.properties.tags;
        this.tags = new lib_1.TagManager(lib_1.TagType.MAP, 'AWS::Fake::Resource', tags);
    }
    testProperties() {
        return this.cfnProperties;
    }
}
module.exports = {
    'Tag visit all children of the applied node'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.Tag('foo', 'bar'));
        lib_1.ConstructNode.synth(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.deepEqual(map.tags.renderTags(), { foo: 'bar' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'foo', value: 'bar', propagateAtLaunch: true }]);
        test.done();
    },
    'The last aspect applied takes precedence'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.Tag('foo', 'bar'));
        res.node.applyAspect(new lib_1.Tag('foo', 'foobar'));
        res.node.applyAspect(new lib_1.Tag('foo', 'baz'));
        res2.node.applyAspect(new lib_1.Tag('foo', 'good'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'baz' }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'foo', value: 'good' }]);
        test.done();
    },
    'RemoveTag will remove a tag if it exists'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        root.node.applyAspect(new lib_1.Tag('root', 'was here'));
        res.node.applyAspect(new lib_1.Tag('first', 'there is only 1'));
        res.node.applyAspect(new lib_1.RemoveTag('root'));
        res.node.applyAspect(new lib_1.RemoveTag('doesnotexist'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.deepEqual(map.tags.renderTags(), { first: 'there is only 1' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'first', value: 'there is only 1', propagateAtLaunch: true }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.done();
    },
    'add will add a tag and remove will remove a tag if it exists'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const asg = new AsgTaggableResource(res, 'AsgFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const map = new MapTaggableResource(res, 'MapFakeResource', {
            type: 'AWS::Fake::Thing',
        });
        lib_1.Tag.add(root, 'root', 'was here');
        lib_1.Tag.add(res, 'first', 'there is only 1');
        lib_1.Tag.remove(res, 'root');
        lib_1.Tag.remove(res, 'doesnotexist');
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.deepEqual(map.tags.renderTags(), { first: 'there is only 1' });
        test.deepEqual(asg.tags.renderTags(), [{ key: 'first', value: 'there is only 1', propagateAtLaunch: true }]);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'first', value: 'there is only 1' }]);
        test.done();
    },
    'the #visit function is idempotent'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.Tag('foo', 'bar'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), [{ key: 'foo', value: 'bar' }]);
        test.done();
    },
    'removeTag Aspects by default will override child Tag Aspects'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.RemoveTag('key'));
        res2.node.applyAspect(new lib_1.Tag('key', 'value'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), undefined);
        test.deepEqual(res2.tags.renderTags(), undefined);
        test.done();
    },
    'removeTag Aspects with priority 0 will not override child Tag Aspects'(test) {
        const root = new lib_1.Stack();
        const res = new TaggableResource(root, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        const res2 = new TaggableResource(res, 'FakeResource', {
            type: 'AWS::Fake::Thing',
        });
        res.node.applyAspect(new lib_1.RemoveTag('key', { priority: 0 }));
        res2.node.applyAspect(new lib_1.Tag('key', 'value'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(res.tags.renderTags(), undefined);
        test.deepEqual(res2.tags.renderTags(), [{ key: 'key', value: 'value' }]);
        test.done();
    },
    'Aspects are merged with tags created by L1 Constructor'(test) {
        const root = new lib_1.Stack();
        const aspectBranch = new TaggableResource(root, 'FakeBranchA', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'aspects', value: 'overwrite' },
                    { key: 'cfn', value: 'is cool' },
                ],
            },
        });
        const asgResource = new AsgTaggableResource(aspectBranch, 'FakeAsg', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'aspects', value: 'overwrite', propagateAtLaunch: false },
                    { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
                ],
            },
        });
        const mapTaggable = new MapTaggableResource(aspectBranch, 'FakeSam', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: {
                    aspects: 'overwrite',
                    cfn: 'is cool',
                },
            },
        });
        const cfnBranch = new TaggableResource(root, 'FakeBranchB', {
            type: 'AWS::Fake::Thing',
            properties: {
                tags: [
                    { key: 'cfn', value: 'is cool' },
                ],
            },
        });
        aspectBranch.node.applyAspect(new lib_1.Tag('aspects', 'rule'));
        lib_1.ConstructNode.prepare(root.node);
        test.deepEqual(aspectBranch.testProperties().tags, [{ key: 'aspects', value: 'rule' }, { key: 'cfn', value: 'is cool' }]);
        test.deepEqual(asgResource.testProperties().tags, [
            { key: 'aspects', value: 'rule', propagateAtLaunch: true },
            { key: 'cfn', value: 'is cool', propagateAtLaunch: true }
        ]);
        test.deepEqual(mapTaggable.testProperties().tags, {
            aspects: 'rule',
            cfn: 'is cool',
        });
        test.deepEqual(cfnBranch.testProperties().tags, [{ key: 'cfn', value: 'is cool' }]);
        test.done();
    },
    'when invalid tag properties are passed from L1s': {
        'map passed instead of array it raises'(test) {
            const root = new lib_1.Stack();
            test.throws(() => {
                new TaggableResource(root, 'FakeBranchA', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: {
                            cfn: 'is cool',
                            aspects: 'overwrite',
                        },
                    },
                });
            });
            test.throws(() => {
                new AsgTaggableResource(root, 'FakeBranchA', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: {
                            cfn: 'is cool',
                            aspects: 'overwrite',
                            propagateAtLaunch: true,
                        },
                    },
                });
            });
            test.done();
        },
        'if array is passed instead of map it raises'(test) {
            const root = new lib_1.Stack();
            test.throws(() => {
                new MapTaggableResource(root, 'FakeSam', {
                    type: 'AWS::Fake::Thing',
                    properties: {
                        tags: [
                            { key: 'cfn', value: 'is cool', propagateAtLaunch: true },
                            { key: 'aspects', value: 'overwrite' },
                        ],
                    },
                });
            });
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC50YWctYXNwZWN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidGVzdC50YWctYXNwZWN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFDQSxnQ0FBNkg7QUFFN0gsTUFBTSxnQkFBaUIsU0FBUSxpQkFBVztJQUV4QyxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXVCO1FBQy9ELEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3hCLE1BQU0sSUFBSSxHQUFHLEtBQUssQ0FBQyxVQUFVLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDO1FBQ2hGLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxnQkFBVSxDQUFDLGFBQU8sQ0FBQyxRQUFRLEVBQUUscUJBQXFCLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDNUUsQ0FBQztJQUNNLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsYUFBYSxDQUFDO0lBQzVCLENBQUM7Q0FDRjtBQUVELE1BQU0sbUJBQW9CLFNBQVEsaUJBQVc7SUFFM0MsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QjtRQUMvRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN4QixNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQztRQUNoRixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxhQUFPLENBQUMsaUJBQWlCLEVBQUUscUJBQXFCLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDckYsQ0FBQztJQUNNLGNBQWM7UUFDbkIsT0FBTyxJQUFJLENBQUMsYUFBYSxDQUFDO0lBQzVCLENBQUM7Q0FDRjtBQUVELE1BQU0sbUJBQW9CLFNBQVEsaUJBQVc7SUFFM0MsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1QjtRQUMvRCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN4QixNQUFNLElBQUksR0FBRyxLQUFLLENBQUMsVUFBVSxLQUFLLFNBQVMsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQztRQUNoRixJQUFJLENBQUMsSUFBSSxHQUFHLElBQUksZ0JBQVUsQ0FBQyxhQUFPLENBQUMsR0FBRyxFQUFFLHFCQUFxQixFQUFFLElBQUksQ0FBQyxDQUFDO0lBQ3ZFLENBQUM7SUFDTSxjQUFjO1FBQ25CLE9BQU8sSUFBSSxDQUFDLGFBQWEsQ0FBQztJQUM1QixDQUFDO0NBQ0Y7QUFFRCxpQkFBUztJQUNQLDRDQUE0QyxDQUFDLElBQVU7UUFDckQsTUFBTSxJQUFJLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUN6QixNQUFNLEdBQUcsR0FBRyxJQUFJLGdCQUFnQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDckQsSUFBSSxFQUFFLGtCQUFrQjtTQUN6QixDQUFDLENBQUM7UUFDSCxNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFnQixDQUFDLEdBQUcsRUFBRSxjQUFjLEVBQUU7WUFDckQsSUFBSSxFQUFFLGtCQUFrQjtTQUN6QixDQUFDLENBQUM7UUFDSCxNQUFNLEdBQUcsR0FBRyxJQUFJLG1CQUFtQixDQUFDLEdBQUcsRUFBRSxpQkFBaUIsRUFBRTtZQUMxRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUVILE1BQU0sR0FBRyxHQUFHLElBQUksbUJBQW1CLENBQUMsR0FBRyxFQUFFLGlCQUFpQixFQUFFO1lBQzFELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDLENBQUM7UUFFNUMsbUJBQWEsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBRS9CLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3BFLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3JFLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUMsQ0FBQyxDQUFDO1FBQ3BELElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBQyxDQUFDLENBQUMsQ0FBQztRQUM3RixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QsMENBQTBDLENBQUMsSUFBVTtRQUNuRCxNQUFNLElBQUksR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQ3pCLE1BQU0sR0FBRyxHQUFHLElBQUksZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQWdCLENBQUMsR0FBRyxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQzVDLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDO1FBQy9DLEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQzVDLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsQ0FBQyxDQUFDO1FBQzlDLG1CQUFhLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBQyxDQUFDLENBQUMsQ0FBQztRQUNwRSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBQyxDQUFDLENBQUMsQ0FBQztRQUN0RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QsMENBQTBDLENBQUMsSUFBVTtRQUNuRCxNQUFNLElBQUksR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQ3pCLE1BQU0sR0FBRyxHQUFHLElBQUksZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQWdCLENBQUMsR0FBRyxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sR0FBRyxHQUFHLElBQUksbUJBQW1CLENBQUMsR0FBRyxFQUFFLGlCQUFpQixFQUFFO1lBQzFELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxHQUFHLEdBQUcsSUFBSSxtQkFBbUIsQ0FBQyxHQUFHLEVBQUUsaUJBQWlCLEVBQUU7WUFDMUQsSUFBSSxFQUFFLGtCQUFrQjtTQUN6QixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLFNBQUcsQ0FBQyxNQUFNLEVBQUUsVUFBVSxDQUFDLENBQUMsQ0FBQztRQUNuRCxHQUFHLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLFNBQUcsQ0FBQyxPQUFPLEVBQUUsaUJBQWlCLENBQUMsQ0FBQyxDQUFDO1FBQzFELEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksZUFBUyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDNUMsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxlQUFTLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQztRQUNwRCxtQkFBYSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFakMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsRUFBQyxHQUFHLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxpQkFBaUIsRUFBQyxDQUFDLENBQUMsQ0FBQztRQUNsRixJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsRUFBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUMsQ0FBQyxDQUFDO1FBQ2xFLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzNHLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsaUJBQWlCLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDbkYsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUNELDhEQUE4RCxDQUFDLElBQVU7UUFDdkUsTUFBTSxJQUFJLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztRQUN6QixNQUFNLEdBQUcsR0FBRyxJQUFJLGdCQUFnQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDckQsSUFBSSxFQUFFLGtCQUFrQjtTQUN6QixDQUFDLENBQUM7UUFDSCxNQUFNLElBQUksR0FBRyxJQUFJLGdCQUFnQixDQUFDLEdBQUcsRUFBRSxjQUFjLEVBQUU7WUFDckQsSUFBSSxFQUFFLGtCQUFrQjtTQUN6QixDQUFDLENBQUM7UUFDSCxNQUFNLEdBQUcsR0FBRyxJQUFJLG1CQUFtQixDQUFDLEdBQUcsRUFBRSxpQkFBaUIsRUFBRTtZQUMxRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUVILE1BQU0sR0FBRyxHQUFHLElBQUksbUJBQW1CLENBQUMsR0FBRyxFQUFFLGlCQUFpQixFQUFFO1lBQzFELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsU0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ2xDLFNBQUcsQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLE9BQU8sRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1FBQ3pDLFNBQUcsQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLE1BQU0sQ0FBQyxDQUFDO1FBQ3hCLFNBQUcsQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFLGNBQWMsQ0FBQyxDQUFDO1FBQ2hDLG1CQUFhLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUVqQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLGlCQUFpQixFQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ2xGLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxFQUFDLEtBQUssRUFBRSxpQkFBaUIsRUFBQyxDQUFDLENBQUM7UUFDbEUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsRUFBQyxHQUFHLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxpQkFBaUIsRUFBRSxpQkFBaUIsRUFBRSxJQUFJLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDM0csSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsRUFBQyxHQUFHLEVBQUUsT0FBTyxFQUFFLEtBQUssRUFBRSxpQkFBaUIsRUFBQyxDQUFDLENBQUMsQ0FBQztRQUNuRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QsbUNBQW1DLENBQUMsSUFBVTtRQUM1QyxNQUFNLElBQUksR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQ3pCLE1BQU0sR0FBRyxHQUFHLElBQUksZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUVILEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksU0FBRyxDQUFDLEtBQUssRUFBRSxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQzVDLG1CQUFhLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBQyxDQUFDLENBQUMsQ0FBQztRQUNwRSxtQkFBYSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDakMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLENBQUMsRUFBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDcEUsbUJBQWEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3BFLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztJQUNkLENBQUM7SUFDRCw4REFBOEQsQ0FBQyxJQUFVO1FBQ3ZFLE1BQU0sSUFBSSxHQUFHLElBQUksV0FBSyxFQUFFLENBQUM7UUFDekIsTUFBTSxHQUFHLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ3JELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxnQkFBZ0IsQ0FBQyxHQUFHLEVBQUUsY0FBYyxFQUFFO1lBQ3JELElBQUksRUFBRSxrQkFBa0I7U0FDekIsQ0FBQyxDQUFDO1FBQ0gsR0FBRyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxlQUFTLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLFNBQUcsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQztRQUMvQyxtQkFBYSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDakMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNsRCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QsdUVBQXVFLENBQUMsSUFBVTtRQUNoRixNQUFNLElBQUksR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQ3pCLE1BQU0sR0FBRyxHQUFHLElBQUksZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILE1BQU0sSUFBSSxHQUFHLElBQUksZ0JBQWdCLENBQUMsR0FBRyxFQUFFLGNBQWMsRUFBRTtZQUNyRCxJQUFJLEVBQUUsa0JBQWtCO1NBQ3pCLENBQUMsQ0FBQztRQUNILEdBQUcsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksZUFBUyxDQUFDLEtBQUssRUFBRSxFQUFDLFFBQVEsRUFBRSxDQUFDLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDMUQsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxTQUFHLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7UUFDL0MsbUJBQWEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLE9BQU8sRUFBQyxDQUFDLENBQUMsQ0FBQztRQUN2RSxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0Qsd0RBQXdELENBQUMsSUFBVTtRQUNqRSxNQUFNLElBQUksR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1FBQ3pCLE1BQU0sWUFBWSxHQUFHLElBQUksZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtZQUM3RCxJQUFJLEVBQUUsa0JBQWtCO1lBQ3hCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUU7b0JBQ0osRUFBQyxHQUFHLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxXQUFXLEVBQUM7b0JBQ3BDLEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFDO2lCQUMvQjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxXQUFXLEdBQUcsSUFBSSxtQkFBbUIsQ0FBQyxZQUFZLEVBQUUsU0FBUyxFQUFFO1lBQ25FLElBQUksRUFBRSxrQkFBa0I7WUFDeEIsVUFBVSxFQUFFO2dCQUNWLElBQUksRUFBRTtvQkFDSixFQUFDLEdBQUcsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFdBQVcsRUFBRSxpQkFBaUIsRUFBRSxLQUFLLEVBQUM7b0JBQzlELEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBQztpQkFDeEQ7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUNILE1BQU0sV0FBVyxHQUFHLElBQUksbUJBQW1CLENBQUMsWUFBWSxFQUFFLFNBQVMsRUFBRTtZQUNuRSxJQUFJLEVBQUUsa0JBQWtCO1lBQ3hCLFVBQVUsRUFBRTtnQkFDVixJQUFJLEVBQUU7b0JBQ0osT0FBTyxFQUFFLFdBQVc7b0JBQ3BCLEdBQUcsRUFBRSxTQUFTO2lCQUNmO2FBQ0Y7U0FDRixDQUFDLENBQUM7UUFDSCxNQUFNLFNBQVMsR0FBRyxJQUFJLGdCQUFnQixDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7WUFDMUQsSUFBSSxFQUFFLGtCQUFrQjtZQUN4QixVQUFVLEVBQUU7Z0JBQ1YsSUFBSSxFQUFFO29CQUNKLEVBQUMsR0FBRyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsU0FBUyxFQUFDO2lCQUMvQjthQUNGO1NBQ0YsQ0FBQyxDQUFDO1FBQ0gsWUFBWSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxTQUFHLENBQUMsU0FBUyxFQUFFLE1BQU0sQ0FBQyxDQUFDLENBQUM7UUFDMUQsbUJBQWEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLGNBQWMsRUFBRSxDQUFDLElBQUksRUFBRSxDQUFDLEVBQUMsR0FBRyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFDLEVBQUUsRUFBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEgsSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsY0FBYyxFQUFFLENBQUMsSUFBSSxFQUFFO1lBQ2hELEVBQUMsR0FBRyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLGlCQUFpQixFQUFFLElBQUksRUFBQztZQUN4RCxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBRSxpQkFBaUIsRUFBRSxJQUFJLEVBQUM7U0FDeEQsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsY0FBYyxFQUFFLENBQUMsSUFBSSxFQUFFO1lBQ2hELE9BQU8sRUFBRSxNQUFNO1lBQ2YsR0FBRyxFQUFFLFNBQVM7U0FDZixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxJQUFJLEVBQUUsQ0FBQyxFQUFDLEdBQUcsRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLFNBQVMsRUFBQyxDQUFDLENBQUMsQ0FBQztRQUNsRixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7SUFDZCxDQUFDO0lBQ0QsaURBQWlELEVBQUU7UUFDakQsdUNBQXVDLENBQUMsSUFBVTtZQUNoRCxNQUFNLElBQUksR0FBRyxJQUFJLFdBQUssRUFBRSxDQUFDO1lBQ3pCLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLElBQUksZ0JBQWdCLENBQUMsSUFBSSxFQUFFLGFBQWEsRUFBRTtvQkFDeEMsSUFBSSxFQUFFLGtCQUFrQjtvQkFDeEIsVUFBVSxFQUFFO3dCQUNWLElBQUksRUFBRTs0QkFDSixHQUFHLEVBQUUsU0FBUzs0QkFDZCxPQUFPLEVBQUUsV0FBVzt5QkFDckI7cUJBQ0Y7aUJBQ0YsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixJQUFJLG1CQUFtQixDQUFDLElBQUksRUFBRSxhQUFhLEVBQUU7b0JBQzNDLElBQUksRUFBRSxrQkFBa0I7b0JBQ3hCLFVBQVUsRUFBRTt3QkFDVixJQUFJLEVBQUU7NEJBQ0osR0FBRyxFQUFFLFNBQVM7NEJBQ2QsT0FBTyxFQUFFLFdBQVc7NEJBQ3BCLGlCQUFpQixFQUFFLElBQUk7eUJBQ3hCO3FCQUNGO2lCQUNGLENBQUMsQ0FBQztZQUNMLENBQUMsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUNELDZDQUE2QyxDQUFDLElBQVU7WUFDdEQsTUFBTSxJQUFJLEdBQUcsSUFBSSxXQUFLLEVBQUUsQ0FBQztZQUN6QixJQUFJLENBQUMsTUFBTSxDQUFDLEdBQUcsRUFBRTtnQkFDZixJQUFJLG1CQUFtQixDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7b0JBQ3ZDLElBQUksRUFBRSxrQkFBa0I7b0JBQ3hCLFVBQVUsRUFBRTt3QkFDVixJQUFJLEVBQUU7NEJBQ0osRUFBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLEtBQUssRUFBRSxTQUFTLEVBQUUsaUJBQWlCLEVBQUUsSUFBSSxFQUFDOzRCQUN2RCxFQUFDLEdBQUcsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLFdBQVcsRUFBQzt5QkFDckM7cUJBQ0Y7aUJBQ0YsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7WUFDSCxJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO0tBQ0Y7Q0FDRixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCB7IENmblJlc291cmNlLCBDZm5SZXNvdXJjZVByb3BzLCBDb25zdHJ1Y3QsIENvbnN0cnVjdE5vZGUsIFJlbW92ZVRhZywgU3RhY2ssIFRhZywgVGFnTWFuYWdlciwgVGFnVHlwZSB9IGZyb20gJy4uL2xpYic7XG5cbmNsYXNzIFRhZ2dhYmxlUmVzb3VyY2UgZXh0ZW5kcyBDZm5SZXNvdXJjZSB7XG4gIHB1YmxpYyByZWFkb25seSB0YWdzOiBUYWdNYW5hZ2VyO1xuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQ2ZuUmVzb3VyY2VQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgIGNvbnN0IHRhZ3MgPSBwcm9wcy5wcm9wZXJ0aWVzID09PSB1bmRlZmluZWQgPyB1bmRlZmluZWQgOiBwcm9wcy5wcm9wZXJ0aWVzLnRhZ3M7XG4gICAgdGhpcy50YWdzID0gbmV3IFRhZ01hbmFnZXIoVGFnVHlwZS5TVEFOREFSRCwgJ0FXUzo6RmFrZTo6UmVzb3VyY2UnLCB0YWdzKTtcbiAgfVxuICBwdWJsaWMgdGVzdFByb3BlcnRpZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMuY2ZuUHJvcGVydGllcztcbiAgfVxufVxuXG5jbGFzcyBBc2dUYWdnYWJsZVJlc291cmNlIGV4dGVuZHMgQ2ZuUmVzb3VyY2Uge1xuICBwdWJsaWMgcmVhZG9ubHkgdGFnczogVGFnTWFuYWdlcjtcbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENmblJlc291cmNlUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcbiAgICBjb25zdCB0YWdzID0gcHJvcHMucHJvcGVydGllcyA9PT0gdW5kZWZpbmVkID8gdW5kZWZpbmVkIDogcHJvcHMucHJvcGVydGllcy50YWdzO1xuICAgIHRoaXMudGFncyA9IG5ldyBUYWdNYW5hZ2VyKFRhZ1R5cGUuQVVUT1NDQUxJTkdfR1JPVVAsICdBV1M6OkZha2U6OlJlc291cmNlJywgdGFncyk7XG4gIH1cbiAgcHVibGljIHRlc3RQcm9wZXJ0aWVzKCkge1xuICAgIHJldHVybiB0aGlzLmNmblByb3BlcnRpZXM7XG4gIH1cbn1cblxuY2xhc3MgTWFwVGFnZ2FibGVSZXNvdXJjZSBleHRlbmRzIENmblJlc291cmNlIHtcbiAgcHVibGljIHJlYWRvbmx5IHRhZ3M6IFRhZ01hbmFnZXI7XG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDZm5SZXNvdXJjZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG4gICAgY29uc3QgdGFncyA9IHByb3BzLnByb3BlcnRpZXMgPT09IHVuZGVmaW5lZCA/IHVuZGVmaW5lZCA6IHByb3BzLnByb3BlcnRpZXMudGFncztcbiAgICB0aGlzLnRhZ3MgPSBuZXcgVGFnTWFuYWdlcihUYWdUeXBlLk1BUCwgJ0FXUzo6RmFrZTo6UmVzb3VyY2UnLCB0YWdzKTtcbiAgfVxuICBwdWJsaWMgdGVzdFByb3BlcnRpZXMoKSB7XG4gICAgcmV0dXJuIHRoaXMuY2ZuUHJvcGVydGllcztcbiAgfVxufVxuXG5leHBvcnQgPSB7XG4gICdUYWcgdmlzaXQgYWxsIGNoaWxkcmVuIG9mIHRoZSBhcHBsaWVkIG5vZGUnKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByb290ID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgcmVzID0gbmV3IFRhZ2dhYmxlUmVzb3VyY2Uocm9vdCwgJ0Zha2VSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICB9KTtcbiAgICBjb25zdCByZXMyID0gbmV3IFRhZ2dhYmxlUmVzb3VyY2UocmVzLCAnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIGNvbnN0IGFzZyA9IG5ldyBBc2dUYWdnYWJsZVJlc291cmNlKHJlcywgJ0FzZ0Zha2VSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICB9KTtcblxuICAgIGNvbnN0IG1hcCA9IG5ldyBNYXBUYWdnYWJsZVJlc291cmNlKHJlcywgJ01hcEZha2VSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICB9KTtcbiAgICByZXMubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdmb28nLCAnYmFyJykpO1xuXG4gICAgQ29uc3RydWN0Tm9kZS5zeW50aChyb290Lm5vZGUpO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwocmVzLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZvbycsIHZhbHVlOiAnYmFyJ31dKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChyZXMyLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZvbycsIHZhbHVlOiAnYmFyJ31dKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChtYXAudGFncy5yZW5kZXJUYWdzKCksIHtmb286ICdiYXInfSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNnLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZvbycsIHZhbHVlOiAnYmFyJywgcHJvcGFnYXRlQXRMYXVuY2g6IHRydWV9XSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG4gICdUaGUgbGFzdCBhc3BlY3QgYXBwbGllZCB0YWtlcyBwcmVjZWRlbmNlJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHJlcyA9IG5ldyBUYWdnYWJsZVJlc291cmNlKHJvb3QsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgY29uc3QgcmVzMiA9IG5ldyBUYWdnYWJsZVJlc291cmNlKHJlcywgJ0Zha2VSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICB9KTtcbiAgICByZXMubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdmb28nLCAnYmFyJykpO1xuICAgIHJlcy5ub2RlLmFwcGx5QXNwZWN0KG5ldyBUYWcoJ2ZvbycsICdmb29iYXInKSk7XG4gICAgcmVzLm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZygnZm9vJywgJ2JheicpKTtcbiAgICByZXMyLm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZygnZm9vJywgJ2dvb2QnKSk7XG4gICAgQ29uc3RydWN0Tm9kZS5wcmVwYXJlKHJvb3Qubm9kZSk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZvbycsIHZhbHVlOiAnYmF6J31dKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChyZXMyLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZvbycsIHZhbHVlOiAnZ29vZCd9XSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG4gICdSZW1vdmVUYWcgd2lsbCByZW1vdmUgYSB0YWcgaWYgaXQgZXhpc3RzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHJlcyA9IG5ldyBUYWdnYWJsZVJlc291cmNlKHJvb3QsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgY29uc3QgcmVzMiA9IG5ldyBUYWdnYWJsZVJlc291cmNlKHJlcywgJ0Zha2VSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICB9KTtcbiAgICBjb25zdCBhc2cgPSBuZXcgQXNnVGFnZ2FibGVSZXNvdXJjZShyZXMsICdBc2dGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG5cbiAgICBjb25zdCBtYXAgPSBuZXcgTWFwVGFnZ2FibGVSZXNvdXJjZShyZXMsICdNYXBGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgcm9vdC5ub2RlLmFwcGx5QXNwZWN0KG5ldyBUYWcoJ3Jvb3QnLCAnd2FzIGhlcmUnKSk7XG4gICAgcmVzLm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZygnZmlyc3QnLCAndGhlcmUgaXMgb25seSAxJykpO1xuICAgIHJlcy5ub2RlLmFwcGx5QXNwZWN0KG5ldyBSZW1vdmVUYWcoJ3Jvb3QnKSk7XG4gICAgcmVzLm5vZGUuYXBwbHlBc3BlY3QobmV3IFJlbW92ZVRhZygnZG9lc25vdGV4aXN0JykpO1xuICAgIENvbnN0cnVjdE5vZGUucHJlcGFyZShyb290Lm5vZGUpO1xuXG4gICAgdGVzdC5kZWVwRXF1YWwocmVzLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZpcnN0JywgdmFsdWU6ICd0aGVyZSBpcyBvbmx5IDEnfV0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKG1hcC50YWdzLnJlbmRlclRhZ3MoKSwge2ZpcnN0OiAndGhlcmUgaXMgb25seSAxJ30pO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzZy50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdmaXJzdCcsIHZhbHVlOiAndGhlcmUgaXMgb25seSAxJywgcHJvcGFnYXRlQXRMYXVuY2g6IHRydWV9XSk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzMi50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdmaXJzdCcsIHZhbHVlOiAndGhlcmUgaXMgb25seSAxJ31dKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ2FkZCB3aWxsIGFkZCBhIHRhZyBhbmQgcmVtb3ZlIHdpbGwgcmVtb3ZlIGEgdGFnIGlmIGl0IGV4aXN0cycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCByZXMgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyb290LCAnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIGNvbnN0IHJlczIgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyZXMsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgY29uc3QgYXNnID0gbmV3IEFzZ1RhZ2dhYmxlUmVzb3VyY2UocmVzLCAnQXNnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbWFwID0gbmV3IE1hcFRhZ2dhYmxlUmVzb3VyY2UocmVzLCAnTWFwRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIFRhZy5hZGQocm9vdCwgJ3Jvb3QnLCAnd2FzIGhlcmUnKTtcbiAgICBUYWcuYWRkKHJlcywgJ2ZpcnN0JywgJ3RoZXJlIGlzIG9ubHkgMScpO1xuICAgIFRhZy5yZW1vdmUocmVzLCAncm9vdCcpO1xuICAgIFRhZy5yZW1vdmUocmVzLCAnZG9lc25vdGV4aXN0Jyk7XG4gICAgQ29uc3RydWN0Tm9kZS5wcmVwYXJlKHJvb3Qubm9kZSk7XG5cbiAgICB0ZXN0LmRlZXBFcXVhbChyZXMudGFncy5yZW5kZXJUYWdzKCksIFt7a2V5OiAnZmlyc3QnLCB2YWx1ZTogJ3RoZXJlIGlzIG9ubHkgMSd9XSk7XG4gICAgdGVzdC5kZWVwRXF1YWwobWFwLnRhZ3MucmVuZGVyVGFncygpLCB7Zmlyc3Q6ICd0aGVyZSBpcyBvbmx5IDEnfSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoYXNnLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZpcnN0JywgdmFsdWU6ICd0aGVyZSBpcyBvbmx5IDEnLCBwcm9wYWdhdGVBdExhdW5jaDogdHJ1ZX1dKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChyZXMyLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZpcnN0JywgdmFsdWU6ICd0aGVyZSBpcyBvbmx5IDEnfV0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAndGhlICN2aXNpdCBmdW5jdGlvbiBpcyBpZGVtcG90ZW50Jyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHJlcyA9IG5ldyBUYWdnYWJsZVJlc291cmNlKHJvb3QsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG5cbiAgICByZXMubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdmb28nLCAnYmFyJykpO1xuICAgIENvbnN0cnVjdE5vZGUucHJlcGFyZShyb290Lm5vZGUpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHJlcy50YWdzLnJlbmRlclRhZ3MoKSwgW3trZXk6ICdmb28nLCB2YWx1ZTogJ2Jhcid9XSk7XG4gICAgQ29uc3RydWN0Tm9kZS5wcmVwYXJlKHJvb3Qubm9kZSk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzLnRhZ3MucmVuZGVyVGFncygpLCBbe2tleTogJ2ZvbycsIHZhbHVlOiAnYmFyJ31dKTtcbiAgICBDb25zdHJ1Y3ROb2RlLnByZXBhcmUocm9vdC5ub2RlKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChyZXMudGFncy5yZW5kZXJUYWdzKCksIFt7a2V5OiAnZm9vJywgdmFsdWU6ICdiYXInfV0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAncmVtb3ZlVGFnIEFzcGVjdHMgYnkgZGVmYXVsdCB3aWxsIG92ZXJyaWRlIGNoaWxkIFRhZyBBc3BlY3RzJyh0ZXN0OiBUZXN0KSB7XG4gICAgY29uc3Qgcm9vdCA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHJlcyA9IG5ldyBUYWdnYWJsZVJlc291cmNlKHJvb3QsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgY29uc3QgcmVzMiA9IG5ldyBUYWdnYWJsZVJlc291cmNlKHJlcywgJ0Zha2VSZXNvdXJjZScsIHtcbiAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICB9KTtcbiAgICByZXMubm9kZS5hcHBseUFzcGVjdChuZXcgUmVtb3ZlVGFnKCdrZXknKSk7XG4gICAgcmVzMi5ub2RlLmFwcGx5QXNwZWN0KG5ldyBUYWcoJ2tleScsICd2YWx1ZScpKTtcbiAgICBDb25zdHJ1Y3ROb2RlLnByZXBhcmUocm9vdC5ub2RlKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChyZXMudGFncy5yZW5kZXJUYWdzKCksIHVuZGVmaW5lZCk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzMi50YWdzLnJlbmRlclRhZ3MoKSwgdW5kZWZpbmVkKTtcbiAgICB0ZXN0LmRvbmUoKTtcbiAgfSxcbiAgJ3JlbW92ZVRhZyBBc3BlY3RzIHdpdGggcHJpb3JpdHkgMCB3aWxsIG5vdCBvdmVycmlkZSBjaGlsZCBUYWcgQXNwZWN0cycodGVzdDogVGVzdCkge1xuICAgIGNvbnN0IHJvb3QgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCByZXMgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyb290LCAnRmFrZVJlc291cmNlJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgIH0pO1xuICAgIGNvbnN0IHJlczIgPSBuZXcgVGFnZ2FibGVSZXNvdXJjZShyZXMsICdGYWtlUmVzb3VyY2UnLCB7XG4gICAgICB0eXBlOiAnQVdTOjpGYWtlOjpUaGluZycsXG4gICAgfSk7XG4gICAgcmVzLm5vZGUuYXBwbHlBc3BlY3QobmV3IFJlbW92ZVRhZygna2V5Jywge3ByaW9yaXR5OiAwfSkpO1xuICAgIHJlczIubm9kZS5hcHBseUFzcGVjdChuZXcgVGFnKCdrZXknLCAndmFsdWUnKSk7XG4gICAgQ29uc3RydWN0Tm9kZS5wcmVwYXJlKHJvb3Qubm9kZSk7XG4gICAgdGVzdC5kZWVwRXF1YWwocmVzLnRhZ3MucmVuZGVyVGFncygpLCB1bmRlZmluZWQpO1xuICAgIHRlc3QuZGVlcEVxdWFsKHJlczIudGFncy5yZW5kZXJUYWdzKCksIFt7a2V5OiAna2V5JywgdmFsdWU6ICd2YWx1ZSd9XSk7XG4gICAgdGVzdC5kb25lKCk7XG4gIH0sXG4gICdBc3BlY3RzIGFyZSBtZXJnZWQgd2l0aCB0YWdzIGNyZWF0ZWQgYnkgTDEgQ29uc3RydWN0b3InKHRlc3Q6IFRlc3QpIHtcbiAgICBjb25zdCByb290ID0gbmV3IFN0YWNrKCk7XG4gICAgY29uc3QgYXNwZWN0QnJhbmNoID0gbmV3IFRhZ2dhYmxlUmVzb3VyY2Uocm9vdCwgJ0Zha2VCcmFuY2hBJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICB0YWdzOiBbXG4gICAgICAgICAge2tleTogJ2FzcGVjdHMnLCB2YWx1ZTogJ292ZXJ3cml0ZSd9LFxuICAgICAgICAgIHtrZXk6ICdjZm4nLCB2YWx1ZTogJ2lzIGNvb2wnfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3QgYXNnUmVzb3VyY2UgPSBuZXcgQXNnVGFnZ2FibGVSZXNvdXJjZShhc3BlY3RCcmFuY2gsICdGYWtlQXNnJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICB0YWdzOiBbXG4gICAgICAgICAge2tleTogJ2FzcGVjdHMnLCB2YWx1ZTogJ292ZXJ3cml0ZScsIHByb3BhZ2F0ZUF0TGF1bmNoOiBmYWxzZX0sXG4gICAgICAgICAge2tleTogJ2NmbicsIHZhbHVlOiAnaXMgY29vbCcsIHByb3BhZ2F0ZUF0TGF1bmNoOiB0cnVlfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3QgbWFwVGFnZ2FibGUgPSBuZXcgTWFwVGFnZ2FibGVSZXNvdXJjZShhc3BlY3RCcmFuY2gsICdGYWtlU2FtJywge1xuICAgICAgdHlwZTogJ0FXUzo6RmFrZTo6VGhpbmcnLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICB0YWdzOiB7XG4gICAgICAgICAgYXNwZWN0czogJ292ZXJ3cml0ZScsXG4gICAgICAgICAgY2ZuOiAnaXMgY29vbCcsXG4gICAgICAgIH0sXG4gICAgICB9LFxuICAgIH0pO1xuICAgIGNvbnN0IGNmbkJyYW5jaCA9IG5ldyBUYWdnYWJsZVJlc291cmNlKHJvb3QsICdGYWtlQnJhbmNoQicsIHtcbiAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgdGFnczogW1xuICAgICAgICAgIHtrZXk6ICdjZm4nLCB2YWx1ZTogJ2lzIGNvb2wnfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgYXNwZWN0QnJhbmNoLm5vZGUuYXBwbHlBc3BlY3QobmV3IFRhZygnYXNwZWN0cycsICdydWxlJykpO1xuICAgIENvbnN0cnVjdE5vZGUucHJlcGFyZShyb290Lm5vZGUpO1xuICAgIHRlc3QuZGVlcEVxdWFsKGFzcGVjdEJyYW5jaC50ZXN0UHJvcGVydGllcygpLnRhZ3MsIFt7a2V5OiAnYXNwZWN0cycsIHZhbHVlOiAncnVsZSd9LCB7a2V5OiAnY2ZuJywgdmFsdWU6ICdpcyBjb29sJ31dKTtcbiAgICB0ZXN0LmRlZXBFcXVhbChhc2dSZXNvdXJjZS50ZXN0UHJvcGVydGllcygpLnRhZ3MsIFtcbiAgICAgIHtrZXk6ICdhc3BlY3RzJywgdmFsdWU6ICdydWxlJywgcHJvcGFnYXRlQXRMYXVuY2g6IHRydWV9LFxuICAgICAge2tleTogJ2NmbicsIHZhbHVlOiAnaXMgY29vbCcsIHByb3BhZ2F0ZUF0TGF1bmNoOiB0cnVlfVxuICAgIF0pO1xuICAgIHRlc3QuZGVlcEVxdWFsKG1hcFRhZ2dhYmxlLnRlc3RQcm9wZXJ0aWVzKCkudGFncywge1xuICAgICAgYXNwZWN0czogJ3J1bGUnLFxuICAgICAgY2ZuOiAnaXMgY29vbCcsXG4gICAgfSk7XG4gICAgdGVzdC5kZWVwRXF1YWwoY2ZuQnJhbmNoLnRlc3RQcm9wZXJ0aWVzKCkudGFncywgW3trZXk6ICdjZm4nLCB2YWx1ZTogJ2lzIGNvb2wnfV0pO1xuICAgIHRlc3QuZG9uZSgpO1xuICB9LFxuICAnd2hlbiBpbnZhbGlkIHRhZyBwcm9wZXJ0aWVzIGFyZSBwYXNzZWQgZnJvbSBMMXMnOiB7XG4gICAgJ21hcCBwYXNzZWQgaW5zdGVhZCBvZiBhcnJheSBpdCByYWlzZXMnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHJvb3QgPSBuZXcgU3RhY2soKTtcbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgbmV3IFRhZ2dhYmxlUmVzb3VyY2Uocm9vdCwgJ0Zha2VCcmFuY2hBJywge1xuICAgICAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICB0YWdzOiB7XG4gICAgICAgICAgICAgIGNmbjogJ2lzIGNvb2wnLFxuICAgICAgICAgICAgICBhc3BlY3RzOiAnb3ZlcndyaXRlJyxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgbmV3IEFzZ1RhZ2dhYmxlUmVzb3VyY2Uocm9vdCwgJ0Zha2VCcmFuY2hBJywge1xuICAgICAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICB0YWdzOiB7XG4gICAgICAgICAgICAgIGNmbjogJ2lzIGNvb2wnLFxuICAgICAgICAgICAgICBhc3BlY3RzOiAnb3ZlcndyaXRlJyxcbiAgICAgICAgICAgICAgcHJvcGFnYXRlQXRMYXVuY2g6IHRydWUsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgfSk7XG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuICAgICdpZiBhcnJheSBpcyBwYXNzZWQgaW5zdGVhZCBvZiBtYXAgaXQgcmFpc2VzJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCByb290ID0gbmV3IFN0YWNrKCk7XG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIG5ldyBNYXBUYWdnYWJsZVJlc291cmNlKHJvb3QsICdGYWtlU2FtJywge1xuICAgICAgICAgIHR5cGU6ICdBV1M6OkZha2U6OlRoaW5nJyxcbiAgICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICB0YWdzOiBbXG4gICAgICAgICAgICAgIHtrZXk6ICdjZm4nLCB2YWx1ZTogJ2lzIGNvb2wnLCBwcm9wYWdhdGVBdExhdW5jaDogdHJ1ZX0sXG4gICAgICAgICAgICAgIHtrZXk6ICdhc3BlY3RzJywgdmFsdWU6ICdvdmVyd3JpdGUnfSxcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSk7XG4gICAgICB9KTtcbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG59O1xuIl19