# Copyright © 2019-2023 CZ.NIC, z. s. p. o.
# SPDX-License-Identifier: GPL-3.0-or-later
#
# This file is part of dns-crawler.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import sys
from os.path import basename
from time import sleep

from redis import Redis
from redis.exceptions import ConnectionError, ExecAbortError, ResponseError
from rq import Queue
from rq.job import Job
from rq.registry import FinishedJobRegistry

from .config_loader import default_config_filename, load_config
from .crawl import get_json_result
from .redis_utils import get_redis_host
from .timestamp import timestamp

POLL_INTERVAL = 5
INPUT_CHUNK_SIZE = 10000


class ControllerNotRunning(Exception):
    pass


def print_help():
    exe = basename(sys.argv[0])
    sys.stderr.write(f"{exe} - the main process controlling the job queue and printing results.\n\n")
    sys.stderr.write(f"Usage: {exe} <file> [redis]\n")
    sys.stderr.write("       file - plaintext domain list, one domain per line, empty lines are ignored\n")
    sys.stderr.write("       redis - redis host:port:db, localhost:6379:0 by default\n\n")
    sys.stderr.write(f"Examples: {exe} domains.txt\n")
    sys.stderr.write(f"          {exe} domains.txt 192.168.0.22:4444:0\n")
    sys.stderr.write(f"          {exe} domains.txt redis.foo.bar:7777:2\n")
    sys.stderr.write(f"          {exe} domains.txt redis.foo.bar # port 6379 and DB 0 will be used if not specified\n")
    sys.exit(1)


def create_jobs(domains, function, redis, queue, timeout):
    pipe = redis.pipeline()
    jobs = []
    for domain in domains:
        jobs.append(Queue.prepare_data(function, (domain,), job_id=domain,
                                       description=domain, timeout=timeout, result_ttl=-1))
    queue.enqueue_many(jobs, pipeline=pipe)
    try:
        pipe.execute()
    except (ExecAbortError, ResponseError):
        sys.stderr.write("Redis reports a reponse error. ")
        maxmemory = int(redis.config_get("maxmemory")["maxmemory"])
        used_memory = int(redis.info()["used_memory"])
        if maxmemory / used_memory < 1.5:
            sys.stderr.write("Looks like it might be running out of memory:\n")
            sys.stderr.write(f"  used:  {used_memory}\n")
            sys.stderr.write(f"  limit: {maxmemory}\n")
            sys.stderr.write("Try increasing the `maxmemory` config option in redis.conf.\n")
        redis.flushdb()
        sys.exit(1)
    return len(jobs)


def main():
    if "-h" in sys.argv or "--help" in sys.argv or len(sys.argv) < 2:
        print_help()

    try:
        redis_host = get_redis_host(sys.argv, 2)
    except Exception as e:
        sys.stderr.write(f"{timestamp()} {str(e)}\n")
        sys.exit(1)
    redis = Redis(host=redis_host[0], port=redis_host[1], db=redis_host[2])

    redis.flushdb()
    config = load_config(default_config_filename, redis, save=True)
    finished_registry = FinishedJobRegistry(connection=redis)

    try:
        filename = sys.argv[1]
    except IndexError:
        print_help()

    try:
        sys.stderr.write(f"{timestamp()} Reading domains from {filename}.\n")
        input_file = open(filename, "r", encoding="utf-8")
        sys.stderr.write(
            f"{timestamp()} Creating job queue…\n")
        redis.set("locked", 1)
        queue = Queue(connection=redis)
        domain_count = 0
        read_domains = []
        for line in input_file:
            read_domains.append(line.rstrip())
            if len(read_domains) == INPUT_CHUNK_SIZE:
                domain_count = domain_count + create_jobs(read_domains, get_json_result, queue=queue,
                                                          redis=redis, timeout=config["timeouts"]["job"])
                sys.stderr.write(f"{timestamp()} {domain_count}\n")
                read_domains = []
        domain_count = domain_count + create_jobs(read_domains, get_json_result, queue=queue,
                                                  redis=redis, timeout=config["timeouts"]["job"])
        sys.stderr.write(f"{timestamp()} {domain_count}\n")
        input_file.close()

        finished_count = 0
        sys.stderr.write(f"{timestamp()} Created {domain_count} jobs. Unlocking queue and waiting for workers…\n")

        redis.set("locked", 0)

        while finished_count < domain_count:
            finished_domains = finished_registry.get_job_ids()
            if len(finished_domains) > 0:
                jobs = Job.fetch_many(finished_domains, connection=redis)

                pipe = redis.pipeline()
                count = 0
                for job in jobs:
                    result = job.latest_result()
                    if result.return_value:
                        count = count + 1
                        print(result.return_value)
                        finished_registry.remove(job, pipeline=pipe, delete_job=True)
                pipe.execute()
                finished_count = finished_count + count
            sys.stderr.write(f"{timestamp()} {finished_count}/{domain_count}\n")
            sleep(POLL_INTERVAL)
        queue.delete(delete_jobs=True)
        sys.exit(0)

    except KeyboardInterrupt:
        created_count = queue.count
        sys.stderr.write(f"{timestamp()} Cancelled. Deleting {created_count} jobs…\n")
        redis.flushdb()
        sys.stderr.write(f"{timestamp()} All jobs deleted, exiting.\n")
        sys.exit(1)

    except ConnectionError:
        sys.stderr.write(f"{timestamp()} Connection to Redis lost. :(\n")
        sys.exit(1)

    except FileNotFoundError:
        sys.stderr.write(f"File '{filename}' does not exist.\n\n")
        print_help()
