"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FullRegionS3CodeCommitBackup = void 0;
const path = require("path");
const core_1 = require("@aws-cdk/core");
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const S3_BUCKET_ENV = 'SCRIPTS_BUCKET';
const S3_KEY_ENV = 'SCRIPTS_BUCKET_KEY';
class FullRegionS3CodeCommitBackup extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { backupBucket, schedule, repositoryNames = [], computeType } = props;
        const asset = new aws_s3_assets_1.Asset(this, 'ScriptsDirectory', {
            path: path.join(__dirname, '..', 'scripts'),
        });
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_2_0;
        this.backupProject = new aws_codebuild_1.Project(this, 'FullRegionBackupProject', {
            environment: {
                buildImage,
                computeType: computeType || buildImage.defaultComputeType,
            },
            environmentVariables: {
                BACKUP_BUCKET: {
                    value: backupBucket.bucketName,
                },
                REPOSITORY_NAMES: {
                    value: repositoryNames.join(' '),
                },
                [S3_BUCKET_ENV]: { value: asset.s3BucketName },
                [S3_KEY_ENV]: { value: asset.s3ObjectKey },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                env: {
                    'git-credential-helper': 'yes',
                },
                phases: {
                    pre_build: {
                        commands: [
                            `echo "Downloading scripts from s3://\${${S3_BUCKET_ENV}}/\${${S3_KEY_ENV}}"`,
                            `aws s3 cp s3://\${${S3_BUCKET_ENV}}/\${${S3_KEY_ENV}} ./`,
                            `unzip ./$(basename \${${S3_KEY_ENV}})`,
                        ],
                    },
                    build: {
                        commands: [
                            'chmod +x backup_codecommit.sh',
                            './backup_codecommit.sh',
                        ],
                    },
                },
            }),
        });
        asset.grantRead(this.backupProject);
        backupBucket.grantPut(this.backupProject);
        this.backupProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: [
                'codecommit:BatchGet*',
                'codecommit:Get*',
                'codecommit:Describe*',
                'codecommit:List*',
                'codecommit:GitPull',
            ],
        }));
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule,
            targets: [new aws_events_targets_1.CodeBuildProject(this.backupProject)],
        });
    }
    /**
     * Defines an event rule which triggers when a backup fails.
     */
    onBackupFailed(id, options) {
        return this.backupProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup starts.
     */
    onBackupStarted(id, options) {
        return this.backupProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup complets successfully.
     */
    onBackupSucceeded(id, options) {
        return this.backupProject.onBuildSucceeded(id, options);
    }
}
exports.FullRegionS3CodeCommitBackup = FullRegionS3CodeCommitBackup;
//# sourceMappingURL=data:application/json;base64,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