"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk_1 = require("@aws-cdk/cdk");
const alias_1 = require("./alias");
const kms_generated_1 = require("./kms.generated");
class KeyBase extends cdk_1.Resource {
    /**
     * Defines a new alias for the key.
     */
    addAlias(alias) {
        return new alias_1.Alias(this, 'Alias', { name: alias, targetKey: this });
    }
    /**
     * Adds a statement to the KMS key resource policy.
     * @param statement The policy statement to add
     * @param allowNoOp If this is set to `false` and there is no policy
     * defined (i.e. external key), the operation will fail. Otherwise, it will
     * no-op.
     */
    addToResourcePolicy(statement, allowNoOp = true) {
        const stack = cdk_1.Stack.of(this);
        if (!this.policy) {
            if (allowNoOp) {
                return;
            }
            throw new Error(`Unable to add statement to IAM resource policy for KMS key: ${JSON.stringify(stack.resolve(this.keyArn))}`);
        }
        this.policy.addStatement(statement);
    }
    /**
     * Grant the indicated permissions on this key to the given principal
     *
     * This modifies both the principal's policy as well as the resource policy,
     * since the default CloudFormation setup for KMS keys is that the policy
     * must not be empty and so default grants won't work.
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalAndResource({
            grantee,
            actions,
            resourceArns: [this.keyArn],
            resource: this,
            resourceSelfArns: ['*']
        });
    }
    /**
     * Grant decryption permisisons using this key to the given principal
     */
    grantDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt');
    }
    /**
     * Grant encryption permisisons using this key to the given principal
     */
    grantEncrypt(grantee) {
        return this.grant(grantee, 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
    /**
     * Grant encryption and decryption permisisons using this key to the given principal
     */
    grantEncryptDecrypt(grantee) {
        return this.grant(grantee, 'kms:Decrypt', 'kms:Encrypt', 'kms:ReEncrypt*', 'kms:GenerateDataKey*');
    }
}
/**
 * Defines a KMS key.
 *
 * @resource AWS::KMS::Key
 */
class Key extends KeyBase {
    /**
     * Import an externally defined KMS Key using its ARN.
     *
     * @param scope  the construct that will "own" the imported key.
     * @param id     the id of the imported key in the construct tree.
     * @param keyArn the ARN of an existing KMS key.
     */
    static fromKeyArn(scope, id, keyArn) {
        class Import extends KeyBase {
            constructor() {
                super(...arguments);
                this.keyArn = keyArn;
                this.policy = undefined;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.policy) {
            this.policy = props.policy;
        }
        else {
            this.policy = new aws_iam_1.PolicyDocument();
            this.allowAccountToAdmin();
        }
        const resource = new kms_generated_1.CfnKey(this, 'Resource', {
            description: props.description,
            enableKeyRotation: props.enableKeyRotation,
            enabled: props.enabled,
            keyPolicy: this.policy,
        });
        this.keyArn = resource.keyArn;
        resource.options.deletionPolicy = props.retain === false
            ? cdk_1.DeletionPolicy.Delete
            : cdk_1.DeletionPolicy.Retain;
    }
    /**
     * Let users from this account admin this key.
     * @link https://aws.amazon.com/premiumsupport/knowledge-center/update-key-policy-future/
     */
    allowAccountToAdmin() {
        const actions = [
            "kms:Create*",
            "kms:Describe*",
            "kms:Enable*",
            "kms:List*",
            "kms:Put*",
            "kms:Update*",
            "kms:Revoke*",
            "kms:Disable*",
            "kms:Get*",
            "kms:Delete*",
            "kms:ScheduleKeyDeletion",
            "kms:CancelKeyDeletion"
        ];
        this.addToResourcePolicy(new aws_iam_1.PolicyStatement()
            .addAllResources()
            .addActions(...actions)
            .addAccountRootPrincipal());
    }
}
exports.Key = Key;
//# sourceMappingURL=data:application/json;base64,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