"use strict";
const assert_1 = require("@aws-cdk/assert");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const cdk_1 = require("@aws-cdk/cdk");
const lib_1 = require("../lib");
module.exports = {
    'default key'(test) {
        const stack = new cdk_1.Stack();
        new lib_1.Key(stack, 'MyKey');
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'default with no retention'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'TestStack');
        new lib_1.Key(stack, 'MyKey', { retain: false });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', { DeletionPolicy: "Delete" }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'default with some permission'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey');
        const p = new aws_iam_1.PolicyStatement().addAllResources().addAction('kms:encrypt');
        p.addAwsPrincipal('arn');
        key.addToResourcePolicy(p);
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: '*'
                                },
                                {
                                    Action: "kms:encrypt",
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: "arn"
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'key with some options'(test) {
        const stack = new cdk_1.Stack();
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false,
        });
        const p = new aws_iam_1.PolicyStatement().addAllResources().addAction('kms:encrypt');
        p.addAwsPrincipal('arn');
        key.addToResourcePolicy(p);
        key.node.applyAspect(new cdk_1.Tag('tag1', 'value1'));
        key.node.applyAspect(new cdk_1.Tag('tag2', 'value2'));
        key.node.applyAspect(new cdk_1.Tag('tag3', ''));
        assert_1.expect(stack).to(assert_1.exactlyMatchTemplate({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: '*'
                                },
                                {
                                    Action: "kms:encrypt",
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: "arn"
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        },
                        Enabled: false,
                        EnableKeyRotation: true,
                        Tags: [
                            {
                                Key: "tag1",
                                Value: "value1"
                            },
                            {
                                Key: "tag2",
                                Value: "value2"
                            },
                            {
                                Key: "tag3",
                                Value: ""
                            }
                        ]
                    },
                    DeletionPolicy: "Retain"
                }
            }
        }));
        test.done();
    },
    'addAlias creates an alias'(test) {
        const app = new cdk_1.App();
        const stack = new cdk_1.Stack(app, 'Test');
        const key = new lib_1.Key(stack, 'MyKey', {
            enableKeyRotation: true,
            enabled: false
        });
        const alias = key.addAlias('alias/xoo');
        test.ok(alias.aliasName);
        assert_1.expect(stack).toMatch({
            Resources: {
                MyKey6AB29FA6: {
                    Type: "AWS::KMS::Key",
                    Properties: {
                        EnableKeyRotation: true,
                        Enabled: false,
                        KeyPolicy: {
                            Statement: [
                                {
                                    Action: [
                                        "kms:Create*",
                                        "kms:Describe*",
                                        "kms:Enable*",
                                        "kms:List*",
                                        "kms:Put*",
                                        "kms:Update*",
                                        "kms:Revoke*",
                                        "kms:Disable*",
                                        "kms:Get*",
                                        "kms:Delete*",
                                        "kms:ScheduleKeyDeletion",
                                        "kms:CancelKeyDeletion"
                                    ],
                                    Effect: "Allow",
                                    Principal: {
                                        AWS: {
                                            "Fn::Join": [
                                                "",
                                                [
                                                    "arn:",
                                                    {
                                                        Ref: "AWS::Partition"
                                                    },
                                                    ":iam::",
                                                    {
                                                        Ref: "AWS::AccountId"
                                                    },
                                                    ":root"
                                                ]
                                            ]
                                        }
                                    },
                                    Resource: "*"
                                }
                            ],
                            Version: "2012-10-17"
                        }
                    },
                    DeletionPolicy: "Retain"
                },
                MyKeyAlias1B45D9DA: {
                    Type: "AWS::KMS::Alias",
                    Properties: {
                        AliasName: "alias/xoo",
                        TargetKeyId: {
                            "Fn::GetAtt": [
                                "MyKey6AB29FA6",
                                "Arn"
                            ]
                        }
                    }
                }
            }
        });
        test.done();
    },
    'grant decrypt on a key'(test) {
        // GIVEN
        const stack = new cdk_1.Stack();
        const key = new lib_1.Key(stack, 'Key');
        const user = new aws_iam_1.User(stack, 'User');
        // WHEN
        key.grantDecrypt(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    // This one is there by default
                    {
                        // tslint:disable-next-line:max-line-length
                        Action: ["kms:Create*", "kms:Describe*", "kms:Enable*", "kms:List*", "kms:Put*", "kms:Update*", "kms:Revoke*", "kms:Disable*", "kms:Get*", "kms:Delete*", "kms:ScheduleKeyDeletion", "kms:CancelKeyDeletion"],
                        Effect: "Allow",
                        Principal: { AWS: { "Fn::Join": ["", ["arn:", { Ref: "AWS::Partition" }, ":iam::", { Ref: "AWS::AccountId" }, ":root"]] } },
                        Resource: "*"
                    },
                    // This is the interesting one
                    {
                        Action: "kms:Decrypt",
                        Effect: "Allow",
                        Principal: { AWS: { "Fn::GetAtt": ["User00B015A1", "Arn"] } },
                        Resource: "*"
                    }
                ],
                Version: "2012-10-17"
            }
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: "kms:Decrypt",
                        Effect: "Allow",
                        Resource: { "Fn::GetAtt": ["Key961B73FD", "Arn"] }
                    }
                ],
                Version: "2012-10-17"
            },
        }));
        test.done();
    },
    'import/export can be used to bring in an existing key'(test) {
        const stack2 = new cdk_1.Stack();
        const myKeyImported = lib_1.Key.fromKeyArn(stack2, 'MyKeyImported', 'arn:of:key');
        // addAlias can be called on imported keys.
        myKeyImported.addAlias('alias/hello');
        assert_1.expect(stack2).toMatch({
            Resources: {
                MyKeyImportedAliasB1C5269F: {
                    Type: "AWS::KMS::Alias",
                    Properties: {
                        AliasName: "alias/hello",
                        TargetKeyId: 'arn:of:key'
                    }
                }
            }
        });
        test.done();
    },
    'addToResourcePolicy allowNoOp and there is no policy': {
        'succeed if set to true (default)'(test) {
            const stack = new cdk_1.Stack();
            const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'foo/bar');
            key.addToResourcePolicy(new aws_iam_1.PolicyStatement().addAllResources().addAction('*'));
            test.done();
        },
        'fails if set to false'(test) {
            const stack = new cdk_1.Stack();
            const key = lib_1.Key.fromKeyArn(stack, 'Imported', 'foo/bar');
            test.throws(() => key.addToResourcePolicy(new aws_iam_1.PolicyStatement().addAllResources().addAction('*'), /* allowNoOp */ false), 'Unable to add statement to IAM resource policy for KMS key: "foo/bar"');
            test.done();
        }
    }
};
//# sourceMappingURL=data:application/json;base64,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