import asyncio
import functools

import pyrogram

loop = asyncio.get_event_loop()


def patch(obj):
    def is_patchable(item):
        return getattr(item[1], "patchable", False)

    def wrapper(container):
        for name, func in filter(is_patchable, container.__dict__.items()):
            old = getattr(obj, name, None)
            setattr(obj, "old" + name, old)
            setattr(obj, name, func)
        return container

    return wrapper


def patchable(func):
    func.patchable = True
    return func


class UserCancelled(Exception):
    pass


pyrogram.errors.UserCancelled = UserCancelled


@patch(pyrogram.client.Client)
class Client:
    @patchable
    def __init__(self, *args, **kwargs):
        self._conversations = {}
        self.old__init__(*args, **kwargs)

    @patchable
    async def listen(self, chat_id, timeout=None):
        if not isinstance(chat_id, int):
            chat = await self.get_chat(chat_id)
            chat_id = chat.id

        future = loop.create_future()
        future.add_done_callback(functools.partial(self._clear, chat_id))
        self._conversations[chat_id] = future

        try:
            return await asyncio.wait_for(future, timeout)
        except asyncio.TimeoutError:
            self.cancel(chat_id)
            raise asyncio.TimeoutError()

    @patchable
    def _clear(self, chat_id, fut):
        if chat_id in self._conversations and self._conversations[chat_id] == fut:
            del self._conversations[chat_id]

    @patchable
    def cancel(self, chat_id):
        future = self._conversations.get(chat_id)
        if future and not future.done():
            future.set_exception(UserCancelled())
            self._clear(chat_id, future)


@patch(pyrogram.handlers.MessageHandler)
class MessageHandler:
    @patchable
    def __init__(self, callback, filters=None):
        self._user_callback = callback
        self.old__init__(self._resolver, filters)

    @patchable
    async def _resolver(self, client, message, *args):
        future = client._conversations.get(message.chat.id)
        if future and not future.done():
            future.set_result(message)
        else:
            await self._user_callback(client, message, *args)


@patch(pyrogram.types.Chat)
class Chat:
    @patchable
    def listen(self, *args, **kwargs):
        return self._client.listen(self.id, *args, **kwargs)

    @patchable
    def cancel(self):
        return self._client.cancel(self.id)

    @patchable
    def ask(self, *args, **kwargs):
        return self.listen(*args, **kwargs)


@patch(pyrogram.types.User)
class User:
    @patchable
    def listen(self, *args, **kwargs):
        return self._client.listen(self.id, *args, **kwargs)

    @patchable
    def cancel(self):
        return self._client.cancel(self.id)

    @patchable
    def ask(self, *args, **kwargs):
        return self.listen(*args, **kwargs)
