from datetime import datetime, timedelta
from json import loads
from time import sleep
from typing import Optional
from catwalk_client.common.exception import CatwalkClientException

from catwalk_common import OpenCase

from catwalk_client.common import CatwalkHTTPClient
from catwalk_client.common.logger import init_logger

log = init_logger(__name__)


class CaseExporter:
    def __init__(
        self,
        http_client: CatwalkHTTPClient,
        from_datetime: datetime,
        to_datetime: datetime,
        submitter_name: Optional[str] = None,
        submitter_version: Optional[str] = None,
        max_retries: int = 5,
        batch: int = 1,
        limit: Optional[int] = None,
    ):
        self.http_client = http_client

        self.from_datetime = from_datetime
        self.to_datetime = to_datetime

        self.filters = self._build_filters(
            submitter_name, submitter_version, from_datetime, to_datetime, limit
        )
        self.max_retries = max_retries
        self.batch = batch

    def export(self):
        for date in self._hour_range():
            current_retry = 0
            path = f"/api/cases/export/{date.year}/{date.month}/{date.day}/{date.hour}"
            next_part = False

            while path:
                response, success = self.http_client.get(
                    path, None if next_part else self.filters
                )

                if not success:
                    self._handle_retry(current_retry, response)
                    current_retry += 1
                    continue

                path = None
                next_part = False
                data = loads(response)
                for item in data["items"]:
                    yield OpenCase.parse_obj(item)

                if data["next_part"] and data["items"]:
                    next_part = True
                    path = str(data["next_part"]).replace(
                        self.http_client.get_url(""), ""
                    )

    def _handle_retry(self, current_retry: int, response):
        if current_retry < self.max_retries:
            multiplier = (current_retry * 2) or 1
            log.error(
                f"\n[catwalk-export] Retry ({current_retry + 1}/{self.max_retries}) "
                f"because got error {response} "
            )
            sleep(0.5 * multiplier)
        else:
            raise CatwalkClientException("Max retries exceeded")

    @staticmethod
    def _build_filters(
        submitter_name: str,
        submitter_version: str,
        from_datetime: datetime,
        to_datetime: datetime,
        limit: Optional[int] = None,
    ):
        filters = {
            "submitter_name": submitter_name,
            "submitter_version": submitter_version,
            "from_timestamp": from_datetime.isoformat(),
            "to_timestamp": to_datetime.isoformat(),
            "limit": limit,
        }
        return {k: v for k, v in filters.items() if v is not None}

    def _hour_range(self):
        start, end = self.from_datetime, self.to_datetime

        while start < end:
            yield start
            start += timedelta(hours=1)
