"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * Base class for the model side of context providers
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 *
 * @experimental
 */
class ContextProvider {
    /**
     * @returns the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
     */
    static getKey(scope, options) {
        const stack = stack_1.Stack.of(scope);
        const props = {
            account: stack.account,
            region: stack.region,
            ...options.props || {},
        };
        if (Object.values(props).find(x => token_1.Token.isUnresolved(x))) {
            throw new Error(`Cannot determine scope for context provider ${options.provider}.\n` +
                `This usually happens when one or more of the provider props have unresolved tokens`);
        }
        const propStrings = propsToArray(props);
        return {
            key: `${options.provider}:${propStrings.join(':')}`,
            props
        };
    }
    static getValue(scope, options) {
        const stack = stack_1.Stack.of(scope);
        if (token_1.Token.isUnresolved(stack.account) || token_1.Token.isUnresolved(stack.region)) {
            throw new Error(`Cannot retrieve value from context provider ${options.provider} since account/region ` +
                `are not specified at the stack level. Either configure "env" with explicit account and region when ` +
                `you define your stack, or use the environment variables "CDK_DEFAULT_ACCOUNT" and "CDK_DEFAULT_REGION" ` +
                `to inherit environment information from the CLI (not recommended for production stacks)`);
        }
        const { key, props } = this.getKey(scope, options);
        const value = scope.node.tryGetContext(key);
        const providerError = extractProviderError(value);
        // if context is missing or an error occurred during context retrieval,
        // report and return a dummy value.
        if (value === undefined || providerError !== undefined) {
            stack.reportMissingContext({ key, props, provider: options.provider, });
            if (providerError !== undefined) {
                scope.node.addError(providerError);
            }
            return { value: options.dummyValue };
        }
        return { value };
    }
    constructor() { }
}
exports.ContextProvider = ContextProvider;
/**
 * If the context value represents an error, return the error message
 */
function extractProviderError(value) {
    if (typeof value === 'object' && value !== null) {
        return value[cxapi.PROVIDER_ERROR_KEY];
    }
    return undefined;
}
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        // skip undefined values
        if (props[key] === undefined) {
            continue;
        }
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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