from typing import Dict
import numpy as np
from qpandalite.qasm import OpenQASM2_BaseParser, OpenQASM2_LineParser
from pathlib import Path
import pickle
from qpandalite.simulator.qasm_simulator import QASM_Simulator
from qpandalite.test._utils import qpandalite_test
from qpandalite.qasm import NotSupportedGateError
from qpandalite.qasm.random_qasm import random_qasm
from qpandalite.qasm.qasm_spec import available_qasm_gates, generate_sub_gateset_qasm

import qiskit
import qiskit.qasm2 as qasm
from qiskit.circuit import library as lib
from qiskit.qasm2.parse import CustomInstruction, LEGACY_CUSTOM_INSTRUCTIONS
from qiskit_aer import AerSimulator
from qiskit import transpile
from qiskit_aer import Aer

from scipy.sparse import coo_array

def simulate_by_qiskit(qasm_str, shots=10000):
    '''Simulate the circuit by qiskit simulator.

    Args:
        qasm_str (str): The QASM code to be simulated.

    Returns:
        numpy.ndarray: The probability of each state.
    '''

    qc = qasm.loads(qasm_str, custom_instructions=LEGACY_CUSTOM_INSTRUCTIONS)
    backend = Aer.get_backend("qasm_simulator")
    
    result = backend.run(qc, shots=shots).result()
    counts = result.get_counts(qc)
    return counts

def simulate_by_qpandalite(qasm_str, backend_type='statevector', shots=10000):
    '''Simulate the circuit by qpandalite simulator.

    Args:
        qasm_str (str): The QASM code to be simulated.

    Returns:
        numpy.ndarray: The probability of each state.
    '''

    qasm_simulator = QASM_Simulator(backend_type=backend_type)

    if qasm_simulator.opcode_simulator.simulator_typestr == 'density_operator':
        counts = qasm_simulator.simulate_pmeasure(qasm_str)
        # convert counts
        counts = {np.binary_repr(k, qasm_simulator.qubit_num) : v * shots for k, v in enumerate(counts)}

    elif qasm_simulator.opcode_simulator.simulator_typestr == 'statevector':
        counts = qasm_simulator.simulate_shots(qasm_str, shots=shots)
        # convert counts
        counts = {np.binary_repr(k, qasm_simulator.qubit_num) : v for k, v in counts.items()}
    else:
        raise ValueError('Unsupported simulator type.')

    return counts

class NotMatchError(Exception):
    pass


def compare_counts(counts1, counts2):
    '''Compare two count dictionaries.

    Args:
        counts1 (dict): The first count dictionary.
        counts2 (dict): The second count dictionary.

    Returns:
        bool: True if the two count dictionaries are the same, False otherwise.
    '''

    counts_difference = {}
    for k in counts1.keys():
        counts_difference[k] = counts1.get(k, 0) - counts2.get(k, 0)

    for k in counts2.keys():
        if k not in counts_difference:
            counts_difference[k] = -counts2.get(k, 0)

    shots = sum(counts1.values())
    counts_difference = {k : v / shots for k, v in counts_difference.items() if v != 0}
    # compare the two sparse vectors with a given threshold
    # if any value is greater than the threshold, raise an error
    threshold = 0.02
    if any(np.abs(v) > threshold for v in counts_difference.values()):
        raise NotMatchError('Counts are not the same.'
                            f'\ncounts1: {counts1}'
                            f'\ncounts2: {counts2}'
                            f'\ncounts_difference: {counts_difference}')


def compare_shots(circuit, backend_type='statevector'):
    # This test compares two density operators generated by qiskit and QPanda-lite.
    # The test is based on the following steps:
    # 1. Simulate the circuit using qiskit and QPanda-lite.
    # 2. Compare the results.
    # 3. If the results are not the same, raise an error.

    sim_qpandalite = QASM_Simulator(backend_type=backend_type)

    count1 = simulate_by_qiskit(circuit)
    count2 = simulate_by_qpandalite(circuit, backend_type=backend_type)

    return compare_counts(count1, count2)
    
def test_random_qasm_compare_shots_impl(  
        random_batchsize = 100, 
        n_qubit = 5,
        n_gates = 20,
        backend_type='statevector',
        instruction_set = available_qasm_gates):
    
    err_list = []    
    good_circuit_list = []
    bad_circuit_list = []
    for i in range(random_batchsize):
        qasm_code = random_qasm(n_qubits=n_qubit,
                                n_gates=n_gates,
                                instruction_set=instruction_set)
        
        err = compare_shots(qasm_code, backend_type=backend_type)    
        
        if err:
            print('Test failed!')
            err_list.append(err)
            bad_circuit_list.append((qasm_code, err))
        else:
            print('Test passed!')
            good_circuit_list.append((qasm_code, None))


    print(len(err_list), 'circuits failed')
    print(random_batchsize - len(err_list), 'circuits passed')

    # log good and bad circuits
    with open('good_circuits.txt', 'w') as f:
        for circuit, result in good_circuit_list:
            f.write(circuit + '\n----Result----\n' + str(result) + '\n-----------------\n\n')

    with open('bad_circuits.txt', 'w') as f:
        for e in err_list:
            f.write(str(e) + '\n')

    if len(err_list) > 0:
        raise ValueError('Some circuits failed!')
    

@qpandalite_test('Test Random QASM Compare Shots')
def test_random_qasm_compare_shots():
    
    gate_set = ['h', 'cx', 'rx', 'ry', 'rz', 
                'u1', 'u2', 'u3', 'id', 'x', 'y', 'z', 
                's', 'sdg', 't', 'tdg', 'swap' 
                'ccx', 'cu1', 'cswap']
    
    gate_set = ['h', 'cx',  'rx', 'ry', 'rz', 
                'u1', 'u2', 'u3', 'id', 'x', 'y', 'z', 
               's', 'sdg', 't', 'tdg', 'swap' 
                'ccx', 'cu1', 'cswap']
    
    gate_set = generate_sub_gateset_qasm(gate_set)
    test_random_qasm_compare_shots_impl(random_batchsize=10, 
                                   n_qubit=5, 
                                   n_gates=50, 
                                   instruction_set=gate_set,
                                   backend_type='statevector')
    
    test_random_qasm_compare_shots_impl(random_batchsize=10, 
                                   n_qubit=5, 
                                   n_gates=50, 
                                   instruction_set=gate_set,
                                   backend_type='density_matrix')
    
    test_random_qasm_compare_shots_impl(random_batchsize=10, 
                                   n_qubit=5, 
                                   n_gates=50, 
                                   instruction_set=gate_set,
                                   backend_type='density_matrix_qutip')

if __name__ == '__main__':
    test_random_qasm_compare_shots()