"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.collectRuntimeInformation = void 0;
const path_1 = require("path");
const node_version_1 = require("./node-version");
// list of NPM scopes included in version reporting e.g. @aws-cdk and @aws-solutions-konstruk
const WHITELIST_SCOPES = ['@aws-cdk', '@aws-solutions-konstruk', '@aws-solutions-constructs'];
/**
 * Returns a list of loaded modules and their versions.
 */
function collectRuntimeInformation() {
    const libraries = {};
    for (const fileName of Object.keys(require.cache)) {
        const pkg = findNpmPackage(fileName);
        if (pkg && !pkg.private) {
            libraries[pkg.name] = pkg.version;
        }
    }
    // include only libraries that are in the whitelistLibraries list
    for (const name of Object.keys(libraries)) {
        let foundMatch = false;
        for (const scope of WHITELIST_SCOPES) {
            if (name.startsWith(`${scope}/`)) {
                foundMatch = true;
            }
        }
        if (!foundMatch) {
            delete libraries[name];
        }
    }
    // add jsii runtime version
    libraries['jsii-runtime'] = getJsiiAgentVersion();
    return { libraries };
}
exports.collectRuntimeInformation = collectRuntimeInformation;
/**
 * Determines which NPM module a given loaded javascript file is from.
 *
 * The only infromation that is available locally is a list of Javascript files,
 * and every source file is associated with a search path to resolve the further
 * ``require`` calls made from there, which includes its own directory on disk,
 * and parent directories - for example:
 *
 * [ '...repo/packages/aws-cdk-resources/lib/cfn/node_modules',
 *   '...repo/packages/aws-cdk-resources/lib/node_modules',
 *   '...repo/packages/aws-cdk-resources/node_modules',
 *   '...repo/packages/node_modules',
 *   // etc...
 * ]
 *
 * We are looking for ``package.json`` that is anywhere in the tree, except it's
 * in the parent directory, not in the ``node_modules`` directory. For this
 * reason, we strip the ``/node_modules`` suffix off each path and use regular
 * module resolution to obtain a reference to ``package.json``.
 *
 * @param fileName a javascript file name.
 * @returns the NPM module infos (aka ``package.json`` contents), or
 *      ``undefined`` if the lookup was unsuccessful.
 */
function findNpmPackage(fileName) {
    const mod = require.cache[fileName];
    if (!mod.paths) {
        // sometimes this can be undefined. for example when querying for .json modules
        // inside a jest runtime environment.
        // see https://github.com/aws/aws-cdk/issues/7657
        // potentially we can remove this if it turns out to be a bug in how jest implemented the 'require' module.
        return undefined;
    }
    // For any path in ``mod.paths`` that is a node_modules folder, use its parent directory instead.
    const paths = mod.paths.map((path) => path_1.basename(path) === 'node_modules' ? path_1.dirname(path) : path);
    try {
        const packagePath = require.resolve(
        // Resolution behavior changed in node 12.0.0 - https://github.com/nodejs/node/issues/27583
        node_version_1.major >= 12 ? './package.json' : 'package.json', { paths });
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(packagePath);
    }
    catch (e) {
        return undefined;
    }
}
function getJsiiAgentVersion() {
    let jsiiAgent = process.env.JSII_AGENT;
    // if JSII_AGENT is not specified, we will assume this is a node.js runtime
    // and plug in our node.js version
    if (!jsiiAgent) {
        jsiiAgent = `node.js/${process.version}`;
    }
    return jsiiAgent;
}
//# sourceMappingURL=data:application/json;base64,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