"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.assertBound = exports.contentHash = exports.addStackArtifactToAssembly = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const crypto = require("crypto");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
/**
 * Shared logic of writing stack artifact to the Cloud Assembly
 *
 * This logic is shared between StackSyntheses.
 *
 * It could have been a protected method on a base class, but it
 * uses `Partial<cxapi.AwsCloudFormationStackProperties>` in the
 * parameters (which is convenient so I can remain typesafe without
 * copy/pasting), and jsii will choke on this type.
 */
function addStackArtifactToAssembly(session, stack, stackProps, additionalStackDependencies) {
    // nested stack tags are applied at the AWS::CloudFormation::Stack resource
    // level and are not needed in the cloud assembly.
    // TODO: move these to the cloud assembly artifact properties instead of metadata
    if (stack.tags.hasTags()) {
        stack.node.addMetadata(cxschema.ArtifactMetadataEntryType.STACK_TAGS, stack.tags.renderTags());
    }
    const deps = [
        ...stack.dependencies.map(s => s.artifactId),
        ...additionalStackDependencies,
    ];
    const meta = collectStackMetadata(stack);
    // backwards compatibility since originally artifact ID was always equal to
    // stack name the stackName attribute is optional and if it is not specified
    // the CLI will use the artifact ID as the stack name. we *could have*
    // always put the stack name here but wanted to minimize the risk around
    // changes to the assembly manifest. so this means that as long as stack
    // name and artifact ID are the same, the cloud assembly manifest will not
    // change.
    const stackNameProperty = stack.stackName === stack.artifactId
        ? {}
        : { stackName: stack.stackName };
    const properties = {
        templateFile: stack.templateFile,
        terminationProtection: stack.terminationProtection,
        ...stackProps,
        ...stackNameProperty,
    };
    // add an artifact that represents this stack
    session.assembly.addArtifact(stack.artifactId, {
        type: cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK,
        environment: stack.environment,
        properties,
        dependencies: deps.length > 0 ? deps : undefined,
        metadata: Object.keys(meta).length > 0 ? meta : undefined,
    });
}
exports.addStackArtifactToAssembly = addStackArtifactToAssembly;
/**
 * Collect the metadata from a stack
 */
function collectStackMetadata(stack) {
    const output = {};
    visit(stack);
    return output;
    function visit(node) {
        // break off if we reached a node that is not a child of this stack
        const parent = findParentStack(node);
        if (parent !== stack) {
            return;
        }
        if (node.node.metadata.length > 0) {
            // Make the path absolute
            output[construct_compat_1.ConstructNode.PATH_SEP + node.node.path] = node.node.metadata.map(md => stack.resolve(md));
        }
        for (const child of node.node.children) {
            visit(child);
        }
    }
    function findParentStack(node) {
        if (node instanceof stack_1.Stack && node.nestedStackParent === undefined) {
            return node;
        }
        if (!node.node.scope) {
            return undefined;
        }
        return findParentStack(node.node.scope);
    }
}
/**
 * Hash a string
 */
function contentHash(content) {
    return crypto.createHash('sha256').update(content).digest('hex');
}
exports.contentHash = contentHash;
/**
 * Throw an error message about binding() if we don't have a value for x.
 *
 * This replaces the ! assertions we would need everywhere otherwise.
 */
function assertBound(x) {
    if (x === null && x === undefined) {
        throw new Error('You must call bindStack() first');
    }
}
exports.assertBound = assertBound;
//# sourceMappingURL=data:application/json;base64,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