"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.findLastCommonElement = exports.pathToTopLevelStack = exports.PostResolveToken = exports.filterUndefined = exports.ignoreEmpty = exports.capitalizePropertyNames = void 0;
const intrinsic_1 = require("./private/intrinsic");
const stack_1 = require("./stack");
/**
 * Given an object, converts all keys to PascalCase given they are currently in camel case.
 * @param obj The object.
 */
function capitalizePropertyNames(construct, obj) {
    const stack = stack_1.Stack.of(construct);
    obj = stack.resolve(obj);
    if (typeof (obj) !== 'object') {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(x => capitalizePropertyNames(construct, x));
    }
    const newObj = {};
    for (const key of Object.keys(obj)) {
        const value = obj[key];
        const first = key.charAt(0).toUpperCase();
        const newKey = first + key.slice(1);
        newObj[newKey] = capitalizePropertyNames(construct, value);
    }
    return newObj;
}
exports.capitalizePropertyNames = capitalizePropertyNames;
/**
 * Turns empty arrays/objects to undefined (after evaluating tokens).
 */
function ignoreEmpty(obj) {
    return new PostResolveToken(obj, o => {
        // undefined/null
        if (o == null) {
            return o;
        }
        if (Array.isArray(o) && o.length === 0) {
            return undefined;
        }
        if (typeof (o) === 'object' && Object.keys(o).length === 0) {
            return undefined;
        }
        return o;
    });
}
exports.ignoreEmpty = ignoreEmpty;
/**
 * Returns a copy of `obj` without `undefined` (or `null`) values in maps or arrays.
 */
function filterUndefined(obj) {
    if (Array.isArray(obj)) {
        return obj.filter(x => x != null).map(x => filterUndefined(x));
    }
    if (typeof (obj) === 'object') {
        const ret = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value == null) {
                continue;
            }
            ret[key] = filterUndefined(value);
        }
        return ret;
    }
    return obj;
}
exports.filterUndefined = filterUndefined;
/**
 * A Token that applies a function AFTER resolve resolution
 */
class PostResolveToken extends intrinsic_1.Intrinsic {
    constructor(value, processor) {
        super(value);
        this.processor = processor;
    }
    resolve(context) {
        context.registerPostProcessor(this);
        return super.resolve(context);
    }
    postProcess(o, _context) {
        return this.processor(o);
    }
}
exports.PostResolveToken = PostResolveToken;
/**
 * @returns the list of stacks that lead from the top-level stack (non-nested) all the way to a nested stack.
 */
function pathToTopLevelStack(s) {
    if (s.nestedStackParent) {
        return [...pathToTopLevelStack(s.nestedStackParent), s];
    }
    else {
        return [s];
    }
}
exports.pathToTopLevelStack = pathToTopLevelStack;
/**
 * Given two arrays, returns the last common element or `undefined` if there
 * isn't (arrays are foriegn).
 */
function findLastCommonElement(path1, path2) {
    let i = 0;
    while (i < path1.length && i < path2.length) {
        if (path1[i] !== path2[i]) {
            break;
        }
        i++;
    }
    return path1[i - 1];
}
exports.findLastCommonElement = findLastCommonElement;
//# sourceMappingURL=data:application/json;base64,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