from openfisca_core.model_api import *
from openfisca_france_dotations_locales.entities import *
from numpy import log10, where
from openfisca_france_dotations_locales.variables.base import safe_divide


class dotation_forfaitaire(Variable):
    value_type = float
    entity = Commune
    definition_period = YEAR
    label = "Montant total de la dotation forfaitaire (DF)"
    reference = "https://www.legifrance.gouv.fr/codes/section_lc/LEGITEXT000006070633/LEGISCTA000006192290?etatTexte=VIGUEUR&etatTexte=VIGUEUR_DIFF#LEGISCTA000006192290"

    def formula_2018(commune, period, parameters):
        # TODO dotation_forfaitaire_an_dernier devrait être "retraitée"
        # = dotation forfaitaire notifiée N-1 + part CPS 2014 au périmètre année N nette de TASCOM
        # et si TASCOM > part CPS, le solde est prélevé sur DF N-1 retraitée de la commune
        # source : notes DGCL DF 2021 à 2025
        dotation_forfaitaire_an_dernier = commune('dotation_forfaitaire', period.last_year)
        df_evolution_part_dynamique = commune('df_evolution_part_dynamique', period)
        df_montant_ecretement = commune('df_montant_ecretement', period)
        return max_(0, dotation_forfaitaire_an_dernier + df_evolution_part_dynamique - df_montant_ecretement)


class df_coefficient_logarithmique(Variable):
    value_type = float
    entity = Commune
    definition_period = YEAR
    label = "Coefficient logarithmique « a » de la dotation forfaitaire appliqué à la population DGF communale"
    reference = [
        "Article L. 2334-7 du Code général des collectivités territoriales",
        "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000048849596/2023-12-31",
        "Note technique relative aux modalités de répartition de la dotation forfaitaire des communes \
        et des groupements de communes bénéficiaires de l'ancienne dotation touristique supplémentaire \
        au titre de l'exercice 2025, page 10",
        "http://www.dotations-dgcl.interieur.gouv.fr/consultation/documentAffichage.php?id=295"
        ]
    documentation = '''
    Extrait du I de l'article L. 2334-7 du CGCT applicable en 2024 et 2025 :
    « La population prise en compte pour la détermination du potentiel fiscal par habitant
    est corrigée par un coefficient logarithmique dont la valeur varie de 1 à 2
    en fonction croissante de la population de la commune tel que défini
    pour l'application du 1° du présent I ; »

    D'après les notes DGCL DF de 2020 à 2025, le coefficient logarithmique « a »
    est calculé d'après la population non majorée de l'année précédente.

    Ce coefficient sert au calcul du montant spontané de l'écrêtement à la dotation forfaitaire.
    Ne pas confondre le « coefficient logarithmique » et le « coefficient multiplicateur »
    de la part population (dite part dynamique) de la DF qui, lui, est déterminé
    par la population DGF majorée de l'année courante.
    '''

    def formula(commune, period, parameters):
        population_dgf = commune('population_dgf', period.last_year)

        plancher_dgcl_population_dgf = 500
        plafond_dgcl_population_dgf = 200_000
        # observation : ces valeurs correspondent au plancher et plafond des strates démographiques
        # cf. openfisca_france_dotations_locales/parameters/population/groupes_demographiques.yaml
        # et aux valeurs de l'Article R2334-3 du Code général des collectivités locales
        # https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000030542913/2015-05-04

        facteur_du_coefficient_logarithmique = 1 / (log10(plafond_dgcl_population_dgf / plancher_dgcl_population_dgf))  # le fameux 0.38431089 cité par les notes DGCL DF
        coefficient_logarithmique = max_(1, min_(2, 1 + facteur_du_coefficient_logarithmique * log10(population_dgf / plancher_dgcl_population_dgf)))
        return coefficient_logarithmique


class df_eligible_ecretement(Variable):
    value_type = bool
    entity = Commune
    definition_period = YEAR
    label = "Eligibilité à l'écrêtement de la dotation forfaitaire:\
        La commune est éligible à subir un écrêtement de sa dotation forfaitaire"
    reference = [
        "Article L. 2334-7 du Code général des collectivités territoriales",
        "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000037994287/2018-12-31",
        ]
    documentation = '''
        Extrait du III de l'article L. 2334-7 du CGCT applicable en 2019, 2020 et 2021 :
        « A compter de 2015[, les communes dont le potentiel fiscal par habitant est inférieur à 0,75 fois
        le potentiel fiscal moyen par habitant constaté pour l'ensemble des communes bénéficient
        d'une attribution au titre de la dotation forfaitaire [en 2012 et en I de cet article on parle de garantie ici]
        égale à celle calculée en application du présent III[en 2012 et en I de cet article, on parle de celle perçue l'année précédente ici].]
        Pour les communes dont le potentiel fiscal par habitant est supérieur ou égal à 0,75 fois
        le potentiel fiscal moyen par habitant constaté pour l'ensemble des communes,
        le montant calculé en application du premier alinéa du présent III est diminué,
        dans les conditions prévues à l'article L. 2334-7-1, en proportion de leur population
        et de l'écart relatif entre le potentiel fiscal par habitant de la commune et 0,75 fois
        le potentiel fiscal moyen par habitant constaté pour l'ensemble des communes. »

        Dans 'df_eligible_ecretement', on déclare comme éligibles les communes qui :
        - ont plus de 0.75 du potentiel fiscal moyen (revalorisé à 0.85 en 2022)
        - auraient une DF non nulle hors écrêtement.
        '''

    def formula(commune, period, parameters):
        pourcentage_potentiel_fiscal = parameters(period).dotation_forfaitaire.ecretement.seuil_rapport_potentiel_fiscal
        potentiel_fiscal = commune('potentiel_fiscal', period.last_year)
        df_coefficient_logarithmique = commune("df_coefficient_logarithmique", period)
        population_dgf = commune('population_dgf', period)

        population_logarithmee = population_dgf * df_coefficient_logarithmique
        potentiel_fiscal_moyen_commune = where(population_logarithmee > 0, potentiel_fiscal / population_logarithmee, 0)
        potentiel_fiscal_moyen_national = commune.etat('potentiel_fiscal_moyen_national', period.last_year)
        df_an_dernier = commune('dotation_forfaitaire', period.last_year)
        df_evolution_part_dynamique = commune("df_evolution_part_dynamique", period)
        df_hors_ecretement = max_(0, df_an_dernier + df_evolution_part_dynamique)
        df_ecretement_eligible = (potentiel_fiscal_moyen_commune >= pourcentage_potentiel_fiscal * potentiel_fiscal_moyen_national) * (df_hors_ecretement > 0)
        return df_ecretement_eligible


class montant_total_ecretement(Variable):
    value_type = int
    entity = Etat
    definition_period = YEAR
    label = "Montant total d'écrêtement qui sera ventilé entre la dotation forfaitaire des communes et la dotation de compensation des EPCI"
    reference = [
        "Article L2334-7-1 du Code général des collectivités locales",
        "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000033878417/2017-01-01",  # de 2017 à 2023
        "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000048849580/2023-12-31"  # 2024+
        ]
    documentation = '''
    'montant_total_ecretement' est un écrêtement - une minoration - qui sera appliqué à :
    * la dotation forfaitaire (DF) des communes
    * et à la dotation de compensation des EPCI à fiscalité propre (via écrêtement de sa part CPS).

    Il devra être écrêté de ces deux dotations selon des pourcentages de répartition
    décidés par le Comité des finances locales (CFL).

    Le montant de 'montant_total_ecretement' est déterminé en fonction d'une dépense à couvrir.
    Il s'agit ici de financer les coûts prévisionnels internes de la DGF du bloc communal (EPCI, communes).
    En particulier, l'écrêtement est destiné du financement de :
    * la progression annuelle des dotations de péréquation qui ne serait pas couverte
      par un abondement de la DGF totale (les augmentations de la loi ; hors majorations du CFL),
    * les majorations décidées par le Comité des finances locales (CFL)
    * le coût de la progression de la population des communes ('df_evolution_part_dynamique'),
    * la variations des préciputs sur la DGF entre l'année précédente et l'année courante
      (citée par les décisions du CFL 2024-1 et 2025-1).

    Pour rappel les dotations de péréquation du bloc communal sont :
    * communes : DSR, DSU, DNP
    * EPCI : dotation d'intercommunalité (DI).
    Source (consulée en 09.2025) :
    https://www.collectivites-locales.gouv.fr/finances-locales/perequation-verticale
    '''

    def formula(etat, period, parameters):
        # On déduit ici le montant total d'écrêtement d'après sa mission : les dépenses qu'il devra couvrir.
        #
        # Hypothèse : les minorations à l'enveloppe de DGF concernant des évolutions de compétences de département
        # n'interviennent pas dans le calcul de cette formule dédiée au bloc communal (c'est à dire qu'on
        # emploie l'abondement de l'Etat et l'enveloppe DGF avant minoration départementale).
        # Cf. notes openfisca_france_dotations_locales/parameters/montant_dotation_globale_fonctionnement.yaml
        # et : tests/dotation_forfaitaire/dotation_forfaitaire.yaml -n "montant total écrêtement - permet d'obtenir les montants réels"

        # progression annuelle des dotations de péréquation (DSR, DSU, DNP, DI ; loi puis décision CFL)
        accroissement_dsr = parameters(period).dotation_solidarite_rurale.augmentation_montant
        accroissement_dsu = parameters(period).dotation_solidarite_urbaine.augmentation_montant
        accroissement_dnp = 0  # + 0 € d'accroissement DNP depuis 2015
        progression_perequation_communale = accroissement_dsr + accroissement_dsu + accroissement_dnp

        # la progression annuelle des dotations de péréquation [décidée par la loi] restant à couvrir
        # après l'abondement, d'abord les augmentations des communes puis des epci
        # cet ordre est défini par le II de l'article L2334-7-1 du CGCT
        abondement_etat_communes_dgf = parameters(period).dotation_globale_fonctionnement.communes.montant_abondement_etat
        reste_a_charge_post_abondement_communes = progression_perequation_communale - abondement_etat_communes_dgf

        acroissement_di = parameters(period).dotation_intercommunalite.augmentation_montant

        # montants d'abondement et de compensation extraits de la note DGCL DI 2025, page 1
        # http://www.dotations-dgcl.interieur.gouv.fr/consultation/documentAffichage.php?id=301

        # abondement_etat_dgf = parameters(period).dotation_globale_fonctionnement.montant_abondement_etat
        # TODO vérifier si résidu entre abondement total et le communal :
        # abondement_etat_epci_dgf = max(abondement_etat_dgf - abondement_etat_communes_dgf, 0)
        abondement_etat_epci_dgf = 30_000_000 if period.start.year == 2024 else 0  # part de la DI "financée par le solde de la dotation d'aménagement" (nommage déduit de la décision 2024-1 du CFL)
        ecretement_dotation_compensation_epci = 60_000_000 if period.start.year == 2024 else 90_000_000  # pour financement de la progression de la DI non couverte par le solde de la dotation d'aménagement
        # TODO ou ? : ecretement_dotation_compensation_epci = acroissement_di - abondement_etat_epci_dgf
        # TODO et limiter à l'enveloppe de la dotation de compensation des EPCI ?

        reste_a_charge_post_abondement_epci = acroissement_di - abondement_etat_epci_dgf - ecretement_dotation_compensation_epci
        # TODO ou ? : reste_a_charge_post_abondement_epci_plf = acroissement_di_plf - ecretement_dotation_compensation_epci_plf

        # "solde" (terme employé par la loi) total négatif si l'abondement est supérieur au montant de la progression
        solde_post_abondement_bloc_communal = (
            reste_a_charge_post_abondement_communes
            + reste_a_charge_post_abondement_epci
            )

        majoration_dsr_cfl = parameters(period).dotation_solidarite_rurale.majoration_montant
        majoration_dsu_cfl = parameters(period).dotation_solidarite_urbaine.majoration_montant

        # coût de la progression de la population
        df_evolution_part_dynamique = etat.members('df_evolution_part_dynamique', period)
        total_evolution_part_dynamique = df_evolution_part_dynamique.sum()

        # Variation des préciputs
        # TODO: vérifier la nécessité d'additionner au résultat final le préciput sur la DGF entre l'année précédente
        # et l'année courante mentionné dans les décisions CFL 2024-1 et 2025-1, 3)
        # Peut-être ne s'applique-t-il pas au périmètre du bloc communal ?
        # variation_preciputs = 450_000 if period.start.year == 2024 else 0  # 0,45 M€ d'après décision du CFL 2024-1

        return (
            solde_post_abondement_bloc_communal
            + majoration_dsr_cfl
            + majoration_dsu_cfl
            + total_evolution_part_dynamique
            # Ajoute la variation du préciput sur DGF (c'est à dire, la correction en N du résultat N-1)
            # + variation_preciputs
            )


class df_montant_total_ecretement(Variable):
    value_type = int
    entity = Etat
    definition_period = YEAR
    label = "Montant total d'écrêtement à la dotation forfaitaire"
    reference = "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000033878417"
    documentation = '''
    Masse totale à prélever par le mécanisme d'écrêtement de la DF.

    Le pourcentage d'écrêtement est décidé par le Comité des Finances Locales.

    En 2019, extrait du rapport du gouvernement au parlement :
    « Compte tenu des règles de financement entre la dotation forfaitaire des communes
    et la dotation de compensation des EPCI prévues à l’article L. 2334-7-1 du CGCT
    et des choix opérés par le CFL en 2019, 60% des coûts sont supportés
    par la dotation forfaitaire. Par conséquent, 60% du surcoût de 12,7 M€ soit 7,6 M€,
    doit être écrêté en plus. »
    Source :
    https://www.banquedesterritoires.fr/sites/default/files/2019-12/Coefficient%20logarithmique%20-%20Rapport%20global%20%282%29.pdf
    '''

    def formula(etat, period, parameters):
        montant_total_ecretement = etat('montant_total_ecretement', period)
        dgf_part_ecretement_attribue_df = parameters(period).dgf_part_ecretement_attribue_df
        return dgf_part_ecretement_attribue_df * montant_total_ecretement


class df_montant_total_ecretement_hors_dsu_dsr(Variable):
    value_type = int
    entity = Etat
    definition_period = YEAR
    label = "Montant total à écrêter à la dotation forfaitaire hors variations de la DSU et de la DSR"
    reference = "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000033878417"
    # TODO supprimer cette variable suite à la correction de montant_total_ecretement ?
    # variable auxiliaire (ignorant l'existance de la DNP parmi les dotations de péréquation des communes)
    # = df_montant_total_ecretement - augmentations DSR et DSU
    # (de la loi, validées par le CFL ; incluant les majorations CFL)


class df_score_attribution_ecretement(Variable):
    value_type = float
    entity = Commune
    definition_period = YEAR
    label = "Score d'attribution de l'écrêtement de la dotation forfaitaire:\
        Score au prorata duquel l'écrêtement de la dotation forfaitaire est calculé"
    reference = "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000037994287"
    documentation = '''
        Le montant [...] est diminué [...] en proportion de leur population et de l'écart relatif \
        entre le potentiel fiscal par habitant de la commune et 0,75 fois le potentiel fiscal moyen \
        par habitant constaté pour l'ensemble des communes.
        '''

    def formula(commune, period, parameters):
        df_eligible_ecretement = commune('df_eligible_ecretement', period)

        population_dgf_annee_precedente = commune('population_dgf', period.last_year)
        population_dgf = commune('population_dgf', period)  # année courante N

        # PF/HAB en année N-1
        potentiel_fiscal_moyen_national_annee_precedente = commune.etat('potentiel_fiscal_moyen_national', period.last_year)  # année N-1

        # coefficient logarithmique "a" de l'année N calculé d'après population N-1
        df_coefficient_logarithmique = commune("df_coefficient_logarithmique", period)

        # Pf/hab en année N-1
        population_logarithmee = population_dgf_annee_precedente * df_coefficient_logarithmique
        potentiel_fiscal = commune('potentiel_fiscal', period.last_year)  # critère année N-1
        potentiel_fiscal_moyen_commune_annee_precedente = where(population_logarithmee > 0, potentiel_fiscal / population_logarithmee, 0)

        pourcentage_potentiel_fiscal = parameters(period).dotation_forfaitaire.ecretement.seuil_rapport_potentiel_fiscal
        seuil_ecretement_national = pourcentage_potentiel_fiscal * potentiel_fiscal_moyen_national_annee_precedente

        return where(
            df_eligible_ecretement,
            (
                (potentiel_fiscal_moyen_commune_annee_precedente - seuil_ecretement_national)
                / seuil_ecretement_national
                ) * population_dgf,
            0)


class df_evolution_part_dynamique(Variable):
    value_type = int
    entity = Commune
    definition_period = YEAR
    label = "Part dynamique (part population) de la dotation forfaitaire"
    reference = [
        "http://www.dotations-dgcl.interieur.gouv.fr/consultation/documentAffichage.php?id=115",
        "Article R2334-3 du Code général des collectivités locales",
        "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000030542913/2015-05-04",
        ]
    documentation = '''
        Évolution de la dotation forfaitaire consécutive aux changements de population DGF majorée.

        "Il est, selon le cas, ajouté ou soustrait à la dotation forfaitaire ainsi retraitée
        une part calculée en fonction de l’évolution de la population DGF entre 2019 et 2020
        et d’un montant compris entre 64,46 € et 128,93 € calculé en fonction croissante
        de la population de la commune."
        '''

    def formula(commune, period, parameters):
        # « coefficient multiplicateur » déterminé
        # en fonction de la population DGF de l'année courante

        plancher_dgcl_population_dgf_majoree = 500  # TODO extraire en paramètre
        plafond_dgcl_population_dgf_majoree = 200_000  # TODO extraire en paramètre
        facteur_du_coefficient_logarithmique = 1 / (log10(plafond_dgcl_population_dgf_majoree / plancher_dgcl_population_dgf_majoree))  # le fameux 0.38431089
        population_majoree_dgf = commune('population_dgf_majoree', period)
        population_majoree_dgf_an_dernier = commune('population_dgf_majoree', period.last_year)
        evolution_population = population_majoree_dgf - population_majoree_dgf_an_dernier

        facteur_minimum = parameters(period).dotation_forfaitaire.montant_minimum_par_habitant
        facteur_maximum = parameters(period).dotation_forfaitaire.montant_maximum_par_habitant
        dotation_supp_par_habitant = facteur_minimum + (facteur_maximum - facteur_minimum) * max_(0, min_(1, facteur_du_coefficient_logarithmique * log10(safe_divide(population_majoree_dgf, plancher_dgcl_population_dgf_majoree, 1))))
        return dotation_supp_par_habitant * evolution_population


class recettes_reelles_fonctionnement(Variable):
    value_type = float
    entity = Commune
    definition_period = YEAR
    label = "Recettes réelles de fonctionnement:\
    Recettes réelles de fonctionnement prises en compte pour le plafonnement de l'écrètement de la dotation forfaitaire"
    reference = "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000037994287"


class df_valeur_point_ecretement(Variable):
    value_type = float
    entity = Commune
    definition_period = YEAR
    label = "Valeur de point de l'écrêtement de la dotation forfaitaire"
    reference = [
        "Note d’information du 3 juillet 2020 relative à la répartition de la dotation forfaitaire des communes pour l’exercice 2020, page 14",
        "http://www.dotations-dgcl.interieur.gouv.fr/consultation/documentAffichage.php?id=115",
        ]
    documentation = '''
    Extrait de la note DGCL DF 2020, page 14 :
    VP
    = valeur de point
    = Masse totale à prélever / ∑ ( ((Pf/hab - 0.75 x PF/HAB) / 0.75 x PF/HAB)) x pop DGF 2020 )
    où :
    * Pf/hab = potentiel fiscal de la commune en 2019 rapporté à la population DGF 2019
      multipliée par un coefficient logarithmique "a" (...)
    * PF/HAB = potentiel fiscal moyen constaté au niveau national en 2019
      rapporté à la population DGF 2019 totale logarithmée

    Ce qui correspond à :
    VP = Masse totale à prélever / ∑ df_score_attribution_ecretement des communes
    où la masse totale correspond au montant total de l'écrêtement de DF à opérer
    pour couvrir les coûts internes de la DGF [du bloc communal uniquement ?]
    et où les éléments entant dans le calcul de Pf/hab et PF/HAB
    sont des critères de l'année précédente
    (ensuite multipliés par la pop DGF de l'année courante).
    '''

    def formula(commune, period, parameters):
        df_montant_total_ecretement = commune.etat("df_montant_total_ecretement", period)
        df_score_attribution_ecretement = commune('df_score_attribution_ecretement', period)

        valeur_point = df_montant_total_ecretement / df_score_attribution_ecretement.sum()
        return valeur_point


class df_montant_ecretement_spontane(Variable):
    value_type = int
    entity = Commune
    definition_period = YEAR
    label = "Montant spontané de l'écrêtement de la dotation forfaitaire"
    reference = [
        "Article L2334-7 du Code général des collectivités locales",
        "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000037994287"
        ]
    documentation = '''
    À partir de 2022 'dotation_forfaitaire.ecretement.seuil_rapport_potentiel_fiscal' est de 0,85
    et contribue au calcul du montant spontané comme suit :

    Montant spontané de l'écrêtement
    = ( ((Pf/hab) - (0,85*PF/HAB)) / (0,85*PF/HAB) ) x Pop DGF 2025 x VP
    '''

    def formula(commune, period, parameters):
        df_score_attribution_ecretement = commune('df_score_attribution_ecretement', period)
        df_valeur_point_ecretement = commune('df_valeur_point_ecretement', period)

        df_montant_ecretement_spontane = df_valeur_point_ecretement * df_score_attribution_ecretement
        return df_montant_ecretement_spontane


class df_montant_ecretement(Variable):
    value_type = int
    entity = Commune
    definition_period = YEAR
    label = "Ecrêtement de la dotation forfaitaire"
    reference = [
        "Article L2334-7 du Code général des collectivités locales",
        "https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000037994287"
        ]
    documentation = '''
    Montant retiré à la dotation forfaitaire de chaque commune.
    Cette minoration ne peut être supérieure à 1 % des recettes réelles de fonctionnement de leur budget principal.
    '''

    def formula(commune, period, parameters):
        df_montant_ecretement_spontane = commune('df_montant_ecretement_spontane', period)

        # l'écrêtement ne peut pas être supérieur à la DF après application de la part dynamique (part "population")

        df_an_dernier = commune('dotation_forfaitaire', period.last_year)
        df_evolution_part_dynamique = commune("df_evolution_part_dynamique", period)
        df_hors_ecretement = max_(0, df_an_dernier + df_evolution_part_dynamique)
        ecretement = min_(df_montant_ecretement_spontane, df_hors_ecretement)

        # plafonnement en fonction des recettes réelles de fonctionnement

        plafond_recettes = parameters(period).dotation_forfaitaire.ecretement.plafond_pourcentage_recettes_max
        recettes = commune("recettes_reelles_fonctionnement", period)
        ecretement = min_(ecretement, plafond_recettes * recettes)

        return ecretement
