# -*- coding: utf-8 -*-

"""

ZUGBRUECKE
Calling routines in Windows DLLs from Python scripts running on unixlike systems
https://github.com/pleiszenburg/zugbruecke

    src/zugbruecke/core/config.py: Handles the module's configuration

    Required to run on platform / side: [UNIX, WINE]

    Copyright (C) 2017-2022 Sebastian M. Ernst <ernst@pleiszenburg.de>

<LICENSE_BLOCK>
The contents of this file are subject to the GNU Lesser General Public License
Version 2.1 ("LGPL" or "License"). You may not use this file except in
compliance with the License. You may obtain a copy of the License at
https://www.gnu.org/licenses/old-licenses/lgpl-2.1.txt
https://github.com/pleiszenburg/zugbruecke/blob/master/LICENSE

Software distributed under the License is distributed on an "AS IS" basis,
WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for the
specific language governing rights and limitations under the License.
</LICENSE_BLOCK>

"""


# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# IMPORT
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

from collections.abc import KeysView
import os
import json
import sys
from typing import Any, Dict, Optional

from wenv import PythonVersion

from .abc import ConfigABC
from .const import CONFIG_FLD, CONFIG_FN
from .errors import ConfigParserError
from .lib import generate_session_id
from .typeguard import typechecked

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# CONFIGURATION CLASS
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++


@typechecked
class Config(dict, ConfigABC):
    """
    Handles the module's configuration. Subclass of ``dict``.
    It holds default values and overwrites them with values found in configuration files and environment variables.

    Args:
        override : Specify custom values for configuration parameters via keyword arguments.
    """

    _KEYS = (
        "stdout",
        "stderr",
        "log_write",
        "log_level",
        "arch",
        "pythonversion",
        "timeout_start",
        "timeout_stop",
    )

    def __init__(self, **override: Any):

        # Call parent constructur, just in case
        super().__init__()

        # Get config from files - only on Unix side
        if not sys.platform.startswith("win"):
            self.update(self._get_config_from_files())

        # Add override parameters
        if len(override) > 0:
            self.update(override)

        # Version type cleanup
        if not isinstance(self['pythonversion'], PythonVersion):
            self['pythonversion'] = PythonVersion.from_config(
                arch = self['arch'],
                version = self['pythonversion'],
            )

        # Add missing stuff
        if "id" not in self.keys():
            self["id"] = generate_session_id()  # Generate unique session id

    def __repr__(self) -> str:

        return f"<Config {super().__repr__():s}>"

    def __getitem__(self, key: str) -> Any:
        """
        Returns values from the following sources in the following order:

        - Environment variables (only on Unix side)
        - Internal storage, i.e. changed in the dictionary or read from configuration files.
        - Default values.

        Args:
            key : Name of configuration value.
        Returns:
            Arbitrary configuration value.
        """

        if key == "platform":
            return "WINE" if sys.platform.startswith("win") else "UNIX"

        if self["platform"] != "WINE":
            env_var = "ZUGBRUECKE_{NAME:s}".format(NAME=key.upper())
            if env_var in os.environ.keys():
                value = os.environ[env_var]
                if len(value) > 0:
                    if key == "pythonversion":
                        return PythonVersion.from_config(self['arch'], value)
                    if value.isnumeric():
                        return int(value)
                    if value.strip().lower() in ("true", "false"):
                        return {"true": True, "false": False}[value.strip().lower()]
                    return value

        if key in self.keys():
            return super().__getitem__(key)

        if key == "stdout":
            return True  # Display messages from stdout
        if key == "stderr":
            return True  # Display messages from stderr
        if key == "log_write":
            return False  # Write log messages into file
        if key == "log_level":
            return 0  # Overall log level: No logs are generated by default
        if key == "arch":
            return "win32"  # Define Wine & Wine-Python architecture
        if key == "pythonversion":
            return PythonVersion(self['arch'], 3, 7, 4, 'stable')  # Define Wine-Python version
        if key == "timeout_start":
            return 30  # Timeout for waiting on Wine-Python start
        if key == "timeout_stop":
            return 30  # Timeout for waiting on Wine-Python stop

        raise KeyError("not a valid configuration key", key)

    def export_dict(self) -> Dict[str, Any]:
        """
        Exports a dictionary.
        """

        return {field: self[field] for field in self._KEYS}

    def export_envvar_dict(self) -> Dict[str, str]:
        """
        Exports a dictionary which can passed to the OS as a set of environment variables for ``zugbruecke`` itself.
        """

        return {
            "ZUGBRUECKE_" + field.upper(): "" if field is None else str(self[field])
            for field in self._KEYS
        }

    def _get_default_config_directory(self) -> str:

        return os.path.join(os.path.expanduser("~"), CONFIG_FLD)

    def _get_config_from_files(self) -> Dict:

        base = {}

        # Look for config in the usual spots
        for fn in [
            "/etc/zugbruecke",
            os.path.join("/etc", CONFIG_FN), # TODO deprecated
            os.path.join("/etc", CONFIG_FN[1:]),
            os.path.join(os.path.expanduser("~"), CONFIG_FN),
            os.path.join(os.getcwd(), CONFIG_FN),
            os.path.join(os.environ.get("ZUGBRUECKE"), CONFIG_FN) if os.environ.get("ZUGBRUECKE") is not None else None,
            os.environ.get("ZUGBRUECKE"),
        ]:

            cnt = self._load_config_from_file(fn)

            if cnt is not None:
                base.update(cnt)

        return base

    def _load_config_from_file(self, try_path: Optional[str] = None) -> Optional[Dict[str, Any]]:

        # If there is a path ...
        if try_path is None:
            return

        # Is this a file?
        if not os.path.isfile(try_path):
            return

        # Read file
        try:
            with open(try_path, "r", encoding="utf-8") as f:
                cnt = f.read()
        except Exception as e:
            raise ConfigParserError(
                'Config file could not be read: "%s"' % try_path
            ) from e

        # Try to parse it
        try:
            cnt = json.loads(cnt)
        except Exception as e:
            raise ConfigParserError(
                'Config file could not be parsed: "%s"' % try_path
            ) from e

        # Ensure that config has the right format
        if not isinstance(cnt, dict):
            raise ConfigParserError('Config file is malformed: "%s"' % try_path)

        return cnt
