"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk_1 = require("@aws-cdk/cdk");
const kms_generated_1 = require("./kms.generated");
const REQUIRED_ALIAS_PREFIX = 'alias/';
const DISALLOWED_PREFIX = REQUIRED_ALIAS_PREFIX + 'AWS';
/**
 * Defines a display name for a customer master key (CMK) in AWS Key Management
 * Service (AWS KMS). Using an alias to refer to a key can help you simplify key
 * management. For example, when rotating keys, you can just update the alias
 * mapping instead of tracking and changing key IDs. For more information, see
 * Working with Aliases in the AWS Key Management Service Developer Guide.
 *
 * You can also add an alias for a key by calling `key.addAlias(alias)`.
 */
class EncryptionKeyAlias extends cdk_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.alias.startsWith(REQUIRED_ALIAS_PREFIX)) {
            throw new Error(`Alias must start with the prefix "${REQUIRED_ALIAS_PREFIX}": ${props.alias}`);
        }
        if (props.alias === REQUIRED_ALIAS_PREFIX) {
            throw new Error(`Alias must include a value after "${REQUIRED_ALIAS_PREFIX}": ${props.alias}`);
        }
        if (props.alias.startsWith(DISALLOWED_PREFIX)) {
            throw new Error(`Alias cannot start with ${DISALLOWED_PREFIX}: ${props.alias}`);
        }
        const resource = new kms_generated_1.CfnAlias(this, 'Resource', {
            aliasName: props.alias,
            targetKeyId: props.key.keyArn
        });
        this.aliasName = resource.aliasName;
    }
}
exports.EncryptionKeyAlias = EncryptionKeyAlias;
//# sourceMappingURL=data:application/json;base64,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