from typing import Any, Dict, Optional, Type
from abc import ABCMeta
from surfgram.types import TypesFactory


class CopyTextButtonMeta(ABCMeta):
    """Metaclass for CopyTextButton classes."""

    def __new__(cls, name: str, bases: tuple, attrs: Dict[str, Any]):
        new_class = super().__new__(cls, name, bases, attrs)
        if bases and new_class.__is_active__:
            CopyTextButtonsFactory.register_copy_text_button(new_class)
        return new_class


class CopyTextButtonsFactory(TypesFactory):
    """Factory for creating CopyTextButton instances."""

    COPYTEXTBUTTONS_REGISTRY: Dict[str, Type] = {}
    __fallback_handler__: Optional[Type] = None
    __type_name__ = "copy_text_button"

    @classmethod
    def register_copy_text_button(cls, copy_text_button_cls: Type) -> None:
        """Register a new copy_text_button handler."""
        instance = copy_text_button_cls()
        names = instance.__names__

        # Check if should be registered as fallback handler
        if not names or None in names or "" in names:
            cls.__fallback_handler__ = copy_text_button_cls
        else:
            for name in names:
                if name:  # Skip empty/None names
                    cls.COPYTEXTBUTTONS_REGISTRY[name] = copy_text_button_cls

    @classmethod
    async def create(cls, update: Any) -> Optional[Any]:
        """Create handler instance from update."""
        obj = update.copy_text_button
        trigger_value = obj.text

        # Try to get specific handler first
        handler_cls = cls.COPYTEXTBUTTONS_REGISTRY.get(trigger_value)

        # If no specific handler found, use fallback if available
        if handler_cls is None and cls.__fallback_handler__:
            handler_cls = cls.__fallback_handler__

        return handler_cls() if handler_cls else None
