from typing import Any, Dict, Optional, Type
from abc import ABCMeta
from surfgram.types import TypesFactory


class GiftMeta(ABCMeta):
    """Metaclass for Gift classes."""

    def __new__(cls, name: str, bases: tuple, attrs: Dict[str, Any]):
        new_class = super().__new__(cls, name, bases, attrs)
        if bases and new_class.__is_active__:
            GiftsFactory.register_gift(new_class)
        return new_class


class GiftsFactory(TypesFactory):
    """Factory for creating Gift instances."""

    GIFTS_REGISTRY: Dict[str, Type] = {}
    __fallback_handler__: Optional[Type] = None
    __type_name__ = "gift"

    @classmethod
    def register_gift(cls, gift_cls: Type) -> None:
        """Register a new gift handler."""
        instance = gift_cls()
        names = instance.__names__

        # Check if should be registered as fallback handler
        if not names or None in names or "" in names:
            cls.__fallback_handler__ = gift_cls
        else:
            for name in names:
                if name:  # Skip empty/None names
                    cls.GIFTS_REGISTRY[name] = gift_cls

    @classmethod
    async def create(cls, update: Any) -> Optional[Any]:
        """Create handler instance from update."""
        obj = update.gift
        trigger_value = obj.sticker

        # Try to get specific handler first
        handler_cls = cls.GIFTS_REGISTRY.get(trigger_value)

        # If no specific handler found, use fallback if available
        if handler_cls is None and cls.__fallback_handler__:
            handler_cls = cls.__fallback_handler__

        return handler_cls() if handler_cls else None
