from typing import Any, Dict, Optional, Type
from abc import ABCMeta
from surfgram.types import TypesFactory


class UserMeta(ABCMeta):
    """Metaclass for User classes."""

    def __new__(cls, name: str, bases: tuple, attrs: Dict[str, Any]):
        new_class = super().__new__(cls, name, bases, attrs)
        if bases and new_class.__is_active__:
            UsersFactory.register_user(new_class)
        return new_class


class UsersFactory(TypesFactory):
    """Factory for creating User instances."""

    USERS_REGISTRY: Dict[str, Type] = {}
    __fallback_handler__: Optional[Type] = None
    __type_name__ = "user"

    @classmethod
    def register_user(cls, user_cls: Type) -> None:
        """Register a new user handler."""
        instance = user_cls()
        names = instance.__names__

        # Check if should be registered as fallback handler
        if not names or None in names or "" in names:
            cls.__fallback_handler__ = user_cls
        else:
            for name in names:
                if name:  # Skip empty/None names
                    cls.USERS_REGISTRY[name] = user_cls

    @classmethod
    async def create(cls, update: Any) -> Optional[Any]:
        """Create handler instance from update."""
        obj = update.user
        trigger_value = obj.first_name

        # Try to get specific handler first
        handler_cls = cls.USERS_REGISTRY.get(trigger_value)

        # If no specific handler found, use fallback if available
        if handler_cls is None and cls.__fallback_handler__:
            handler_cls = cls.__fallback_handler__

        return handler_cls() if handler_cls else None
