# This script compares solar irradiance generated by NEUVAC to other models of solar EUV, including EUVAC, HEUVAC, and
# SOLOMON.
# The figures generated in this script correspond to those in the main NEUVAC publication.

#-----------------------------------------------------------------------------------------------------------------------
# Top-level Imports:
import sys, os
import numpy as np
import matplotlib.pyplot as plt
from datetime import datetime, timedelta
#-----------------------------------------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------------------------------------
# Local Imports:
from src.EUVpy.tools import toolbox
from src.EUVpy.tools.EUV.fism2_process import read_euv_csv_file
from src.EUVpy.tools.processIrradiances import obtainFism2
from src.EUVpy.NEUVAC import neuvac
from src.EUVpy.empiricalModels.models.EUVAC import euvac
from src.EUVpy.empiricalModels.models.HEUVAC import heuvac
from src.EUVpy.empiricalModels.models.SOLOMON import solomon
from src.EUVpy.tools.spectralAnalysis import irradiance_ensemble
#-----------------------------------------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------------------------------------
# Directory Management
euv_folder = '../tools/EUV/'
neuvac_tableFile = '../data/neuvac_table.txt'
neuvac_tableFile_Stan_Bands = '../data/neuvac_table_stan_bands.txt'
results_dir = 'Results/'
#-----------------------------------------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------------------------------------
# Global variables:
solomonTable = solomon.solomonBands
# Global Plotting Settings:
import matplotlib.pylab as pylab
params = {'legend.fontsize': 'large',
          'figure.figsize': (16, 8),
         'axes.labelsize': 'large',
         'axes.titlesize':'x-large',
         'xtick.labelsize':'x-large',
         'ytick.labelsize':'x-large'}
pylab.rcParams.update(params)
#-----------------------------------------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------------------------------------
# Helper Functions:
def energy(watt):
    """
    Given the power in units of Watts, return the energy in units of Joules (provided that the time given is in seconds).
    """
    return watt * 86400

def power(myEnergy):
    """
    Given the energy in units of Joules, return the power in units of Watts (provided that hte time is given in seconds).
    """
    return myEnergy / 86400
#-----------------------------------------------------------------------------------------------------------------------

#-----------------------------------------------------------------------------------------------------------------------
# Execution:
if __name__=="__main__":
    # ==================================================================================================================
    # fname = '../../../Downloads/fism2flux_daily_1947-2024.nc'
    # fism2data = toolbox.readFISM2(fname)
    # 1: OBTAIN DATA
    # Load in F10.7 and F10.7A data (OMNIWeb):
    omniTimesData = '../solarIndices/F107/OMNIWeb/OMNIF107times.pkl'
    omniF107Data = '../solarIndices/F107/OMNIWeb/OMNIF107vals.pkl'
    omniF107AveData = '../solarIndices/F107/OMNIWeb/OMNIF107averageVals.pkl'
    times = toolbox.loadPickle(omniTimesData)
    times = np.array([element + timedelta(hours=12) for element in times])
    F107 = toolbox.loadPickle(omniF107Data)
    F107A = toolbox.rollingAverage(F107, 81, True, True)
    F107B = toolbox.loadPickle(omniF107AveData)

    # Obtain FISM2 data:
    euv_data_59 = read_euv_csv_file(euv_folder + 'euv_59.csv', band=False)
    mids = 0.5 * (euv_data_59['long'] + euv_data_59['short'])
    fism2file = '../empiricalModels/irradiances/FISM2/daily_data_1947-2023.nc'
    myIrrTimesFISM2, wavelengthsFISM2, myIrrDataAllFISM2, myIrrUncAllFISM2 = obtainFism2(fism2file)
    # Rebin the data (old method):
    # myIrrDataWavelengthsFISM2, rebinnedIrrDataFISM2 = toolbox.rebin(wavelengthsFISM2, myIrrDataAllFISM2, euv_data_59,
    #                                                                 zero=False)
    # Rebin the data (new method):
    myIrrDataWavelengthsFISM2_n, rebinnedIrrDataFISM2_n = toolbox.newbins(wavelengthsFISM2, myIrrDataAllFISM2, euv_data_59,
                                                                          zero=True)

    fism2Irr = rebinnedIrrDataFISM2_n[:, 7:44] # rebinnedIrrDataFISM2[:, 7:44]
    # Harmonize the times for NEUVAC and FISM2:
    correspondingIndsFISM2 = np.where((myIrrTimesFISM2 >= times[0]) & (myIrrTimesFISM2 <= times[-1]))[0]
    correspondingIrrTimesFISM2 = myIrrTimesFISM2[correspondingIndsFISM2]
    correspondingFism2Irr = fism2Irr[correspondingIndsFISM2, :]

    # Do the same for the FISM2 data in the SOLOMON bins:
    fism2file = '../empiricalModels/irradiances/FISM2/daily_bands_1947-2024.nc'
    myIrrTimesFISM2Bands, wavelengthsFISM2Bands, myDataAllFISM2Bands, _ = obtainFism2(fism2file, bands=True)
    myFluxDataAllFISM2Bands, myIrrDataAllFISM2Bands = myDataAllFISM2Bands

    # Replace bad values with NaNs:
    myIrrDataAllFISM2BandsFixed = myIrrDataAllFISM2Bands.copy()
    myIrrDataAllFISM2BandsFixed[myIrrDataAllFISM2BandsFixed <= 0] = np.nan

    # Harmonize the times for NEUVAC and FISM2:
    correspondingIndsFISM2StanBands = \
    np.where((myIrrTimesFISM2Bands >= times[0]) & (myIrrTimesFISM2Bands <= times[-1]))[0]
    correspondingIrrTimesFISM2StanBands = myIrrTimesFISM2Bands[correspondingIndsFISM2StanBands]
    correspondingIrrFISM2StanBands = myIrrDataAllFISM2BandsFixed[correspondingIndsFISM2StanBands, :]

    # Determine how the raw FISM2 data are to be rebinned into the Solomon Bands:
    # toolbox.solomonAnalysis(correspondingIrrFISM2StanBands, solomonTable, myIrrDataAllFISM2[correspondingIndsFISM2, :], wavelengthsFISM2)

    # Manually rebinning of raw FISM2 data to the Solomon bins (to check the binning procedure):
    myIrrDataWavelengthsFISM2_Solomon_n, rebinnedIrrDataFISM2_Solomon_n = toolbox.newbins(wavelengthsFISM2,
                                                                                          myIrrDataAllFISM2,
                                                                                          solomonTable, zero=True)
    correspondingRebinnedIrrDataFISM2_Solomon_n = rebinnedIrrDataFISM2_Solomon_n[correspondingIndsFISM2StanBands, :]

    # Make a chart of different wavelength resolutions laid on top of each other:
    fig, ax = plt.subplots(figsize=(18, 6), nrows=2, ncols=1, sharex=True)
    # EUVAC:
    euv_data_37 = read_euv_csv_file(euv_folder + 'euv.csv', band=False)
    EUVACBandInds = np.where(np.subtract(euv_data_37['long'], euv_data_37['short']) != 0)[0]
    #xEUVAC = np.concatenate((euv_data_59['short'], np.array([euv_data_59['long'][-1]])), axis=-1)
    for i in EUVACBandInds:
        ax[0].axvspan(euv_data_37['short'][i], euv_data_37['long'][i], ymin=0.5, ymax=1, color='tab:green')
        ax[0].vlines(x=[euv_data_37['short'][i], euv_data_37['long'][i]], ymin=0, ymax=1, color='k')
    EUVACLineInds = np.where(np.subtract(euv_data_37['long'], euv_data_37['short']) == 0)[0]
    ax[0].vlines(x=euv_data_37['short'][EUVACLineInds], ymin=-1, ymax=0, color='r')
    ax[0].axhline(y=0, linewidth=2, color='gray')
    ax[0].set_xlim([solomonTable['short'][0], euv_data_37['long'][-1]])
    ax[0].set_ylim([-1, 1])
    ax[0].set_ylabel('EUVAC Bins', rotation='horizontal', labelpad=90, fontsize=25)
    plt.setp(ax[0].get_yticklabels(), visible=False)
    ax[0].tick_params(axis='y', which='both', length=0)
    # ax[0].set_xscale('log')
    # SOLOMON:
    for i in range(len(solomonTable['short'])):
        ax[1].axvspan(solomonTable['short'][i], solomonTable['long'][i], color='lightblue')
        ax[1].vlines(x=[solomonTable['short'][i], solomonTable['long'][i]], ymin=0, ymax=1, color='k')
    ax[1].set_ylim([0, 1])
    ax[1].set_ylabel('Solomon Bins', rotation='horizontal', labelpad=90, fontsize=25)
    plt.setp(ax[1].get_yticklabels(), visible=False)
    ax[1].tick_params(axis='y', which='both', length=0)
    ax[1].tick_params(axis='x', labelsize=25)
    ax[1].set_xlabel(r'Wavelength ($\mathrm{\AA}$)', fontsize=25)
    # Spacing and titles:
    # textstr1 = 'EUVAC Bins'
    # textstr2 = 'Solomon Bins'
    plt.subplots_adjust(wspace=0, hspace=0)
    fig.suptitle('Wavelength Coverage for the EUVAC and Solomon Bins', fontsize=30, y=0.97)
    fig.tight_layout()
    plt.savefig(results_dir + 'bin_overview.png', dpi=300)

    # ==================================================================================================================
    # 2: Run the empirical models:
    override = False # If True, recompute everything
    if os.path.isfile(results_dir+'cachedData.pkl') == True and override == False:
        cached=True
        cachedData = toolbox.loadPickle(results_dir + 'cachedData.pkl')
        iterations = cachedData["iterations"]
        ensemble_NeuvacIrr = cachedData["ensemble_NeuvacIrr"]
        ensemble_average_NeuvacIrr = cachedData["ensemble_average_NeuvacIrr"]
        ensemble_stddev_NeuvacIrr = cachedData["ensemble_stddev_NeuvacIrr"]
        neuvacIrr = cachedData["neuvacIrr"]
        perturbedNeuvacIrr = cachedData["perturbedNeuvacIrr"]
        savedPertsNeuvac = cachedData["savedPertsNeuvac"]
        cc2Neuvac = cachedData["cc2Neuvac"]
        euvacFlux = cachedData["euvacFlux"]
        euvacIrr = cachedData["euvacIrr"]
        perturbedEuvacIrr = cachedData["perturbedEuvacIrr"]
        savedPertsEuvac = cachedData["savedPertsEuvac"]
        cc2Euvac = cachedData["cc2Euvac"]
        heuvac_wav = cachedData["heuvac_wav"]
        heuvacFlux = cachedData["heuvacFlux"]
        heuvacIrr = cachedData["heuvacIrr"]
        perturbedEuvIrradiance = cachedData["perturbedEuvIrradiance"]
        savedPertsHeuvac = cachedData["savedPertsHeuvac"]
        cc2Heuvac = cachedData["cc2Heuvac"]
        ensemble_NeuvacIrrSolomon = cachedData["ensemble_NeuvacIrrSolomon"]
        ensemble_average_NeuvacIrrSolomon = cachedData["ensemble_average_NeuvacIrrSolomon"]
        ensemble_stddev_NeuvacIrrSolomon = cachedData["ensemble_stddev_NeuvacIrrSolomon"]
        solomonFluxHFG = cachedData["solomonFluxHFG"]
        solomonIrrHFG = cachedData["solomonIrrHFG"]
        solomonFluxEUVAC = cachedData["solomonFluxEUVAC"]
        solomonIrrEUVAC = cachedData["solomonIrrEUVAC"]
    else:
        cached=False
        # Generate NEUVAC data:
        iterations = 100
        ensemble_NeuvacIrr, ensemble_average_NeuvacIrr, ensemble_stddev_NeuvacIrr = irradiance_ensemble(F107, F107B,
                                                                                                        iterations=iterations,
                                                                                                        model='NEUVAC-E') # plt.figure(); plt.fill_between(times, (ensemble_average_NeuvacIrr-ensemble_stddev_NeuvacIrr)[:, 0], (ensemble_average_NeuvacIrr+ensemble_stddev_NeuvacIrr)[:, 0], color='gray', alpha=0.8); plt.plot(times, ensemble_average_NeuvacIrr[:, 0], color='k')
        neuvacIrr, perturbedNeuvacIrr, savedPertsNeuvac, cc2Neuvac = neuvac.neuvacEUV(F107, F107B, bands='EUVAC', tableFile=neuvac_tableFile,
                                                                                      statsFiles=['corMat.pkl', 'sigma_NEUVAC.pkl'])

        # Generate EUVAC data:
        # ensemble_EuvacIrr, ensemble_average_EuvacIrr, ensemble_stddev_EuvacIrr = irradiance_ensemble(F107, F107A,
        #                                                                                                 iterations=iterations,
        #                                                                                                 model='EUVAC')
        euvacFlux, euvacIrr, perturbedEuvacIrr, savedPertsEuvac, cc2Euvac = euvac.euvac(F107, F107A,
                                                                                        statsFiles=None) #['corMatEUVAC.pkl',
                                                                                                    #'sigma_EUVAC.pkl'])

        # Generate HEUVAC data:
        # ensemble_HeuvacIrr, ensemble_average_HeuvacIrr, ensemble_stddev_HeuvacIrr = irradiance_ensemble(F107,
        #                                                                                              F107A,
        #                                                                                              iterations=iterations,
        #                                                                                              model='HEUVAC')

        heuvac_wav, heuvacFlux, heuvacIrr, perturbedEuvIrradiance, savedPertsHeuvac, cc2Heuvac = heuvac.heuvac(F107, F107A,
                                                                                                               torr=True,
                                                                                                               statsFiles=None) #[
                                                                                                                   # 'corMatHEUVAC.pkl',
                                                                                                                   # 'sigma_HEUVAC.pkl'])

        # Generate SOLOMON data and rebin everything into the SOLOMON bins:
        ensemble_NeuvacIrrSolomon, ensemble_average_NeuvacIrrSolomon, ensemble_stddev_NeuvacIrrSolomon = irradiance_ensemble(F107, F107B,
                                                                                                        iterations=iterations,
                                                                                                        model='NEUVAC-S')
        neuvacIrrSolomon, perturbedNeuvacIrrSolomon, savedPertsNeuvacSolomon, cc2NeuvacSolomon = neuvac.neuvacEUV(F107, F107B, bands='SOLOMON',
                                                                                                                  tableFile=neuvac_tableFile_Stan_Bands,
                                                                                                                  statsFiles=['corMatStanBands.pkl',
                                                                                                  'sigma_NEUVAC_StanBands.pkl'])

        solomonFluxHFG, solomonIrrHFG = solomon.solomon(F107, F107A, model='HFG')
        solomonFluxEUVAC, solomonIrrEUVAC = solomon.solomon(F107, F107A, model='EUVAC')

        cachedData = {
            "iterations": iterations,
            "ensemble_NeuvacIrr": ensemble_NeuvacIrr,
            "ensemble_average_NeuvacIrr": ensemble_average_NeuvacIrr,
            "ensemble_stddev_NeuvacIrr": ensemble_stddev_NeuvacIrr,
            "neuvacIrr": neuvacIrr,
            "perturbedNeuvacIrr": perturbedNeuvacIrr,
            "savedPertsNeuvac": savedPertsNeuvac,
            "cc2Neuvac": cc2Neuvac,
            "euvacFlux": euvacFlux,
            "euvacIrr": euvacIrr,
            "perturbedEuvacIrr": perturbedEuvacIrr,
            "savedPertsEuvac": savedPertsEuvac,
            "cc2Euvac": cc2Euvac,
            "heuvac_wav": heuvac_wav,
            "heuvacFlux": heuvacFlux,
            "heuvacIrr": heuvacIrr,
            "perturbedEuvIrradiance": perturbedEuvIrradiance,
            "savedPertsHeuvac": savedPertsHeuvac,
            "cc2Heuvac": cc2Heuvac,
            "ensemble_NeuvacIrrSolomon": ensemble_NeuvacIrrSolomon,
            "ensemble_average_NeuvacIrrSolomon": ensemble_average_NeuvacIrrSolomon,
            "ensemble_stddev_NeuvacIrrSolomon": ensemble_stddev_NeuvacIrrSolomon,
            "solomonFluxHFG": solomonFluxHFG,
            "solomonIrrHFG": solomonIrrHFG,
            "solomonFluxEUVAC": solomonFluxEUVAC,
            "solomonIrrEUVAC": solomonIrrEUVAC
        }
        toolbox.savePickle(cachedData, results_dir + 'cachedData.pkl')

    # ==================================================================================================================
    # NOTE: In any analysis below, only 3 bands will be considered for plotting (425 A, 725 A, 975 A)
    # (12, 24, 31); (9, 11, 19)
    # ==================================================================================================================

    # FIRST FIGURE:
    # 3 wavelength bins: FISM2 vs P and EUVAC/HEUVAC vs P scatter (with best fit line) - show poor match
    P = 0.5*(F107+F107B)
    sortIndsP = np.argsort(P)
    sortedP = P[sortIndsP]
    fig, ax = plt.subplots(nrows=2, ncols=3)
    # FIRST (TOP) ROW:
    # 425 A (index 12)
    ax[0, 0].scatter(sortedP, correspondingFism2Irr[sortIndsP, 12], label='FISM2')
    ax[0, 0].scatter(sortedP, euvacIrr[sortIndsP, 12], label='EUVAC', color='tab:green')
    ax[0, 0].scatter(sortedP, heuvacIrr[sortIndsP, 12], label='HEUVAC', color='tab:red')
    ax[0, 0].legend(loc='best', fontsize=16)
    ax[0, 0].set_ylabel('Irradiance (W/m$^2$)', fontsize=20)
    ax[0, 0].set_title('425 $\mathrm{\AA}$', fontsize=20)
    ax[0, 0].tick_params(axis='both', labelsize=17)
    # 725 A (index 24)
    ax[0, 1].scatter(sortedP, correspondingFism2Irr[sortIndsP, 24], label='FISM2')
    ax[0, 1].scatter(sortedP, euvacIrr[sortIndsP, 24], label='EUVAC', color='tab:green')
    ax[0, 1].scatter(sortedP, heuvacIrr[sortIndsP, 24], label='HEUVAC', color='tab:red')
    ax[0, 1].legend(loc='best', fontsize=16)
    ax[0, 1].set_title('725 $\mathrm{\AA}$', fontsize=20)
    ax[0, 1].tick_params(axis='both', labelsize=17)
    # 975 (index 31)
    ax[0, 2].scatter(sortedP, correspondingFism2Irr[sortIndsP, 31], label='FISM2')
    ax[0, 2].scatter(sortedP, euvacIrr[sortIndsP, 31], label='EUVAC', color='tab:green')
    ax[0, 2].scatter(sortedP, heuvacIrr[sortIndsP, 31], label='HEUVAC', color='tab:red')
    ax[0, 2].legend(loc='best', fontsize=16)
    ax[0, 2].set_title('975 $\mathrm{\AA}$', fontsize=20)
    ax[0, 2].tick_params(axis='both', labelsize=17)
    # SECOND (BOTTOM) ROW:
    solomonWavs = 0.5 * (solomonTable['short'] + solomonTable['long'])
    # 430 A (index 9)
    ax[1, 0].scatter(sortedP, correspondingIrrFISM2StanBands[sortIndsP, 9], label='FISM2')
    ax[1, 0].scatter(sortedP, solomonIrrEUVAC[sortIndsP, 9], label='EUVAC-22', color='tab:green')
    ax[1, 0].scatter(sortedP, solomonIrrHFG[sortIndsP, 9], label='HFG', color='purple')
    ax[1, 0].legend(loc='best', fontsize=16)
    ax[1, 0].set_xlabel('P (sfu)', fontsize=20)
    ax[1, 0].set_ylabel('Irradiance (W/m$^2$)', fontsize=20)
    ax[1, 0].set_title('430 $\mathrm{\AA}$', fontsize=20)
    ax[1, 0].tick_params(axis='both', labelsize=17)
    # 724 A (11)
    ax[1, 1].scatter(sortedP, correspondingIrrFISM2StanBands[sortIndsP, 11], label='FISM2')
    ax[1, 1].scatter(sortedP, solomonIrrEUVAC[sortIndsP, 11], label='EUVAC-22', color='tab:green')
    ax[1, 1].scatter(sortedP, solomonIrrHFG[sortIndsP, 11], label='HFG', color='purple')
    ax[1, 1].legend(loc='best', fontsize=16)
    ax[1, 1].set_xlabel('P (sfu)', fontsize=20)
    ax[1, 1].set_title('724 $\mathrm{\AA}$', fontsize=20)
    ax[1, 1].tick_params(axis='both', labelsize=17)
    # 981 A (19)
    ax[1, 2].scatter(sortedP, correspondingIrrFISM2StanBands[sortIndsP, 19], label='FISM2')
    ax[1, 2].scatter(sortedP, solomonIrrEUVAC[sortIndsP, 19], label='EUVAC-22', color='tab:green')
    ax[1, 2].scatter(sortedP, solomonIrrHFG[sortIndsP, 19], label='HFG', color='purple')
    ax[1, 2].legend(loc='best', fontsize=16)
    ax[1, 2].set_xlabel('P (sfu)', fontsize=20)
    ax[1, 2].set_title('981 $\mathrm{\AA}$', fontsize=20)
    ax[1, 2].tick_params(axis='both', labelsize=17)
    # Save the figure:
    fig.tight_layout()
    fig.suptitle('Solar Irradiance vs. P', fontsize=25, fontweight='bold')
    fig.subplots_adjust(top=0.88)
    plt.savefig(results_dir + 'irradiance_vs_P_6_bands.png', dpi=300)

    # SECOND FIGURE:
    # Same as first figure, but with NEUVAC
    fig, ax = plt.subplots(nrows=2, ncols=3)
    # FIRST (TOP) ROW:
    # 425 A (index 12)
    ax[0, 0].scatter(sortedP, correspondingFism2Irr[sortIndsP, 12], label='FISM2')
    ax[0, 0].scatter(sortedP, ensemble_average_NeuvacIrr[sortIndsP, 12], label='NEUVAC', color='tab:orange')
    ax[0, 0].legend(loc='best', fontsize=16)
    ax[0, 0].set_ylabel('Irradiance (W/m$^2$)', fontsize=20)
    ax[0, 0].set_title('425 $\mathrm{\AA}$', fontsize=20)
    ax[0, 0].tick_params(axis='both', labelsize=17)
    # 725 A (index 24)
    ax[0, 1].scatter(sortedP, correspondingFism2Irr[sortIndsP, 24], label='FISM2')
    ax[0, 1].scatter(sortedP, ensemble_average_NeuvacIrr[sortIndsP, 24], label='NEUVAC', color='tab:orange')
    ax[0, 1].legend(loc='best', fontsize=16)
    ax[0, 1].set_title('725 $\mathrm{\AA}$', fontsize=20)
    ax[0, 1].tick_params(axis='both', labelsize=17)
    # 975 (index 31)
    ax[0, 2].scatter(sortedP, correspondingFism2Irr[sortIndsP, 31], label='FISM2')
    ax[0, 2].scatter(sortedP, ensemble_average_NeuvacIrr[sortIndsP, 31], label='NEUVAC', color='tab:orange')
    ax[0, 2].legend(loc='best', fontsize=16)
    ax[0, 2].set_title('975 $\mathrm{\AA}$', fontsize=20)
    ax[0, 2].tick_params(axis='both', labelsize=17)
    # SECOND (BOTTOM) ROW:
    # 430 A (index 9)
    ax[1, 0].scatter(sortedP, correspondingIrrFISM2StanBands[sortIndsP, 9], label='FISM2')
    ax[1, 0].scatter(sortedP, ensemble_average_NeuvacIrrSolomon[sortIndsP, 9], label='NEUVAC-22', color='tab:orange')
    ax[1, 0].legend(loc='best', fontsize=16)
    ax[1, 0].set_xlabel('P (sfu)', fontsize=20)
    ax[1, 0].set_ylabel('Irradiance (W/m$^2$)', fontsize=20)
    ax[1, 0].set_title('430 $\mathrm{\AA}$', fontsize=20)
    ax[1, 0].tick_params(axis='both', labelsize=17)
    # 724 A (11)
    ax[1, 1].scatter(sortedP, correspondingIrrFISM2StanBands[sortIndsP, 11], label='FISM2')
    ax[1, 1].scatter(sortedP, ensemble_average_NeuvacIrrSolomon[sortIndsP, 11], label='NEUVAC-22', color='tab:orange')
    ax[1, 1].legend(loc='best', fontsize=16)
    ax[1, 1].set_xlabel('P (sfu)', fontsize=20)
    ax[1, 1].set_title('724 $\mathrm{\AA}$', fontsize=20)
    ax[1, 1].tick_params(axis='both', labelsize=17)
    # 981 A (19)
    ax[1, 2].scatter(sortedP, correspondingIrrFISM2StanBands[sortIndsP, 19], label='FISM2-S')
    ax[1, 2].scatter(sortedP, ensemble_average_NeuvacIrrSolomon[sortIndsP, 19], label='NEUVAC-22', color='tab:orange')
    ax[1, 2].legend(loc='best', fontsize=16)
    ax[1, 2].set_xlabel('P (sfu)', fontsize=20)
    ax[1, 2].set_title('981 $\mathrm{\AA}$', fontsize=20)
    ax[1, 2].tick_params(axis='both', labelsize=17)
    # Save the figure:
    fig.tight_layout()
    fig.suptitle('Solar Irradiance vs. P', fontsize=25, fontweight='bold')
    fig.subplots_adjust(top=0.88)
    plt.savefig(results_dir + 'irradiance_vs_P_6_bands_with_NEUVAC.png', dpi=300)

    # 3A: Plot Solar Spectra in Low and High Solar Activity
    euvacTable = euvac.euvacTable
    mids = 0.5 * (euvacTable[:, 1] + euvacTable[:, 2])
    bandInds = np.where(euvacTable[:, 1] != euvacTable[:, 2])[0]
    xPos = np.append(euvacTable[:, 1][bandInds], euvacTable[:, 2][-1]) # np.append(euvacTable[:, 1], euvacTable[:, 2][-1])
    sortInds = np.argsort(xPos)
    xPosSorted = xPos[sortInds]

    fig, ax = plt.subplots(nrows=1, ncols=2, sharex=True, sharey=True)
    # i: Low Activity:
    chosenDateLow = datetime(2004, 3, 30) # 1985-11-04; Beginning of Solar Cycle 21
    idx, val = toolbox.find_nearest(times, chosenDateLow)
    ax[0].stairs(values=correspondingFism2Irr[idx, :][bandInds], edges=xPosSorted, label='FISM2', lw=9) # sortInds[:-1]
    # lowerBound = ensemble_average_NeuvacIrr[idx, :][bandInds] - ensemble_stddev_NeuvacIrr[idx, :][bandInds]
    # upperBound = ensemble_average_NeuvacIrr[idx, :][bandInds] + ensemble_stddev_NeuvacIrr[idx, :][bandInds]
    # ax[0].fill_between(xPosSorted, np.hstack([lowerBound, lowerBound[-1][None]]), np.hstack([upperBound, upperBound[-1][None]]), step="post", color='orange', alpha=0.6)
    ax[0].stairs(values=ensemble_average_NeuvacIrr[idx, :][bandInds], edges=xPosSorted, label='NEUVAC-37 (n='+str(iterations)+')', lw=3)
    ax[0].errorbar(xPosSorted[:-1]+25, ensemble_average_NeuvacIrr[idx, :][bandInds], yerr=ensemble_stddev_NeuvacIrr[idx, :][bandInds], capsize=7, capthick=2, fmt='o', color='orange')
    ax[0].stairs(values=euvacIrr[idx, :][bandInds], edges=xPosSorted, label='EUVAC', lw=3)
    ax[0].stairs(values=heuvacIrr[idx, :][bandInds], edges=xPosSorted, label='HEUVAC-37', lw=3)
    ax[0].set_yscale('log')
    ax[0].legend(loc='best')
    ax[0].grid()
    ax[0].set_xlabel('Wavelength ($\mathrm{\AA}$)')
    ax[0].set_ylabel('Irradiance (W/m$^2$)')
    secax = ax[0].secondary_yaxis('right', functions=(energy, power))
    secax.set_ylabel('Radiant Exposure (J/m$^2$)', rotation=-90, labelpad=15)
    ax[0].set_title('Solar Spectra during Low Solar Activity ('+str(chosenDateLow)[:-9]+')')
    # ii: High Activity:
    chosenDateHigh = datetime(2002, 2, 8) # 1991-01-31; Peak of Solar Cycle 21
    idx, val = toolbox.find_nearest(times, chosenDateHigh)
    ax[1].stairs(values=correspondingFism2Irr[idx, :][bandInds], edges=xPosSorted, label='FISM2', lw=9)
    ax[1].stairs(values=ensemble_average_NeuvacIrr[idx, :][bandInds], edges=xPosSorted, label='NEUVAC-37 (n='+str(iterations)+')', lw=3)
    ax[1].errorbar(xPosSorted[:-1] + 25, ensemble_average_NeuvacIrr[idx, :][bandInds],
                   yerr=ensemble_stddev_NeuvacIrr[idx, :][bandInds], capsize=7, capthick=2, fmt='o', color='orange')
    ax[1].stairs(values=euvacIrr[idx, :][bandInds], edges=xPosSorted, label='EUVAC', lw=3)
    ax[1].stairs(values=heuvacIrr[idx, :][bandInds], edges=xPosSorted, label='HEUVAC-37', lw=3)
    ax[1].set_yscale('log')
    ax[1].legend(loc='best')
    ax[1].grid()
    ax[1].set_xlabel('Wavelength ($\mathrm{\AA}$)')
    ax[1].set_ylabel('Irradiance (W/m$^2$)')
    secax = ax[1].secondary_yaxis('right', functions=(energy, power))
    secax.set_ylabel('Radiant Exposure (J/m$^2$)', rotation=-90, labelpad=15)
    ax[1].set_title('Solar Spectra during High Solar Activity ('+str(chosenDateHigh)[:-9]+')')
    plt.savefig(results_dir+'sample_spectra_low_and_high_solar_activity.png', dpi=300)

    # 3B: Plot sample TIME SERIES during Low and High Solar Activity (3 bands only):
    fig, axs = plt.subplots(nrows=2, ncols=3)
    lowSolarTimeBounds = [chosenDateLow-timedelta(days=40), chosenDateLow+timedelta(days=41)]
    lowSolarTimeInds = np.where((times >= lowSolarTimeBounds[0]) & (times <= lowSolarTimeBounds[-1]))[0]
    lowSolarTimes = times[lowSolarTimeInds]
    highSolarTimeBounds = [chosenDateHigh - timedelta(days=40), chosenDateHigh + timedelta(days=41)]
    highSolarTimeInds = np.where((times >= highSolarTimeBounds[0]) & (times <= highSolarTimeBounds[-1]))[0]
    highSolarTimes = times[highSolarTimeInds]
    # Top-left: Low Solar Activity 75 A
    axs[0, 0].plot(lowSolarTimes, correspondingFism2Irr[lowSolarTimeInds, 12], label='FISM2', lw=3)
    axs[0, 0].plot(lowSolarTimes, ensemble_average_NeuvacIrr[lowSolarTimeInds, 12], label='NEUVAC (n='+str(iterations)+')', lw=3)
    axs[0, 0].plot(lowSolarTimes, euvacIrr[lowSolarTimeInds, 12], label='EUVAC', lw=3)
    axs[0, 0].plot(lowSolarTimes, heuvacIrr[lowSolarTimeInds, 12], label='HEUVAC', lw=3)
    axs[0, 0].set_ylabel('Irradiance (W/m$^2$)')
    axs[0, 0].set_title('425 $\mathrm{\AA}$')
    axs[0, 0].legend(loc='best')
    axs[0, 0].set_xticklabels(axs[0, 0].get_xticklabels(), rotation=45, ha='right')
    # Top-middle: Low Solar Activity 475 A
    axs[0, 1].plot(lowSolarTimes, correspondingFism2Irr[lowSolarTimeInds, 24], label='FISM2', lw=3)
    axs[0, 1].plot(lowSolarTimes, ensemble_average_NeuvacIrr[lowSolarTimeInds, 24], label='NEUVAC (n='+str(iterations)+')', lw=3)
    axs[0, 1].plot(lowSolarTimes, euvacIrr[lowSolarTimeInds, 24], label='EUVAC', lw=3)
    axs[0, 1].plot(lowSolarTimes, heuvacIrr[lowSolarTimeInds, 24], label='HEUVAC', lw=3)
    axs[0, 1].set_title('725 $\mathrm{\AA}$')
    axs[0, 1].legend(loc='best')
    axs[0, 1].set_xticklabels(axs[0, 1].get_xticklabels(), rotation=45, ha='right')
    # Top-right: Low Solar Activity 1025 A
    axs[0, 2].plot(lowSolarTimes, correspondingFism2Irr[lowSolarTimeInds, 31], label='FISM2', lw=3)
    axs[0, 2].plot(lowSolarTimes, ensemble_average_NeuvacIrr[lowSolarTimeInds, 31], label='NEUVAC (n='+str(iterations)+')', lw=3)
    axs[0, 2].plot(lowSolarTimes, euvacIrr[lowSolarTimeInds, 31], label='EUVAC', lw=3)
    axs[0, 2].plot(lowSolarTimes, heuvacIrr[lowSolarTimeInds, 31], label='HEUVAC', lw=3)
    axs[0, 2].set_title('975 $\mathrm{\AA}$')
    axs[0, 2].legend(loc='best')
    axs[0, 2].set_xticklabels(axs[0, 2].get_xticklabels(), rotation=45, ha='right')
    # Bottom-left: High Solar Activity 75 A
    axs[1, 0].plot(highSolarTimes, correspondingFism2Irr[highSolarTimeInds, 12], label='FISM2', lw=3)
    axs[1, 0].plot(highSolarTimes, ensemble_average_NeuvacIrr[highSolarTimeInds, 12], label='NEUVAC (n='+str(iterations)+')', lw=3)
    axs[1, 0].plot(highSolarTimes, euvacIrr[highSolarTimeInds, 12], label='EUVAC', lw=3)
    axs[1, 0].plot(highSolarTimes, heuvacIrr[highSolarTimeInds, 12], label='HEUVAC', lw=3)
    axs[1, 0].set_ylabel('Irradiance (W/m$^2$)')
    axs[1, 0].set_title('425 $\mathrm{\AA}$')
    axs[1, 0].legend(loc='best')
    axs[1, 0].set_xticklabels(axs[1, 0].get_xticklabels(), rotation=45, ha='right')
    # Bottom-middle: High Solar Activity 475 A
    axs[1, 1].plot(highSolarTimes, correspondingFism2Irr[highSolarTimeInds, 24], label='FISM2', lw=3)
    axs[1, 1].plot(highSolarTimes, ensemble_average_NeuvacIrr[highSolarTimeInds, 24], label='NEUVAC (n='+str(iterations)+')', lw=3)
    axs[1, 1].plot(highSolarTimes, euvacIrr[highSolarTimeInds, 24], label='EUVAC', lw=3)
    axs[1, 1].plot(highSolarTimes, heuvacIrr[highSolarTimeInds, 24], label='HEUVAC', lw=3)
    axs[1, 1].set_title('725 $\mathrm{\AA}$')
    axs[1, 1].legend(loc='best')
    axs[1, 1].set_xticklabels(axs[1, 1].get_xticklabels(), rotation=45, ha='right')
    # Bottom-right: High Solar Activity 1025 A
    axs[1, 2].plot(highSolarTimes, correspondingFism2Irr[highSolarTimeInds, 31], label='FISM2', lw=3)
    axs[1, 2].plot(highSolarTimes, ensemble_average_NeuvacIrr[highSolarTimeInds, 31], label='NEUVAC (n='+str(iterations)+')', lw=3)
    axs[1, 2].plot(highSolarTimes, euvacIrr[highSolarTimeInds, 31], label='EUVAC', lw=3)
    axs[1, 2].plot(highSolarTimes, heuvacIrr[highSolarTimeInds, 31], label='HEUVAC', lw=3)
    axs[1, 2].set_title('975 $\mathrm{\AA}$')
    axs[1, 2].legend(loc='best')
    axs[1, 2].set_xticklabels(axs[1, 2].get_xticklabels(), rotation=45, ha='right')
    # Save the figure:
    fig.tight_layout()
    fig.suptitle('Irradiance During Low Solar Activity ('+str(lowSolarTimeBounds[0])[:-9]+' to '+
                 str(lowSolarTimeBounds[-1])[:-9]+') and High Solar Activity ('+str(highSolarTimeBounds[0])[:-9]+' to '
                 +str(highSolarTimeBounds[-1])[:-9]+')\n', fontsize=16, fontweight='bold')
    fig.subplots_adjust(top=0.9)
    plt.savefig(results_dir+'sampleTimeSeriesSpectra_Low_and_High_Solar_Activity.png', dpi=300)

    # 3C: Simple Time Series (with uncertainty bands) of NEUVAC with other models, during Solar Cycle 25:
    cycle25inds = np.where(times >= datetime(2019, 12, 1))[0]
    cycle25times = times[cycle25inds]
    ind = 12
    fig = plt.figure()
    plt.plot(cycle25times, correspondingFism2Irr[cycle25inds, ind], label='FISM2')
    plt.fill_between(cycle25times, (ensemble_average_NeuvacIrr-ensemble_stddev_NeuvacIrr)[cycle25inds, ind],
                              (ensemble_average_NeuvacIrr+ensemble_stddev_NeuvacIrr)[cycle25inds, ind],
                     color='orange', alpha=0.6)
    plt.plot(cycle25times, ensemble_average_NeuvacIrr[cycle25inds, ind], label='NEUVAC-37 (n='+str(iterations)+')')
    plt.plot(cycle25times, euvacIrr[cycle25inds, ind], label='EUVAC')
    plt.plot(cycle25times, heuvacIrr[cycle25inds, ind], label='HEUVAC-37')
    plt.legend(loc='best', fontsize=25)
    plt.xlabel('Time', fontsize=27)
    plt.ylabel('Irradiance (W/m$^2$)', fontsize=27)
    plt.title('Solar Irradiance Centered at '+str(mids[ind])+' $\mathrm{\AA}$ (Solar Cycle 25)', fontsize=32)
    plt.tick_params(axis='both', labelsize=27)
    plt.xticks(rotation=45)
    plt.tight_layout()
    plt.savefig(results_dir+'sampleTimeSeries_cycle25.png', dpi=300)
    # ==================================================================================================================
    # TODO 4: [STATISTICS OF] PERTURBATIONS (FOR NEUVAC)

    # ==================================================================================================================
    # 5: BEHAVIOR OF NORMALIZED RESIDUALS (SQUARE DIFFERENCES)

    # i: SQDF vs. Solar activity (F10.7) - 3 Bands:
    # sqdf_NEUVAC_75 = toolbox.squareDiff(ensemble_average_NeuvacIrr[:, 0], correspondingFism2Irr[:, 0])
    # sqdf_NEUVAC_475 = toolbox.squareDiff(ensemble_average_NeuvacIrr[:, 14], correspondingFism2Irr[:, 14])
    # sqdf_NEUVAC_1025 = toolbox.squareDiff(ensemble_average_NeuvacIrr[:, -1], correspondingFism2Irr[:, -1])
    # sqdf_EUVAC_75 = toolbox.squareDiff(euvacIrr[:, 0], correspondingFism2Irr[:, 0])
    # sqdf_EUVAC_475 = toolbox.squareDiff(euvacIrr[:, 14], correspondingFism2Irr[:, 14])
    # sqdf_EUVAC_1025 = toolbox.squareDiff(euvacIrr[:, -1], correspondingFism2Irr[:, -1])
    # sqdf_HEUVAC_75 = toolbox.squareDiff(heuvacIrr[:, 0], correspondingFism2Irr[:, 0])
    # sqdf_HEUVAC_475 = toolbox.squareDiff(heuvacIrr[:, 14], correspondingFism2Irr[:, 14])
    # sqdf_HEUVAC_1025 = toolbox.squareDiff(heuvacIrr[:, -1], correspondingFism2Irr[:, -1])
    sortF107 = np.argsort(F107)
    # fig, axs = plt.subplots(nrows=1, ncols=3)
    # axs[0].scatter(F107[sortF107], sqdf_NEUVAC_75[sortF107], color='orange', label='NEUVAC (n='+str(iterations)+')', alpha=0.6)
    # axs[0].scatter(F107[sortF107], sqdf_EUVAC_75[sortF107], color='green', label='EUVAC', alpha=0.6)
    # axs[0].scatter(F107[sortF107], sqdf_HEUVAC_75[sortF107], color='red', label='HEUVAC', alpha=0.6)
    # axs[0].set_xlabel('F10.7 (sfu)')
    # axs[0].set_ylabel('Squared Difference from FISM2 (W/m$^2$)')
    # axs[0].set_title('75 $\mathrm{\AA}$')
    # axs[0].set_yscale('log')
    # axs[0].legend(loc='best')
    # axs[1].scatter(F107[sortF107], sqdf_NEUVAC_475[sortF107], color='orange', label='NEUVAC (n='+str(iterations)+')', alpha=0.6)
    # axs[1].scatter(F107[sortF107], sqdf_EUVAC_475[sortF107], color='green', label='EUVAC', alpha=0.6)
    # axs[1].scatter(F107[sortF107], sqdf_HEUVAC_475[sortF107], color='red', label='HEUVAC', alpha=0.6)
    # axs[1].set_xlabel('F10.7 (sfu)')
    # axs[1].set_title('475 $\mathrm{\AA}$')
    # axs[1].set_yscale('log')
    # axs[1].legend(loc='best')
    # axs[2].scatter(F107[sortF107], sqdf_NEUVAC_1025[sortF107], color='orange', label='NEUVAC (n='+str(iterations)+')', alpha=0.6)
    # axs[2].scatter(F107[sortF107], sqdf_EUVAC_1025[sortF107], color='green', label='EUVAC', alpha=0.6)
    # axs[2].scatter(F107[sortF107], sqdf_HEUVAC_1025[sortF107], color='red', label='HEUVAC', alpha=0.6)
    # axs[2].set_xlabel('F10.7 (sfu)')
    # axs[2].set_title('1025 $\mathrm{\AA}$')
    # axs[2].set_yscale('log')
    # axs[2].legend(loc='best')
    # plt.savefig(results_dir + 'SQDF_by_F107.png', dpi=300)

    # ii: MAPE vs. Wavelength Band:
    mapeNEUVAC = []
    for i in range(neuvacIrr.shape[1]):
        mapeNEUVAC.append(toolbox.mape(correspondingFism2Irr[:, i], ensemble_average_NeuvacIrr[:, i]) * 100)
    mapeEUVAC = []
    for i in range(neuvacIrr.shape[1]):
        mapeEUVAC.append(toolbox.mape(correspondingFism2Irr[:, i], euvacIrr[:, i]) * 100)
    mapeHEUVAC = []
    for i in range(neuvacIrr.shape[1]):
        mapeHEUVAC.append(toolbox.mape(correspondingFism2Irr[:, i], heuvacIrr[:, i]) * 100)
    sortWav = np.argsort(heuvac_wav)
    plt.figure()
    plt.plot(heuvac_wav[sortWav], np.asarray(mapeNEUVAC)[sortWav], color='orange', marker='o', label='NEUVAC (n='+str(iterations)+')')
    plt.plot(heuvac_wav[sortWav], np.asarray(mapeEUVAC)[sortWav], color='green', marker='o', label='EUVAC')
    plt.plot(heuvac_wav[sortWav], np.asarray(mapeHEUVAC)[sortWav], color='red', marker='o', label='HEUVAC')
    plt.yscale('log')
    plt.xlabel('Wavelength ($\mathrm{\AA}$)')
    plt.ylabel('MAPE (%)')
    plt.title('MAPE for NEUVAC-37, EUVAC, and HEUVAC-37 vs. Wavelength')
    plt.legend(loc='best')
    plt.grid()
    plt.savefig(results_dir + 'MAPE_by_band.png', dpi=300)
    # Stair plot of the above:
    plt.figure()
    plt.stairs(values=np.asarray(mapeNEUVAC)[sortWav][bandInds], edges=xPosSorted, color='orange', label='NEUVAC (n='+str(iterations)+')', lw=9)
    plt.stairs(values=np.asarray(mapeEUVAC)[sortWav][bandInds], edges=xPosSorted, color='green',
               label='EUVAC', lw=9)
    plt.stairs(values=np.asarray(mapeHEUVAC)[sortWav][bandInds], edges=xPosSorted, color='red',
               label='HEUVAC', lw=9)
    plt.yscale('log')
    plt.xlabel('Wavelength ($\mathrm{\AA}$)', fontsize=25)
    plt.ylabel('MAPE (%)', fontsize=25)
    plt.title('MAPE for NEUVAC-37, EUVAC, and HEUVAC-37 vs. Wavelength', fontsize=30)
    plt.tick_params(axis='both', labelsize=25)
    plt.legend(loc='best', fontsize=25)
    plt.grid()
    plt.tight_layout()
    plt.savefig(results_dir + 'MAPE_by_band_stairs.png', dpi=300)

    # ==================================================================================================================
    # 6: BEHAVIOR OF PERCENT DEVIATIONS FROM FISM2
    # i: Distribution of Percent Deviation
    vfunc = np.vectorize(toolbox.percDev)
    NEUVAC_resids = vfunc(ensemble_average_NeuvacIrr, correspondingFism2Irr)
    EUVAC_resids = vfunc(euvacIrr, correspondingFism2Irr)
    HEUVAC_resids = vfunc(heuvacIrr, correspondingFism2Irr)
    NEUVAC_resids_flat = np.ravel(NEUVAC_resids)
    EUVAC_resids_flat = np.ravel(EUVAC_resids)
    HEUVAC_resids_flat = np.ravel(HEUVAC_resids)
    # bins = np.linspace(np.min([np.nanpercentile(NEUVAC_resids, 25), np.nanpercentile(EUVAC_resids, 25), np.nanpercentile(HEUVAC_resids[HEUVAC_resids != -np.inf], 25)]),
    #                    np.max([np.nanpercentile(NEUVAC_resids, 75), np.nanpercentile(EUVAC_resids, 75), np.nanpercentile(HEUVAC_resids, 75)]), num=100)
    # bins = np.linspace(np.nanmin(NEUVAC_resids_flat), np.nanmax(NEUVAC_resids_flat), num=100)
    bins = np.linspace(-25., 25., num=100)

    # myLabels = ['Percent Deviation (%)', 'Count', 'NEUVAC Percent Deviation from FISM2']
    # figHist = toolbox.plotHist(NEUVAC_resids_flat, bins=bins, color='orange', saveLoc=results_dir + 'NEUVAC_percDev.png', labels=myLabels, density=False)

    # Percent Deviation for just three bins: 425 A (12), 725 A (24), and 975 A (31)
    NEUVAC_resids_425 = toolbox.percDev(ensemble_average_NeuvacIrr[:, 12], correspondingFism2Irr[:, 12])
    myLabels425 = ['Percent Deviation (%)', 'Count', r'NEUVAC Percent Deviation from FISM2: 425 $\mathrm{\AA}$']
    figHist425 = toolbox.plotHist(NEUVAC_resids_425, bins=bins, color='orange', saveLoc=results_dir + 'NEUVAC_percDev_425.png', labels=myLabels425)

    NEUVAC_resids_725 = toolbox.percDev(ensemble_average_NeuvacIrr[:, 24], correspondingFism2Irr[:, 24])
    myLabels725 = ['Percent Deviation (%)', 'Count', r'NEUVAC Percent Deviation from FISM2: 725 $\mathrm{\AA}$']
    figHist725 = toolbox.plotHist(NEUVAC_resids_725, bins=bins, color='orange',
                                  saveLoc=results_dir + 'NEUVAC_percDev_725.png', labels=myLabels725)

    NEUVAC_resids_975 = toolbox.percDev(ensemble_average_NeuvacIrr[:, 31], correspondingFism2Irr[:, 31])
    myLabels975 = ['Percent Deviation (%)', 'Count', r'NEUVAC Percent Deviation from FISM2: 975 $\mathrm{\AA}$']
    figHist975 = toolbox.plotHist(NEUVAC_resids_975, bins=bins, color='orange',
                                  saveLoc=results_dir + 'NEUVAC_percDev_975.png', labels=myLabels975)

    # ii: Behavior of Percent Deviations as a Function Solar Activity (F10.7)
    NEUVAC_resids_425 = NEUVAC_resids[:, 12]
    NEUVAC_resids_725 = NEUVAC_resids[:, 24]
    NEUVAC_resids_975 = NEUVAC_resids[:, 31]
    EUVAC_resids_425 = EUVAC_resids[:, 12]
    EUVAC_resids_725 = EUVAC_resids[:, 24]
    EUVAC_resids_975 = EUVAC_resids[:, 31]
    HEUVAC_resids_425 = HEUVAC_resids[:, 12]
    HEUVAC_resids_725 = HEUVAC_resids[:, 24]
    HEUVAC_resids_975 = HEUVAC_resids[:, 31]
    fig, axs = plt.subplots(nrows=1, ncols=3, sharey=True)
    axs[0].scatter(F107[sortF107], NEUVAC_resids_425[sortF107], color='orange', label='NEUVAC-37 (n='+str(iterations)+')')
    axs[0].scatter(F107[sortF107], EUVAC_resids_425[sortF107], color='green', label='EUVAC')
    axs[0].scatter(F107[sortF107], HEUVAC_resids_425[sortF107], color='red', label='HEUVAC-37')
    axs[0].axhline(y=0, linewidth=3, linestyle='--', color='grey')
    axs[0].set_xlabel('F10.7 (sfu)', fontsize=30)
    axs[0].set_ylabel('Percent Deviation from FISM2 (%)', fontsize=30)
    axs[0].set_title('425 $\mathrm{\AA}$', fontsize=35)
    axs[0].legend(loc='best', fontsize=18)
    axs[0].tick_params(axis='both', labelsize=22)
    axs[1].scatter(F107[sortF107], NEUVAC_resids_725[sortF107], color='orange', label='NEUVAC-37 (n='+str(iterations)+')')
    axs[1].scatter(F107[sortF107], EUVAC_resids_725[sortF107], color='green', label='EUVAC')
    axs[1].scatter(F107[sortF107], HEUVAC_resids_725[sortF107], color='red', label='HEUVAC-37')
    axs[1].axhline(y=0, linewidth=3, linestyle='--', color='grey')
    axs[1].set_xlabel('F10.7 (sfu)', fontsize=30)
    axs[1].set_title('725 $\mathrm{\AA}$', fontsize=35)
    axs[1].legend(loc='best', fontsize=18)
    axs[1].tick_params(axis='both', labelsize=22)
    axs[2].scatter(F107[sortF107], NEUVAC_resids_975[sortF107], color='orange', label='NEUVAC-37 (n='+str(iterations)+')')
    axs[2].scatter(F107[sortF107], EUVAC_resids_975[sortF107], color='green', label='EUVAC')
    axs[2].scatter(F107[sortF107], HEUVAC_resids_975[sortF107], color='red', label='HEUVAC-37')
    axs[2].axhline(y=0, linewidth=3, linestyle='--', color='grey')
    axs[2].set_xlabel('F10.7 (sfu)', fontsize=30)
    axs[2].set_title('975 $\mathrm{\AA}$', fontsize=35)
    axs[2].legend(loc='best', fontsize=18)
    axs[2].tick_params(axis='both', labelsize=22)
    plt.tight_layout()
    plt.savefig(results_dir + 'percDev_by_F107.png', dpi=300)

    # ==================================================================================================================
    # 7: INTEGRATED ENERGY (ACROSS THE SUN-FACING SIDE OF THE EARTH)
    Re = 6.3781e6 # Nominal Terrestrial Radius in meters (Mamajek, et al. 2015: https://arxiv.org/abs/1510.07674)
    A = np.pi * (Re**2) # Earth cross-sectional area in m^2

    fism2Sums = np.zeros(neuvacIrr.shape[0])
    for i in range(len(fism2Sums)):
        fism2Sums[i] = np.sum(correspondingFism2Irr[i, :])
    fism2IntegEnergy = A*fism2Sums
    neuvacSums = np.zeros_like(fism2Sums)
    for i in range(len(fism2Sums)):
        neuvacSums[i] = np.sum(ensemble_average_NeuvacIrr[i, :])
    neuvacIntegEnergy = A*neuvacSums
    euvacSums = np.zeros_like(fism2Sums)
    for i in range(len(fism2Sums)):
        euvacSums[i] = np.sum(euvacIrr[i, :])
    euvacIntegEnergy = A*euvacSums
    heuvacSums = np.zeros_like(fism2Sums)
    for i in range(len(fism2Sums)):
        heuvacSums[i] = np.sum(heuvacIrr[i, :])
    heuvacIntegEnergy = A * heuvacSums

    fig, axs = plt.subplots(nrows=3, ncols=1)
    #
    lowSolarTimeBoundsLonger = [chosenDateLow - timedelta(days=181), chosenDateLow + timedelta(days=182)]
    lowSolarTimeIndsLonger = np.where((times >= lowSolarTimeBoundsLonger[0]) & (times <= lowSolarTimeBoundsLonger[-1]))[0]
    lowSolarTimesLonger = times[lowSolarTimeIndsLonger]
    highSolarTimeBoundsLonger = [chosenDateHigh - timedelta(days=181), chosenDateHigh + timedelta(days=182)]
    highSolarTimeIndsLonger = np.where((times >= highSolarTimeBoundsLonger[0]) & (times <= highSolarTimeBoundsLonger[-1]))[0]
    highSolarTimesLonger = times[highSolarTimeIndsLonger]
    #
    axs[0].plot(lowSolarTimesLonger, fism2IntegEnergy[lowSolarTimeIndsLonger] / (1e9), label='FISM2')
    axs[0].plot(lowSolarTimesLonger, neuvacIntegEnergy[lowSolarTimeIndsLonger] / (1e9), label='NEUVAC (n=' + str(iterations) + ')')
    axs[0].plot(lowSolarTimesLonger, euvacIntegEnergy[lowSolarTimeIndsLonger] / (1e9), label='EUVAC')
    axs[0].plot(lowSolarTimesLonger, heuvacIntegEnergy[lowSolarTimeIndsLonger] / (1e9), label='HEUVAC')
    axs[0].set_ylabel('Integrated Energy (GW)')
    axs[0].legend(loc='best')
    axs[0].set_title('Low Solar Activity: '+str(lowSolarTimeBoundsLonger[0])[:-9]+' to '+str(lowSolarTimeBoundsLonger[-1])[:-9])
    #
    axs[1].plot(highSolarTimesLonger, fism2IntegEnergy[highSolarTimeIndsLonger] / (1e9), label='FISM2')
    axs[1].plot(highSolarTimesLonger, neuvacIntegEnergy[highSolarTimeIndsLonger] / (1e9), label='NEUVAC (n=' + str(iterations) + ')')
    axs[1].plot(highSolarTimesLonger, euvacIntegEnergy[highSolarTimeIndsLonger] / (1e9), label='EUVAC')
    axs[1].plot(highSolarTimesLonger, heuvacIntegEnergy[highSolarTimeIndsLonger] / (1e9), label='HEUVAC')
    axs[1].set_ylabel('Integrated Energy (GW)')
    axs[1].legend(loc='best')
    axs[1].set_title('High Solar Activity: '+str(highSolarTimeBoundsLonger[0])[:-9]+' to '+str(highSolarTimeBoundsLonger[-1])[:-9])
    #
    axs[2].plot(times, fism2IntegEnergy / (1e9), label='FISM2')
    axs[2].plot(times, neuvacIntegEnergy / (1e9), label='NEUVAC (n='+str(iterations)+')')
    axs[2].plot(times, euvacIntegEnergy / (1e9), label='EUVAC')
    axs[2].plot(times, heuvacIntegEnergy / (1e9), label='HEUVAC')
    axs[2].set_xlabel('Time')
    axs[2].set_ylabel('Integrated Energy (GW)')
    axs[2].legend(loc='best')
    axs[2].set_title('Solar Cycle 20 through Ascending Phase of Solar Cycle 25')
    # Save the figure
    fig.tight_layout()
    fig.suptitle('Integrated Earth-Incident Energy in b37 Bins', fontsize=16, fontweight='bold')
    fig.subplots_adjust(top=0.9)
    plt.savefig(results_dir + 'integrated_energy.png', dpi=300)

    # TABLE: Compute and print the MAPE for all models during low, moderate, and high solar activity (see Jin, et al. 2021 for definitions for low/moderate/high: https://agupubs.onlinelibrary.wiley.com/doi/10.1029/2020JA028932)
    lowSolarActivityInds = np.where(F107 < 80)[0]
    modSolarActivityInds = np.where((F107 >= 80) & (F107 < 120))[0]
    highSolarActivityInds = np.where(F107 >= 120)[0]
    neuvacEnergyLowMape = toolbox.mape(neuvacIntegEnergy[lowSolarActivityInds], fism2IntegEnergy[lowSolarActivityInds]) * 100
    euvacEnergyLowMape = toolbox.mape(euvacIntegEnergy[lowSolarActivityInds], fism2IntegEnergy[lowSolarActivityInds]) * 100
    heuvacEnergyLowMape = toolbox.mape(heuvacIntegEnergy[lowSolarActivityInds], fism2IntegEnergy[lowSolarActivityInds]) * 100
    neuvacEnergyModMape = toolbox.mape(neuvacIntegEnergy[modSolarActivityInds], fism2IntegEnergy[modSolarActivityInds]) * 100
    euvacEnergyModMape = toolbox.mape(euvacIntegEnergy[modSolarActivityInds], fism2IntegEnergy[modSolarActivityInds]) * 100
    heuvacEnergyModMape = toolbox.mape(heuvacIntegEnergy[modSolarActivityInds], fism2IntegEnergy[modSolarActivityInds]) * 100
    neuvacEnergyHighMape = toolbox.mape(neuvacIntegEnergy[highSolarActivityInds], fism2IntegEnergy[highSolarActivityInds]) * 100
    euvacEnergyHighMape = toolbox.mape(euvacIntegEnergy[highSolarActivityInds], fism2IntegEnergy[highSolarActivityInds]) * 100
    heuvacEnergyHighMape = toolbox.mape(heuvacIntegEnergy[highSolarActivityInds], fism2IntegEnergy[highSolarActivityInds]) * 100
    print('NEUVAC MAPE: ' + str(np.round(neuvacEnergyLowMape, 2)) + '% (low activity), ' + str(
        np.round(neuvacEnergyModMape, 2)) + '% (moderate activity), ' + str(
        np.round(neuvacEnergyHighMape, 2)) + '% (high activity)')
    print('EUVAC MAPE: ' + str(np.round(euvacEnergyLowMape, 2)) + '% (low activity), ' + str(
        np.round(euvacEnergyModMape, 2)) + '% (moderate activity), ' + str(
        np.round(euvacEnergyHighMape, 2)) + '% (high activity)')
    print('HEUVAC MAPE: ' + str(np.round(heuvacEnergyLowMape, 2)) + '% (low activity), ' + str(
        np.round(heuvacEnergyModMape, 2)) + '% (moderate activity), ' + str(
        np.round(heuvacEnergyHighMape, 2)) + '% (high activity)')
    # ==================================================================================================================
    # ==================================================================================================================
    # ==================================================================================================================
    #
    #
    #
    #
    #
    #
    #
    #
    #
    #
    #
    # ==================================================================================================================
    # 8: ANALYSIS: SOLOMON BANDS (FISM2-S, NEUVAC-S, EUVAC-S, and HFG)
    # NOTE: In any analysis below, only 3 bands will be considered for plotting (25 A, 595 A, 1007 A)
    # ==================================================================================================================
    #
    #
    #
    #
    #
    #
    #
    #
    #
    #
    # ==================================================================================================================
    # ==================================================================================================================
    # ==================================================================================================================
    # 8A: Plot Solar Spectra in Low and High Solar Activity
    xPosSolomonInitial = 0.5 * (solomonTable['short'] + solomonTable['long'])
    xPosSolomon = np.append(xPosSolomonInitial, 1130.)
    bandWidths = np.append(solomonTable['long'], 1130.) - np.append(solomonTable['short'], solomonTable['long'][-1])
    halfBandWidths = 0.5* bandWidths
    # Constrain the bins; i.e. for the overlapping bins, sum everything in there together:
    xPosSolomonNew = []
    goodInds = np.array([0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 13, 16, 20, 21]) # badInds: 12, 14, 15, 17, 18
    FISM2S_sums = correspondingIrrFISM2StanBands[:, goodInds]
    NEUVAC_22_sums = ensemble_average_NeuvacIrrSolomon[:, goodInds]
    NEUVAC_22_sttdev = ensemble_stddev_NeuvacIrr[:, goodInds]
    EUVAC_22_sums = solomonIrrEUVAC[:, goodInds]
    HFG_sums = solomonIrrHFG[:, goodInds]
    j = 0
    for i in range(len(xPosSolomon)-1):
        if xPosSolomon[i] != xPosSolomon[i+1]:
            xPosSolomonNew.append(xPosSolomon[i])
            j += 1
        else:
            FISM2S_sums[:, j] += correspondingIrrFISM2StanBands[:, i]
            NEUVAC_22_sums[:, j] += ensemble_average_NeuvacIrrSolomon[:, i]
            NEUVAC_22_sttdev[:, j] *= 0.5 * ensemble_stddev_NeuvacIrr[:, i]
            EUVAC_22_sums[:, j] += solomonIrrEUVAC[:, i]
            HFG_sums[:, j] += solomonIrrHFG[:, i]
    # Continue:
    xPosSolomon = np.asarray(xPosSolomonNew)
    sortIndsSolomon = np.argsort(xPosSolomon)
    # The edges of the bins are what we want:
    xPosSortedSolomon = np.array([0.5, 4, 8, 18, 32, 70, 155, 224, 290, 320, 540, 650, 798, 913, 975, 987, 1027]) # xPosSolomon[sortIndsSolomon]
    fig, ax = plt.subplots(nrows=1, ncols=2, sharex=True, sharey=True)
    # i: Low Activity:
    chosenDateLow = datetime(2004, 3, 30)  # 1985-11-04 Beginning of Solar Cycle 21
    idx, val = toolbox.find_nearest(times, chosenDateLow)
    ax[0].stairs(values=FISM2S_sums[idx, :][sortIndsSolomon[:-1]], edges=xPosSortedSolomon, label='FISM2', lw=9)
    ax[0].errorbar(xPosSortedSolomon[:-1]+halfBandWidths[goodInds], NEUVAC_22_sums[idx, :],
                   yerr=NEUVAC_22_sttdev[idx, :], capsize=7, capthick=2, fmt='o', color='orange')
    ax[0].stairs(values=NEUVAC_22_sums[idx, :][sortIndsSolomon[:-1]], edges=xPosSortedSolomon,
                 label='NEUVAC-22 (n=' + str(iterations) + ')', lw=3)
    ax[0].stairs(values=EUVAC_22_sums[idx, :][sortIndsSolomon[:-1]], edges=xPosSortedSolomon, label='EUVAC-22', lw=3)
    ax[0].stairs(values=HFG_sums[idx, :][sortIndsSolomon[:-1]], edges=xPosSortedSolomon, label='HFG', lw=3, color='purple')
    ax[0].set_yscale('log')
    ax[0].legend(loc='best')
    ax[0].grid()
    ax[0].set_xlabel('Wavelength ($\mathrm{\AA}$)')
    ax[0].set_ylabel('Irradiance (W/m$^2$)')
    secax = ax[0].secondary_yaxis('right', functions=(energy, power))
    secax.set_ylabel('Radiant Exposure (J/m$^2$)', rotation=-90)
    ax[0].set_title('Solar Spectra during Low Solar Activity (' + str(chosenDateLow)[:-9] + ')')
    ax[0].set_ylim([1e-6, 1e-2])
    # ii: High Activity:
    chosenDateHigh = datetime(2002, 2, 8)  # 1991-01-31 Peak of Solar Cycle 21
    idx, val = toolbox.find_nearest(times, chosenDateHigh)
    ax[1].stairs(values=FISM2S_sums[idx, :][sortIndsSolomon[:-1]], edges=xPosSortedSolomon, label='FISM2', lw=9)
    ax[1].errorbar(xPosSortedSolomon[:-1]+halfBandWidths[goodInds], NEUVAC_22_sums[idx, :],
                   yerr=NEUVAC_22_sttdev[idx, :], capsize=7, capthick=2, fmt='o', color='orange')
    ax[1].stairs(values=NEUVAC_22_sums[idx, :][sortIndsSolomon[:-1]], edges=xPosSortedSolomon,
                 label='NEUVAC-22 (n=' + str(iterations) + ')', lw=3)
    ax[1].stairs(values=EUVAC_22_sums[idx, :][sortIndsSolomon[:-1]], edges=xPosSortedSolomon, label='EUVAC-22', lw=3)
    ax[1].stairs(values=HFG_sums[idx, :][sortIndsSolomon[:-1]], edges=xPosSortedSolomon, label='HFG', lw=3, color='purple')
    ax[1].set_yscale('log')
    ax[1].legend(loc='best')
    ax[1].grid()
    ax[1].set_xlabel('Wavelength ($\mathrm{\AA}$)')
    ax[1].set_ylabel('Irradiance (W/m$^2$)')
    secax = ax[1].secondary_yaxis('right', functions=(energy, power))
    secax.set_ylabel('Radiant Exposure (J/m$^2$)', rotation=-90)
    ax[1].set_title('Solar Spectra during High Solar Activity (' + str(chosenDateHigh)[:-9] + ')')
    plt.savefig(results_dir + 'sample_spectra_low_and_high_solar_activity_SOLOMON.png', dpi=300)

    # 8B: Plot sample TIME SERIES during Low and High Solar Activity (3 bands only): (9, 11, 19)
    fig, axs = plt.subplots(nrows=2, ncols=3)
    # Top-left: Low Solar Activity 430 A
    axs[0, 0].plot(lowSolarTimes, correspondingIrrFISM2StanBands[lowSolarTimeInds, 9], label='FISM2', lw=3)
    axs[0, 0].plot(lowSolarTimes, ensemble_average_NeuvacIrrSolomon[lowSolarTimeInds, 9], label='NEUVAC-22 (n='+str(iterations)+')', lw=3)
    axs[0, 0].plot(lowSolarTimes, solomonIrrEUVAC[lowSolarTimeInds, 9], label='EUVAC-22', lw=3)
    axs[0, 0].plot(lowSolarTimes, solomonIrrHFG[lowSolarTimeInds, 9], label='HFG', lw=3, color='purple')
    axs[0, 0].set_ylabel('Irradiance (W/m$^2$)')
    axs[0, 0].set_title('430 $\mathrm{\AA}$')
    axs[0, 0].legend(loc='best')
    axs[0, 0].set_xticklabels(axs[0, 0].get_xticklabels(), rotation=45, ha='right')
    # Top-middle: Low Solar Activity 724 A
    axs[0, 1].plot(lowSolarTimes, correspondingIrrFISM2StanBands[lowSolarTimeInds, 11], label='FISM2', lw=3)
    axs[0, 1].plot(lowSolarTimes, ensemble_average_NeuvacIrrSolomon[lowSolarTimeInds, 11], label='NEUVAC (n='+str(iterations)+')', lw=3)
    axs[0, 1].plot(lowSolarTimes, solomonIrrEUVAC[lowSolarTimeInds, 11], label='EUVAC', lw=3)
    axs[0, 1].plot(lowSolarTimes, solomonIrrHFG[lowSolarTimeInds, 11], label='HFG', lw=3, color='purple')
    axs[0, 1].set_title('724 $\mathrm{\AA}$')
    axs[0, 1].legend(loc='best')
    axs[0, 1].set_xticklabels(axs[0, 1].get_xticklabels(), rotation=45, ha='right')
    # Top-right: Low Solar Activity 981 A
    axs[0, 2].plot(lowSolarTimes, correspondingIrrFISM2StanBands[lowSolarTimeInds, 19], label='FISM2', lw=3)
    axs[0, 2].plot(lowSolarTimes, ensemble_average_NeuvacIrrSolomon[lowSolarTimeInds, 19], label='NEUVAC-22 (n='+str(iterations)+')', lw=3)
    axs[0, 2].plot(lowSolarTimes, solomonIrrEUVAC[lowSolarTimeInds, 19], label='EUVAC-22', lw=3)
    axs[0, 2].plot(lowSolarTimes, solomonIrrHFG[lowSolarTimeInds, 19], label='HFG', lw=3, color='purple')
    axs[0, 2].set_title('981 $\mathrm{\AA}$')
    axs[0, 2].legend(loc='best')
    axs[0, 2].set_xticklabels(axs[0, 2].get_xticklabels(), rotation=45, ha='right')
    # Bottom-left: High Solar Activity 430 A
    axs[1, 0].plot(highSolarTimes, correspondingIrrFISM2StanBands[highSolarTimeInds, 9], label='FISM2', lw=3)
    axs[1, 0].plot(highSolarTimes, ensemble_average_NeuvacIrrSolomon[highSolarTimeInds, 9], label='NEUVAC-22 (n='+str(iterations)+')', lw=3)
    axs[1, 0].plot(highSolarTimes, solomonIrrEUVAC[highSolarTimeInds, 9], label='EUVAC-22', lw=3)
    axs[1, 0].plot(highSolarTimes, solomonIrrHFG[highSolarTimeInds, 9], label='HFG', lw=3, color='purple')
    axs[1, 0].set_ylabel('Irradiance (W/m$^2$)')
    axs[1, 0].set_title('430 $\mathrm{\AA}$')
    axs[1, 0].legend(loc='best')
    axs[1, 0].set_xticklabels(axs[1, 0].get_xticklabels(), rotation=45, ha='right')
    # Bottom-middle: High Solar Activity 724 A
    axs[1, 1].plot(highSolarTimes, correspondingIrrFISM2StanBands[highSolarTimeInds, 11], label='FISM2', lw=3)
    axs[1, 1].plot(highSolarTimes, ensemble_average_NeuvacIrrSolomon[highSolarTimeInds, 11], label='NEUVAC-22 (n='+str(iterations)+')', lw=3)
    axs[1, 1].plot(highSolarTimes, solomonIrrEUVAC[highSolarTimeInds, 11], label='EUVAC-22', lw=3)
    axs[1, 1].plot(highSolarTimes, solomonIrrHFG[highSolarTimeInds, 11], label='HFG', lw=3, color='purple')
    axs[1, 1].set_title('724 $\mathrm{\AA}$')
    axs[1, 1].legend(loc='best')
    axs[1, 1].set_xticklabels(axs[1, 1].get_xticklabels(), rotation=45, ha='right')
    # Bottom-right: High Solar Activity 981 A
    axs[1, 2].plot(highSolarTimes, correspondingIrrFISM2StanBands[highSolarTimeInds, 19], label='FISM2', lw=3)
    axs[1, 2].plot(highSolarTimes, ensemble_average_NeuvacIrrSolomon[highSolarTimeInds, 19], label='NEUVAC-22 (n='+str(iterations)+')', lw=3)
    axs[1, 2].plot(highSolarTimes, solomonIrrEUVAC[highSolarTimeInds, 19], label='EUVAC-22', lw=3)
    axs[1, 2].plot(highSolarTimes, solomonIrrHFG[highSolarTimeInds, 19], label='HFG', lw=3, color='purple')
    axs[1, 2].set_title('981 $\mathrm{\AA}$')
    axs[1, 2].legend(loc='best')
    axs[1, 2].set_xticklabels(axs[1, 2].get_xticklabels(), rotation=45, ha='right')
    # Save the figure:
    fig.tight_layout()
    fig.suptitle('Irradiance During Low Solar Activity ('+str(lowSolarTimeBounds[0])[:-9]+' to '+
                 str(lowSolarTimeBounds[-1])[:-9]+') and High Solar Activity ('+str(highSolarTimeBounds[0])[:-9]+' to '
                 +str(highSolarTimeBounds[-1])[:-9]+')\n', fontsize=16, fontweight='bold')
    fig.subplots_adjust(top=0.9)
    plt.savefig(results_dir+'sampleTimeSeriesSpectra_Low_and_High_Solar_Activity_SOLOMON.png', dpi=300)

    # 8C: Simple Time Series (with uncertainty bands) of NEUVAC with other models, during Solar Cycle 25:
    indSolomon = 9
    fig = plt.figure()
    plt.plot(cycle25times, correspondingIrrFISM2StanBands[cycle25inds, indSolomon], label='FISM2')
    plt.fill_between(cycle25times, (ensemble_average_NeuvacIrrSolomon-ensemble_stddev_NeuvacIrrSolomon)[cycle25inds, indSolomon],
                              (ensemble_average_NeuvacIrrSolomon+ensemble_stddev_NeuvacIrrSolomon)[cycle25inds, indSolomon],
                     color='orange', alpha=0.6)
    plt.plot(cycle25times, ensemble_average_NeuvacIrrSolomon[cycle25inds, indSolomon], label='NEUVAC-22 (n='+str(iterations)+')')
    plt.plot(cycle25times, solomonIrrEUVAC[cycle25inds, indSolomon], label='EUVAC-22')
    plt.plot(cycle25times, solomonIrrHFG[cycle25inds, indSolomon], label='HFG', color='purple')
    plt.legend(loc='best', fontsize=25)
    plt.xlabel('Time', fontsize=27)
    plt.ylabel('Irradiance (W/m$^2$)', fontsize=27)
    plt.title('Solar Irradiance Centered at '+str(xPosSolomon[indSolomon])+' $\mathrm{\AA}$ (Solar Cycle 25)', fontsize=32)
    plt.tick_params(axis='both', labelsize=27)
    plt.xticks(rotation=45)
    plt.tight_layout()
    plt.savefig(results_dir+'sampleTimeSeries_cycle25_SOLOMON.png', dpi=300)

    # 8D: SQDF vs. Solar activity (F10.7) - 3 Bands:
    # sqdf_NEUVAC_25 = toolbox.squareDiff(ensemble_average_NeuvacIrrSolomon[:, 3], correspondingIrrFISM2StanBands[:, 3])
    # sqdf_NEUVAC_595 = toolbox.squareDiff(ensemble_average_NeuvacIrrSolomon[:, 10], correspondingIrrFISM2StanBands[:, 10])
    # sqdf_NEUVAC_1007 = toolbox.squareDiff(ensemble_average_NeuvacIrrSolomon[:, -2], correspondingIrrFISM2StanBands[:, -3])
    # sqdf_EUVAC_25 = toolbox.squareDiff(solomonIrrEUVAC[:, 3], correspondingIrrFISM2StanBands[:, 3])
    # sqdf_EUVAC_595 = toolbox.squareDiff(solomonIrrEUVAC[:, 10], correspondingIrrFISM2StanBands[:, 10])
    # sqdf_EUVAC_1007 = toolbox.squareDiff(solomonIrrEUVAC[:, -2], correspondingIrrFISM2StanBands[:, -3])
    # sqdf_HFG_25 = toolbox.squareDiff(solomonIrrHFG[:, 3], correspondingIrrFISM2StanBands[:, 3])
    # sqdf_HFG_595 = toolbox.squareDiff(solomonIrrHFG[:, 10], correspondingIrrFISM2StanBands[:, 10])
    # sqdf_HFG_1007 = toolbox.squareDiff(solomonIrrHFG[:, -2], correspondingIrrFISM2StanBands[:, -3])
    # fig, axs = plt.subplots(nrows=1, ncols=3)
    # axs[0].scatter(F107[sortF107], sqdf_NEUVAC_25[sortF107], color='orange', label='NEUVAC (n='+str(iterations)+')', alpha=0.6)
    # axs[0].scatter(F107[sortF107], sqdf_EUVAC_25[sortF107], color='green', label='EUVAC', alpha=0.6)
    # axs[0].scatter(F107[sortF107], sqdf_HFG_25[sortF107], color='purple', label='HFG', alpha=0.6)
    # axs[0].set_xlabel('F10.7 (sfu)')
    # axs[0].set_ylabel('Squared Difference from FISM2 (W/m$^2$)')
    # axs[0].set_title('25 $\mathrm{\AA}$')
    # axs[0].set_yscale('log')
    # axs[0].legend(loc='best')
    # axs[1].scatter(F107[sortF107], sqdf_NEUVAC_595[sortF107], color='orange', label='NEUVAC (n='+str(iterations)+')', alpha=0.6)
    # axs[1].scatter(F107[sortF107], sqdf_EUVAC_595[sortF107], color='green', label='EUVAC', alpha=0.6)
    # axs[1].scatter(F107[sortF107], sqdf_HFG_595[sortF107], color='purple', label='HFG', alpha=0.6)
    # axs[1].set_xlabel('F10.7 (sfu)')
    # axs[1].set_title('595 $\mathrm{\AA}$')
    # axs[1].set_yscale('log')
    # axs[1].legend(loc='best')
    # axs[2].scatter(F107[sortF107], sqdf_NEUVAC_1007[sortF107], color='orange', label='NEUVAC (n='+str(iterations)+')', alpha=0.6)
    # axs[2].scatter(F107[sortF107], sqdf_EUVAC_1007[sortF107], color='green', label='EUVAC', alpha=0.6)
    # axs[2].scatter(F107[sortF107], sqdf_HFG_1007[sortF107], color='purple', label='HFG', alpha=0.6)
    # axs[2].set_xlabel('F10.7 (sfu)')
    # axs[2].set_title('1007 $\mathrm{\AA}$')
    # axs[2].set_yscale('log')
    # axs[2].legend(loc='best')
    # plt.savefig(results_dir + 'SQDF_by_F107_SOLOMON.png', dpi=300)

    # 8E: MAPE vs. Wavelength Band:
    mapeNEUVACS = []
    for i in range(ensemble_average_NeuvacIrrSolomon.shape[1]):
        mapeNEUVACS.append(
            toolbox.mape(correspondingIrrFISM2StanBands[:, i], ensemble_average_NeuvacIrrSolomon[:, i]) * 100)
    mapeEUVACS = []
    for i in range(ensemble_average_NeuvacIrrSolomon.shape[1]):
        mapeEUVACS.append(toolbox.mape(correspondingIrrFISM2StanBands[:, i], solomonIrrEUVAC[:, i]) * 100)
    mapeHFG = []
    for i in range(ensemble_average_NeuvacIrrSolomon.shape[1]):
        mapeHFG.append(toolbox.mape(correspondingIrrFISM2StanBands[:, i], solomonIrrHFG[:, i]) * 100)
    sortWavSolomon = xPosSortedSolomon[:-1]
    plt.figure()
    plt.plot(xPosSortedSolomon[:-1], np.asarray(mapeNEUVACS)[sortIndsSolomon[:-1]], color='orange', marker='o', label='NEUVAC-22 (n='+str(iterations)+')')
    plt.plot(xPosSortedSolomon[:-1], np.asarray(mapeEUVACS)[sortIndsSolomon[:-1]], color='green', marker='o', label='EUVAC-22')
    plt.plot(xPosSortedSolomon[:-1], np.asarray(mapeHFG)[sortIndsSolomon[:-1]], color='purple', marker='o', label='HFG')
    plt.yscale('log')
    plt.xlabel('Wavelength ($\mathrm{\AA}$)')
    plt.ylabel('MAPE (%)')
    plt.title('MAPE for NEUVAC-22, EUVAC-22, and HFG vs. Wavelength')
    plt.legend(loc='best')
    plt.grid()
    plt.savefig(results_dir + 'MAPE_by_band_SOLOMON.png', dpi=300)
    # Stair plot of the above:
    plt.figure()
    plt.stairs(values=np.asarray(mapeNEUVACS)[sortIndsSolomon[:-1]], edges=xPosSortedSolomon, color='orange',
               label='NEUVAC-22 (n=' + str(iterations) + ')', lw=9)
    plt.stairs(values=np.asarray(mapeEUVACS)[sortIndsSolomon[:-1]], edges=xPosSortedSolomon, color='green',
               label='EUVAC-22', lw=9)
    plt.stairs(values=np.asarray(mapeHFG)[sortIndsSolomon[:-1]], edges=xPosSortedSolomon, color='purple',
               label='HFG', lw=9)
    plt.yscale('log')
    plt.xlabel('Wavelength ($\mathrm{\AA}$)', fontsize=25)
    plt.ylabel('MAPE (%)', fontsize=25)
    plt.title('MAPE for NEUVAC-22, EUVAC-22, and HFG vs. Wavelength', fontsize=30)
    plt.tick_params(axis='both', labelsize=25)
    plt.legend(loc='best', fontsize=25)
    plt.grid()
    plt.tight_layout()
    plt.savefig(results_dir + 'MAPE_by_band_SOLOMON_stairs.png', dpi=300)

    # 8F: Distribution of Percent Deviation
    NEUVAC_resids_Solomon = vfunc(ensemble_average_NeuvacIrrSolomon, correspondingIrrFISM2StanBands[:, :-1])
    EUVAC_resids_Solomon = vfunc(solomonIrrEUVAC, correspondingIrrFISM2StanBands[:, :-1])
    HFG_resids_Solomon = vfunc(solomonIrrHFG, correspondingIrrFISM2StanBands[:, :-1])
    NEUVAC_resids_flat_Solomon = np.ravel(NEUVAC_resids_Solomon)
    EUVAC_resids_flat_Solomon = np.ravel(EUVAC_resids_Solomon)
    HFG_resids_flat_Solomon = np.ravel(HFG_resids_Solomon)
    # bins = np.linspace(np.min([np.nanpercentile(NEUVAC_resids, 25), np.nanpercentile(EUVAC_resids, 25), np.nanpercentile(HEUVAC_resids[HEUVAC_resids != -np.inf], 25)]),
    #                    np.max([np.nanpercentile(NEUVAC_resids, 75), np.nanpercentile(EUVAC_resids, 75), np.nanpercentile(HEUVAC_resids, 75)]), num=100)
    # bins = np.linspace(np.nanmin(NEUVAC_resids_flat_Solomon), np.nanmax(NEUVAC_resids_flat_Solomon), num=100)
    bins = np.linspace(-25., 25., num=100)
    # myLabels = ['Percent Deviation (%)', 'Count', 'NEUVAC Percent Deviation from FISM2']
    # figHist = toolbox.plotHist(NEUVAC_resids_flat_Solomon, bins=bins, color='orange', saveLoc=results_dir + 'NEUVAC_percDev_SOLOMON.png', labels=myLabels, density=False)

    # Percent Deviation for just three bins: 430 A (9), 724 A (11), and 981 A (19)
    NEUVAC_resids_430 = toolbox.percDev(ensemble_average_NeuvacIrrSolomon[:, 9], correspondingIrrFISM2StanBands[:, 9])
    myLabels430 = ['Percent Deviation (%)', 'Count', r'NEUVAC Percent Deviation from FISM2: 430 $\mathrm{\AA}$']
    figHist430 = toolbox.plotHist(NEUVAC_resids_430, bins=bins, color='orange',
                                  saveLoc=results_dir + 'NEUVAC_percDev_430_SOLOMON.png', labels=myLabels430)

    NEUVAC_resids_724 = toolbox.percDev(ensemble_average_NeuvacIrrSolomon[:, 11], correspondingIrrFISM2StanBands[:, 11])
    myLabels724 = ['Percent Deviation (%)', 'Count', r'NEUVAC Percent Deviation from FISM2: 724 $\mathrm{\AA}$']
    figHist724 = toolbox.plotHist(NEUVAC_resids_724, bins=bins, color='orange',
                                  saveLoc=results_dir + 'NEUVAC_percDev_724_SOLOMON.png', labels=myLabels724)

    NEUVAC_resids_981 = toolbox.percDev(ensemble_average_NeuvacIrrSolomon[:, 19], correspondingIrrFISM2StanBands[:, 19])
    myLabels981 = ['Percent Deviation (%)', 'Count', r'NEUVAC Percent Deviation from FISM2: 981 $\mathrm{\AA}$']
    figHist981 = toolbox.plotHist(NEUVAC_resids_981, bins=bins, color='orange',
                                  saveLoc=results_dir + 'NEUVAC_percDev_981_SOLOMON.png', labels=myLabels981)


    # 8G: Behavior of Percent Deviations as a Function Solar Activity (F10.7)
    NEUVAC_resids_430_Solomon = NEUVAC_resids_Solomon[:, 9]
    NEUVAC_resids_724_Solomon = NEUVAC_resids_Solomon[:, 11]
    NEUVAC_resids_981_Solomon = NEUVAC_resids_Solomon[:, 19]
    EUVAC_resids_430_Solomon = EUVAC_resids_Solomon[:, 9]
    EUVAC_resids_724_Solomon = EUVAC_resids_Solomon[:, 1]
    EUVAC_resids_981_Solomon = EUVAC_resids_Solomon[:, 19]
    HFG_resids_430_Solomon = HFG_resids_Solomon[:, 9]
    HFG_resids_724_Solomon = HFG_resids_Solomon[:, 1]
    HFG_resids_981_Solomon = HFG_resids_Solomon[:, 19]
    fig, axs = plt.subplots(nrows=1, ncols=3, sharey=True)
    axs[0].scatter(F107[sortF107], NEUVAC_resids_430_Solomon[sortF107], color='orange',
                   label='NEUVAC-22 (n=' + str(iterations) + ')')
    axs[0].scatter(F107[sortF107], EUVAC_resids_430_Solomon[sortF107], color='green', label='EUVAC-22')
    axs[0].scatter(F107[sortF107], HFG_resids_430_Solomon[sortF107], color='purple', label='HFG')
    axs[0].axhline(y=0, linewidth=3, linestyle='--', color='grey')
    axs[0].set_xlabel('F10.7 (sfu)', fontsize=30)
    axs[0].set_ylabel('Percent Deviation from FISM2 (%)', fontsize=30)
    axs[0].set_title('430 $\mathrm{\AA}$', fontsize=35)
    axs[0].legend(loc='best', fontsize=18)
    axs[0].tick_params(axis='both', labelsize=22)
    axs[1].scatter(F107[sortF107], NEUVAC_resids_724_Solomon[sortF107], color='orange', label='NEUVAC-22 (n='+str(iterations)+')')
    axs[1].scatter(F107[sortF107], EUVAC_resids_724_Solomon[sortF107], color='green', label='EUVAC-22')
    axs[1].scatter(F107[sortF107], HFG_resids_724_Solomon[sortF107], color='purple', label='HFG')
    axs[1].axhline(y=0, linewidth=3, linestyle='--', color='grey')
    axs[1].set_ylim([-150, 150])
    axs[1].set_xlabel('F107 (sfu)', fontsize=30)
    axs[1].set_title('725 $\mathrm{\AA}$', fontsize=35)
    axs[1].legend(loc='best', fontsize=18)
    axs[1].tick_params(axis='both', labelsize=22)
    axs[2].scatter(F107[sortF107], NEUVAC_resids_981_Solomon[sortF107], color='orange', label='NEUVAC-22 (n='+str(iterations)+')')
    axs[2].scatter(F107[sortF107], EUVAC_resids_981_Solomon[sortF107], color='green', label='EUVAC-22')
    axs[2].scatter(F107[sortF107], HFG_resids_981_Solomon[sortF107], color='purple', label='HFG')
    axs[2].axhline(y=0, linewidth=3, linestyle='--', color='grey')
    axs[2].set_xlabel('F107 (sfu)', fontsize=30)
    axs[2].set_title('981 $\mathrm{\AA}$', fontsize=35)
    axs[2].legend(loc='best', fontsize=18)
    axs[2].tick_params(axis='both', labelsize=22)
    plt.savefig(results_dir + 'percDev_by_F107_SOLOMON.png', dpi=300)

    # 8H: INTEGRATED ENERGY (ACROSS THE SUN-FACING SIDE OF THE EARTH)
    fism2SumsSolomon = np.zeros(ensemble_average_NeuvacIrrSolomon.shape[0])
    for i in range(len(fism2SumsSolomon)):
        fism2SumsSolomon[i] = np.nansum(correspondingIrrFISM2StanBands[i, :])
    fism2IntegEnergySolomon = A*fism2SumsSolomon
    neuvacSumsSolomon = np.zeros_like(fism2SumsSolomon)
    for i in range(len(fism2SumsSolomon)):
        neuvacSumsSolomon[i] = np.nansum(ensemble_average_NeuvacIrrSolomon[i, :])
    neuvacIntegEnergySolomon = A*neuvacSumsSolomon
    euvacSumsSolomon = np.zeros_like(fism2SumsSolomon)
    for i in range(len(fism2SumsSolomon)):
        euvacSumsSolomon[i] = np.nansum(solomonIrrEUVAC[i, :])
    euvacIntegEnergySolomon = A*euvacSumsSolomon
    hfgSumsSolomon = np.zeros_like(fism2SumsSolomon)
    for i in range(len(fism2SumsSolomon)):
        hfgSumsSolomon[i] = np.nansum(solomonIrrHFG[i, :])
    hfgIntegEnergySolomon = A * hfgSumsSolomon

    fig, axs = plt.subplots(nrows=3, ncols=1)
    #
    axs[0].plot(lowSolarTimesLonger, fism2IntegEnergySolomon[lowSolarTimeIndsLonger] / (1e9), label='FISM2')
    axs[0].plot(lowSolarTimesLonger, neuvacIntegEnergySolomon[lowSolarTimeIndsLonger] / (1e9), label='NEUVAC-22 (n=' + str(iterations) + ')')
    axs[0].plot(lowSolarTimesLonger, euvacIntegEnergySolomon[lowSolarTimeIndsLonger] / (1e9), label='EUVAC-22')
    axs[0].plot(lowSolarTimesLonger, hfgIntegEnergySolomon[lowSolarTimeIndsLonger] / (1e9), label='HFG', color='purple')
    axs[0].set_ylabel('Integrated Energy (GW)')
    axs[0].legend(loc='best')
    axs[0].set_title('Low Solar Activity: '+str(lowSolarTimeBoundsLonger[0])[:-9]+' to '+str(lowSolarTimeBoundsLonger[-1])[:-9])
    #
    axs[1].plot(highSolarTimesLonger, fism2IntegEnergySolomon[highSolarTimeIndsLonger] / (1e9), label='FISM2')
    axs[1].plot(highSolarTimesLonger, neuvacIntegEnergySolomon[highSolarTimeIndsLonger] / (1e9), label='NEUVAC-22 (n=' + str(iterations) + ')')
    axs[1].plot(highSolarTimesLonger, euvacIntegEnergySolomon[highSolarTimeIndsLonger] / (1e9), label='EUVAC-22')
    axs[1].plot(highSolarTimesLonger, hfgIntegEnergySolomon[highSolarTimeIndsLonger] / (1e9), label='HFG', color='purple')
    axs[1].set_ylabel('Integrated Energy (GW)')
    axs[1].legend(loc='best')
    axs[1].set_title('High Solar Activity: '+str(highSolarTimeBoundsLonger[0])[:-9]+' to '+str(highSolarTimeBoundsLonger[-1])[:-9])
    #
    axs[2].plot(times, fism2IntegEnergySolomon / (1e9), label='FISM2')
    axs[2].plot(times, neuvacIntegEnergySolomon / (1e9), label='NEUVAC-22 (n='+str(iterations)+')')
    axs[2].plot(times, euvacIntegEnergySolomon / (1e9), label='EUVAC-22')
    axs[2].plot(times, hfgIntegEnergySolomon / (1e9), label='HFG', color='purple')
    axs[2].set_xlabel('Time')
    axs[2].set_ylabel('Integrated Energy (GW)')
    axs[2].legend(loc='best')
    axs[2].set_title('Solar Cycle 20 through Ascending Phase of Solar Cycle 25')
    # Save the figure
    fig.tight_layout()
    fig.suptitle('Integrated Earth-Incident Energy in b22 Bins', fontsize=16, fontweight='bold')
    fig.subplots_adjust(top=0.9)
    plt.savefig(results_dir + 'integrated_energy_SOLOMON.png', dpi=300)
    #
    # TABLE: Compute and print the MAPE for all models during low, moderate, and high solar activity (see Jin, et al. 2021 for definitions for low/moderate/high: https://agupubs.onlinelibrary.wiley.com/doi/10.1029/2020JA028932)
    neuvacEnergyLowMapeSolomon = toolbox.mape(neuvacIntegEnergySolomon[lowSolarActivityInds], fism2IntegEnergySolomon[lowSolarActivityInds]) * 100
    euvacEnergyLowMapeSolomon = toolbox.mape(euvacIntegEnergySolomon[lowSolarActivityInds], fism2IntegEnergySolomon[lowSolarActivityInds]) * 100
    hfgEnergyLowMapeSolomon = toolbox.mape(hfgIntegEnergySolomon[lowSolarActivityInds], fism2IntegEnergySolomon[lowSolarActivityInds]) * 100
    neuvacEnergyModMapeSolomon = toolbox.mape(neuvacIntegEnergySolomon[modSolarActivityInds], fism2IntegEnergySolomon[modSolarActivityInds]) * 100
    euvacEnergyModMapeSolomon = toolbox.mape(euvacIntegEnergySolomon[modSolarActivityInds], fism2IntegEnergySolomon[modSolarActivityInds]) * 100
    hfgEnergyModMapeSolomon = toolbox.mape(hfgIntegEnergySolomon[modSolarActivityInds], fism2IntegEnergySolomon[modSolarActivityInds]) * 100
    neuvacEnergyHighMapeSolomon = toolbox.mape(neuvacIntegEnergySolomon[highSolarActivityInds], fism2IntegEnergySolomon[highSolarActivityInds]) * 100
    euvacEnergyHighMapeSolomon = toolbox.mape(euvacIntegEnergySolomon[highSolarActivityInds], fism2IntegEnergySolomon[highSolarActivityInds]) * 100
    hfgEnergyHighMapeSolomon = toolbox.mape(hfgIntegEnergySolomon[highSolarActivityInds], fism2IntegEnergySolomon[highSolarActivityInds]) * 100
    print('NEUVAC MAPE (SOLOMON): ' + str(np.round(neuvacEnergyLowMapeSolomon, 2)) + '% (low activity), ' + str(
        np.round(neuvacEnergyModMapeSolomon, 2)) + '% (moderate activity), ' + str(
        np.round(neuvacEnergyHighMapeSolomon, 2)) + '% (high activity)')
    print('EUVAC MAPE (SOLOMON): ' + str(np.round(euvacEnergyLowMapeSolomon, 2)) + '% (low activity), ' + str(
        np.round(euvacEnergyModMapeSolomon, 2)) + '% (moderate activity), ' + str(
        np.round(euvacEnergyHighMapeSolomon, 2)) + '% (high activity)')
    print('HFG MAPE (SOLOMON): ' + str(np.round(hfgEnergyLowMapeSolomon, 2)) + '% (low activity), ' + str(
        np.round(hfgEnergyModMapeSolomon, 2)) + '% (moderate activity), ' + str(
        np.round(hfgEnergyHighMapeSolomon, 2)) + '% (high activity)')
    # ==================================================================================================================
    # Exit with a zero error code:
    sys.exit(0)
#-----------------------------------------------------------------------------------------------------------------------
