"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AccessPoint = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("aws-cdk-lib/aws-iam");
const s3 = require("aws-cdk-lib/aws-s3");
const core = require("aws-cdk-lib/core");
const aws_s3objectlambda_1 = require("aws-cdk-lib/aws-s3objectlambda");
const metadata_resource_1 = require("aws-cdk-lib/core/lib/metadata-resource");
class AccessPointBase extends core.Resource {
    /** Implement the `IAccessPoint.domainName` field. */
    get domainName() {
        const urlSuffix = this.stack.urlSuffix;
        return `${this.accessPointName}-${this.env.account}.s3-object-lambda.${urlSuffix}`;
    }
    /** Implement the `IAccessPoint.regionalDomainName` field. */
    get regionalDomainName() {
        const urlSuffix = this.stack.urlSuffix;
        const region = this.env.region;
        return `${this.accessPointName}-${this.env.account}.s3-object-lambda.${region}.${urlSuffix}`;
    }
    /** Implement the `IAccessPoint.virtualHostedUrlForObject` method. */
    virtualHostedUrlForObject(key, options) {
        const domainName = options?.regional ?? true ? this.regionalDomainName : this.domainName;
        const prefix = `https://${domainName}`;
        if (!key) {
            return prefix;
        }
        if (key.startsWith('/')) {
            key = key.slice(1);
        }
        if (key.endsWith('/')) {
            key = key.slice(0, -1);
        }
        return `${prefix}/${key}`;
    }
}
/**
 * Checks the access point name against the rules in https://docs.aws.amazon.com/AmazonS3/latest/userguide/creating-access-points.html#access-points-names
 * @param name The name of the access point
 */
function validateAccessPointName(name) {
    if (name.length < 3 || name.length > 50) {
        throw new Error('Access point name must be between 3 and 50 characters long');
    }
    if (name.endsWith('-s3alias')) {
        throw new Error('Access point name cannot end with the suffix -s3alias');
    }
    if (name[0] === '-' || name[name.length - 1] === '-') {
        throw new Error('Access point name cannot begin or end with a dash');
    }
    if (!/^[0-9a-z](.(?![\.A-Z_]))+[0-9a-z]$/.test(name)) {
        throw new Error('Access point name must begin with a number or lowercase letter and not contain underscores, uppercase letters, or periods');
    }
}
/**
 * An S3 object lambda access point for intercepting and
 * transforming `GetObject` requests.
 */
class AccessPoint extends AccessPointBase {
    /**
     * Reference an existing AccessPoint defined outside of the CDK code.
     */
    static fromAccessPointAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_s3objectlambda_alpha_AccessPointAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromAccessPointAttributes);
            }
            throw error;
        }
        const arn = core.Arn.split(attrs.accessPointArn, core.ArnFormat.SLASH_RESOURCE_NAME);
        if (!arn.resourceName) {
            throw new Error('Unable to parse access point name');
        }
        const name = arn.resourceName;
        class Import extends AccessPointBase {
            constructor() {
                super(...arguments);
                this.accessPointArn = attrs.accessPointArn;
                this.accessPointCreationDate = attrs.accessPointCreationDate;
                this.accessPointName = name;
            }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.accessPointName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_s3objectlambda_alpha_AccessPointProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, AccessPoint);
            }
            throw error;
        }
        // Enhanced CDK Analytics Telemetry
        (0, metadata_resource_1.addConstructMetadata)(this, props);
        if (props.accessPointName) {
            validateAccessPointName(props.accessPointName);
        }
        const supporting = new s3.CfnAccessPoint(this, 'SupportingAccessPoint', {
            bucket: props.bucket.bucketName,
        });
        const allowedFeatures = [];
        if (props.supportsGetObjectPartNumber) {
            allowedFeatures.push('GetObject-PartNumber');
        }
        if (props.supportsGetObjectRange) {
            allowedFeatures.push('GetObject-Range');
        }
        const accessPoint = new aws_s3objectlambda_1.CfnAccessPoint(this, id, {
            name: this.physicalName,
            objectLambdaConfiguration: {
                allowedFeatures,
                cloudWatchMetricsEnabled: props.cloudWatchMetricsEnabled,
                supportingAccessPoint: supporting.attrArn,
                transformationConfigurations: [
                    {
                        actions: ['GetObject'],
                        contentTransformation: {
                            AwsLambda: {
                                FunctionArn: props.handler.functionArn,
                                FunctionPayload: props.payload ? JSON.stringify(props.payload) : undefined,
                            },
                        },
                    },
                ],
            },
        });
        this.s3AccessPointArn = supporting.attrArn;
        this.accessPointName = accessPoint.ref;
        this.accessPointArn = accessPoint.attrArn;
        this.accessPointCreationDate = accessPoint.attrCreationDate;
        props.handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['s3-object-lambda:WriteGetObjectResponse'],
            resources: ['*'],
        }));
    }
}
exports.AccessPoint = AccessPoint;
_a = JSII_RTTI_SYMBOL_1;
AccessPoint[_a] = { fqn: "@aws-cdk/aws-s3objectlambda-alpha.AccessPoint", version: "2.185.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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