# Test the simulator with randomly generated OriginIR code.

import numpy as np
from qpandalite.originir.random_originir import random_originir
from qpandalite.originir.originir_spec import (available_originir_gates, 
                                               available_originir_error_channels,
                                               generate_sub_gateset_originir,
                                               generate_sub_error_channel_originir)

from qpandalite.simulator import OriginIR_NoisySimulator
from qpandalite.test._utils import qpandalite_test
class NotMatchError(Exception):
    pass

def compare_density_operator(circuit):
    # This test compares two density operators generated by QuTip and QPanda-lite.
    # The test is based on the following steps:
    # 1. Simulate the circuit using QuTip and QPanda-lite.
    # 2. Compare the results.
    # 3. If the results are not the same, raise an error.

    # Step 1: Simulate the circuit using QuTip and QPanda-lite
    sim_qpandalite = OriginIR_NoisySimulator(backend_type='density_operator')
    sim_qutip = OriginIR_NoisySimulator(backend_type='density_operator_qutip')

    mat_qpandalite = sim_qpandalite.simulate_density_matrix(circuit)
    mat_qutip = sim_qutip.simulate_density_matrix(circuit)

    # Step 2: Compare the results
    if not np.allclose(mat_qpandalite, mat_qutip):
        return NotMatchError(
            '---------------\n'
            f'{circuit}\n'
            '---------------\n'
            'Result not match!\n'
            f'QPanda-lite Result = {mat_qpandalite}\n'
            f'QuTip Result = {mat_qutip}\n'
        )


def test_random_originir_compare_density_operator(
    random_batchsize = 100,
    n_qubits = 5,
    n_gates = 20,
    gate_set = available_originir_gates,
    error_channel = available_originir_error_channels):
    
    err_list = []    
    good_circuit_list = []
    bad_circuit_list = []
    for i in range(random_batchsize):
        originir_code = random_originir(
            n_qubits=n_qubits, 
            n_gates=n_gates, instruction_set=gate_set, channel_set=error_channel)
        
        err = compare_density_operator(originir_code)    
        
        if err:
            print('Test failed!')
            err_list.append(err)
            bad_circuit_list.append((originir_code, err))
        else:
            print('Test passed!')
            good_circuit_list.append((originir_code, None))


    print(len(err_list), 'circuits failed')
    print(random_batchsize - len(err_list), 'circuits passed')

    # log good and bad circuits
    with open('good_circuits.txt', 'w') as f:
        for circuit, result in good_circuit_list:
            f.write(circuit + '\n----Result----\n' + str(result) + '\n-----------------\n\n')

    with open('bad_circuits.txt', 'w') as f:
        for e in err_list:
            f.write(str(e) + '\n')

    if len(err_list) > 0:
        raise ValueError('Some circuits failed!')
    

@qpandalite_test('Test Random OriginIR Density Operator')
def test_random_originir_density_operator():
    
    gate_set =  ['H', 'X', 'Y', 'Z', 'S', 'SX', 'T',
                 'RX', 'RY', 'RZ', 'U1', 'RPhi90', 'RPhi180',
                 'RPhi', 'U2',
                 'U3',
                 'CNOT', 'CZ', 'ISWAP', 
                 'XX', 'YY', 'ZZ', 'XY', 
                 'PHASE2Q', 
                 'UU15',
                 'TOFFOLI', 'CSWAP']
    
    gate_set =  ['H', 'X', 'Y', 'Z', 'S', 'SX', 'T',
                 'RX', 'RY', 'RZ', 'U1', 'RPhi90', 'RPhi180',
                 'RPhi', 'U2',
                 'U3',
                 'CNOT', 'CZ', 'ISWAP', 
                 'XX', 'YY', 'ZZ', 'XY', 
                 'PHASE2Q', 
                 'UU15',
                 'TOFFOLI', 'CSWAP']
    
    gate_set = generate_sub_gateset_originir(gate_set)

    error_channel = ['PauliError1Q', 
                    'Depolarizing', 
                    'BitFlip', 
                    'PhaseFlip', 
                    'AmplitudeDamping',
                    'Kraus1Q',
                    'PauliError2Q', 
                    'TwoQubitDepolarizing']
    
    error_channel = ['PauliError1Q', 
                    'Depolarizing', 
                    'BitFlip', 
                    'PhaseFlip', 
                    'AmplitudeDamping',
                    'PauliError2Q',
                    'TwoQubitDepolarizing'
                    ]

    error_channel = generate_sub_error_channel_originir(error_channel)

    test_random_originir_compare_density_operator(
        random_batchsize = 100,
        n_qubits = 5,
        n_gates = 20,
        gate_set = gate_set,
        error_channel = error_channel)


if __name__ == '__main__':
    test_random_originir_density_operator()
